/**
 * SkyrimCharacterHelper 1.15
 * 
 * UI based tool to save and backup skyrim character save-files
 * 
 */
package skyrimcharacterhelper;



/**
 * Imports
 * 
 */
import java.io.FileNotFoundException;
import java.io.File             ;
import java.io.IOException      ;
import java.io.FileInputStream  ;
import java.io.FileOutputStream ;
import java.io.InputStreamReader;
import java.io. BufferedReader  ;

import java.io.UnsupportedEncodingException;
import java.util.ResourceBundle  ;
import java.util.Properties      ;
import java.util.List            ;
import java.util.ArrayList       ;
        
import java.awt.event.WindowListener;
import java.awt.event.WindowEvent   ;
import java.awt.image.BufferedImage ;
import java.awt.Toolkit             ;
import java.awt.Frame               ;

import java.util.Locale;
import org.jdesktop.application.Application;
import org.jdesktop.application.SingleFrameApplication;

import javax.swing.JFileChooser;
import javax.swing.JOptionPane ;
import javax.swing.SwingWorker ;



/**
 * The main class of the application. Note: There's always only one WorkerThread active at a given point of time.
 * 
 * @author Ulf Wagemann
 */
public class SkyrimCharacterHelperApp extends    SingleFrameApplication
                                      implements SkyrimCharacterHelperGuiConnector    ,
                                                 SkyrimCharacterHelperThreadNotifier  ,
                                                 WindowListener
{
 private SkyrimCharacterHelperView m_tView             ; 
 private ResourceBundle            m_tLocalizedMessages;
 private Properties                m_tProperties       ;
 private String                    m_sSeparator        ;
 
 private SwingWorkerDirectoryScan  m_tWorkerDirectoryScan  ; // worker for directory scans
 private SwingWorkerSaveGameDelete m_tWorkerDeleteSaveGames; // worker for deleting savegames
 private SwingWorkerSaveGameCopy   m_tWorkerCopySaveGames  ; // worker for copying savegames
 private SwingWorkerSaveGameLoader m_tWorkerLoaderSaveGamesActive; // worker for copying active savegames. 2 workers because they must be able to run simultaneously.
 private SwingWorkerSaveGameLoader m_tWorkerLoaderSaveGamesBackup; // worker for copying backup savegames. 2 workers because they must be able to run simultaneously.
 
 private boolean m_bInitializing       ; // flag for the intial directory scans. true = in init mode, false = no init mode active
 private boolean m_bPrepareForLaunching; // flag whether we're preparing launching Skyrim with just a single character (true) or not preparing for launching (false)
 private boolean m_bScanAll            ; // flag whether both savegames folders shoudl be scanned on start
 private int     m_iSaveGameChanges    ; // counts how many savegames were changed when monitoring Skyrim
 
 private String  m_sLaunchPlayer; // nameof player to be launched
 
 private SkyrimCharacterHelperPlayerManager m_tPlayerManagerActiveSaveGames; // player manager for active savegames
 private SkyrimCharacterHelperPlayerManager m_tPlayerManagerBackupSaveGames; // player manager for backup savegames
 private SkyrimCharacterHelperLaunchThread  m_tLaunchThread                ; // Skyrim launcher thread

 

 /**-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------
  * 
  * Internal class needed to load a savegame on selection
  * 
  * @author Ulf Wagemann
  */
 class SwingWorkerSaveGameLoader extends    SwingWorker
                                 implements SkyrimCharacterHelperProgressNotifier 
 {
  private String  m_sFilePath;
  private String  m_sMessage ;
  private boolean m_bMode    ;
  
  private SkyrimCharacterHelperPlayerManager m_tCharacterManager;
  private SkyrimCharacterHelperSaveGame      m_tSaveGame        ;
  private SwingWorkerSaveGameLoader          m_tWorkerToWaitFor ;
  
  /**
   * Constructor
   * 
   * @param pa_tWorkerToWaitFor  other worker thread which we may need to wait for
   */
  public SwingWorkerSaveGameLoader(SwingWorkerSaveGameLoader pa_tWorkerToWaitFor)
  {
   super();
   
   m_tCharacterManager = null ;
   m_sFilePath         = null ;
   m_tSaveGame         = null ;
   m_sMessage          = null ;
   m_bMode             = false;
   m_tWorkerToWaitFor  = pa_tWorkerToWaitFor;
  }
  
  
  
  /**
   * Returs the savegame
   * 
   * @return  savegame, may be null
   */
  public SkyrimCharacterHelperSaveGame getSaveGame() {return m_tSaveGame;}
  
  
  
  /**
   * Stores the file name
   * 
   * @param pa_tCharacterManager char manager to use
   * @param pa_sDirectory        directory to scan  
   * @param pa_bMode             scanner mode, true = active save games, false = backup save games
   */
  public void init(String pa_sFilePath, SkyrimCharacterHelperSaveGame pa_tSaveGame, boolean pa_bMode, String pa_sMessage)
  {
   m_sFilePath         = pa_sFilePath;
   m_tSaveGame         = pa_tSaveGame;
   m_sMessage          = pa_sMessage ;
   m_bMode             = pa_bMode    ;   
   m_tCharacterManager = (true == pa_bMode ? m_tPlayerManagerActiveSaveGames : m_tPlayerManagerBackupSaveGames);   
  }
    
  
  
  /**
   * Returns the mode
   * 
   * @return scanner mode, true = save games, false = backup save games
   */
  public boolean getMode() {return m_bMode;}
  
  
  
  /**
   * This is where the lengthy stuff happens
   * 
   * @return 
   */
  public Object doInBackground() 
  {
   //
   // wait for another worker, if needed
   //
   if (null != m_tWorkerToWaitFor) 
   {
    while (null != m_tWorkerToWaitFor && false == m_tWorkerToWaitFor.isDone() && false == m_tWorkerToWaitFor.isCancelled()) 
    {
     try
     {
      Thread.sleep(SkyrimCharacterHelperConstants.SKH_FILE_SAVEGAME_WORKER_DELAY_TIME); 
     }
     catch (InterruptedException lc_tException){}
    }
   }
    
   //
   // now we can start
   //
   if (null != m_sFilePath)
   {
    //
    // init progress display
    //
    m_tView.initCurrentProgress(100);      
    m_tView.setStringCurrentProgress(m_sFilePath);
    m_tView.setMessage(m_sMessage);
    
    m_tSaveGame.read(m_sFilePath, (SkyrimCharacterHelperProgressNotifier) this, true); //read screenshot data 
    m_tCharacterManager.add(m_tSaveGame);
    
    if (true == m_bMode) m_tView.addActiveSaveGame(m_tSaveGame, false);
    else                 m_tView.addBackupSaveGame(m_tSaveGame, false);
   }      
   return null; 
  }  
  
  
  
  /**
   * Receives progress notification from reading a file and displays it
   * 
   * @param pa_iValue   new value
   */
  public void notifyProgress(int pa_iValue) 
  {
   if (false == isCancelled()) {m_tView.setCurrentProgress(pa_iValue);}
  }
  

  
  /**
   * When we're done, we notify the main thread about this
   * 
   */
  @Override
  protected void done() 
  {
   handleSaveGameWorkerThreadFinished(m_bMode, isCancelled());
  }  
  
  
  
 } // end of inner class SwingWorkerDirectoryScan
 //------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
 

 
 
 /**-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------
  * 
  * Internal class needed to update the progress bars properly when deleting savegames. 
  * 
  * @author Ulf Wagemann
  */
 class SwingWorkerSaveGameDelete extends SwingWorker 
 {
  private int     m_iModeId              ;
  private int     m_iAmountProcessedFiles;   
  private boolean m_bMode                ;
  
  private List<SkyrimCharacterHelperSaveGame> m_tSaveGames                  ;
  private SkyrimCharacterHelperPlayerManager  m_tCharacterManagerSource     ;
  private SkyrimCharacterHelperPlayerManager  m_tCharacterManagerDestination;

    
  /**
   * Constructor
   * 
   */
  public SwingWorkerSaveGameDelete()
  {
   super();
   m_bMode                      = false;
   m_iAmountProcessedFiles      = 0    ;
   m_iModeId                    = 0    ;
   
   m_sLaunchPlayer                = null;
   m_tSaveGames                   = null;
   m_tCharacterManagerSource      = null;
   m_tCharacterManagerDestination = null; 
  }

  
 /**
   * Stores the directory name and more
   * 
   * @param pa_iModeId           thread mode id
   * @param pa_tSaveGames        save games to delete
   * @param pa_bMode             copy mode, true = copy active save games, false = copy backup save games
   */
  public final void init(int pa_iModeId, List<SkyrimCharacterHelperSaveGame> pa_tSaveGames, boolean pa_bMode)
  {
   m_tCharacterManagerSource      = (true == pa_bMode ? m_tPlayerManagerActiveSaveGames   : m_tPlayerManagerBackupSaveGames);
   m_tCharacterManagerDestination = (true == pa_bMode ? m_tPlayerManagerBackupSaveGames   : m_tPlayerManagerActiveSaveGames);
   
   m_tSaveGames = pa_tSaveGames;
   m_bMode      = pa_bMode     ;
   m_iModeId    = pa_iModeId   ;
  }

  
  
  /**
   * This is where the lengthy stuff happens
   * 
   * @return 
   */
  public Object doInBackground() 
  {
   int     lc_iCountSaveGames    = 0;
   int     lc_iCountTotal        = 0;
   int     lc_iAmountSaveGames   = 0;
   boolean lc_bReselect          = false;
   boolean lc_bEmpty             = false;
   
   String                      lc_sSourceFileName   = null;
   String                      lc_sSourcePlayerName = null;
   SkyrimCharacterHelperPlayer lc_tSourcePlayer     = null; 
   

   //
   // only continue, if everything is setup nicely
   //
   if (null != m_tCharacterManagerSource && null != m_tCharacterManagerDestination && null != m_tSaveGames)
   {
    //
    // only contiue, if we have a valid list of savegames
    //
    if (0 < (lc_iAmountSaveGames = m_tSaveGames.size()))
    {
     m_tView.initTotalProgress(lc_iAmountSaveGames);   
     
     if (false == isCancelled()) // check for cancellation
     {
      //
      // check all save-files. we do *not* allow thread cancellation while writing files, but only in gaps
      // between read and write operations
      // 
      for (SkyrimCharacterHelperSaveGame lc_tSaveGame : m_tSaveGames)
      {       
       if (true == isCancelled()) return null;

       //
       // get name of player owning this savegame and the related player entry from source player manager
       //
       lc_sSourcePlayerName = lc_tSaveGame.getPlayerName();
        
       //
       // process a single savegame
       //
       if (null != lc_tSaveGame)
       {         
        //
        // delete savegame file and delete savegame from source player manager
        //
        if (null != (lc_sSourceFileName = lc_tSaveGame.getFilePath()))
        {
         m_tCharacterManagerSource.removeSaveGame(lc_tSaveGame); // this also decreases the savegame counter of that player
         this.deleteFile(lc_tSaveGame.getFilePath());

         m_iAmountProcessedFiles++;
         lc_iCountSaveGames++;
         
         lc_bReselect = (lc_iCountSaveGames == lc_iAmountSaveGames);
         lc_bEmpty    = (0 == m_tCharacterManagerSource.getSaveGameAmount(lc_sSourcePlayerName));
         
         //
         // remove savegame from display
         //
         if (true == m_bMode) {m_tView.removeActiveSaveGame(lc_tSaveGame, (false == lc_bEmpty && true == lc_bReselect));} // avoid reselecting in empty savegame tables
         else                 {m_tView.removeBackupSaveGame(lc_tSaveGame, (false == lc_bEmpty && true == lc_bReselect));} // avoid reselecting in empty savegame tables         
        }
       }
       if (false == isCancelled()) m_tView.setTotalProgress(++lc_iCountTotal);            
        
       //
       // check if the current source player's savegame is empty. if so, we need to remove the player
       //
       if (null != (lc_tSourcePlayer = m_tCharacterManagerSource.getPlayer(lc_sSourcePlayerName))) // after removing save games, get the updated version from playermanager
       {
        if (0 == lc_tSourcePlayer.getSaveGameAmount())
        {
         m_tCharacterManagerSource.removePlayer(lc_sSourcePlayerName);
         if (true == m_bMode)  m_tView.removeActivePlayer(lc_tSourcePlayer, (0 < m_tCharacterManagerSource.getSaveGameAmount()), false); // avoid reselecting in empty player lists
         else                  m_tView.removeBackupPlayer(lc_tSourcePlayer, (0 < m_tCharacterManagerSource.getSaveGameAmount()), false); // avoid reselecting in empty player lists
        }     
       }
       
       //
       // update amount display
       //
       if (true == m_bMode) {m_tView.setAmountActive(m_tPlayerManagerActiveSaveGames, m_tPlayerManagerActiveSaveGames.getSaveGameAmountForList(m_tView.getSelectedActivePlayers()));}
       else                 {m_tView.setAmountBackup(m_tPlayerManagerBackupSaveGames, m_tPlayerManagerBackupSaveGames.getSaveGameAmountForList(m_tView.getSelectedBackupPlayers()));}       
      } // for lc_tSaveGame
        
     }
    }     
   }
   return null; 
  }    
  
  
  /**
   * Deletes the given file
   * 
   * @param pa_sFilePath   file path
   */
  private void deleteFile(String pa_sFilePath)
  {
   File lc_tFile = null;
   
   if (null != pa_sFilePath)
   {
    if (null != (lc_tFile = new File(pa_sFilePath))) 
    {
     lc_tFile.delete(); 
    }
   }
  }
 
 
 
  /**
   * When we're done, we notify the main thread about this
   * 
   */
  @Override
  protected void done() 
  {
   handleNonScanWorkerThreadFinished(m_iModeId, m_iAmountProcessedFiles, m_bMode);
  }    
  
  
  
  /**
   * Returns the amount of proccesed files
   * 
   * @return   amount, may be 0
   */
  public int getProcessedAmount() {return m_iAmountProcessedFiles;}
  
  
  
  
 } // end of inner class SwingWorkerPlayerClear
 //------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

 
 
 /**-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------
  * 
  * Internal class needed to update the progress bars properly when copying savegames. 
  * 
  * @author Ulf Wagemann
  */
 class SwingWorkerSaveGameCopy extends SwingWorker 
 {
  private int     m_iThreadModeId             ; 
  private int     m_iWorkMode                 ;
  private int     m_iAmountProcessedFiles     ;   

  private boolean m_bMode                     ;
  private boolean m_bDestinationMultiSelection;
  private String  m_sDestinationSelection     ;
  
  private List<SkyrimCharacterHelperSaveGame>   m_tSaveGames;
  private SkyrimCharacterHelperPlayerManager m_tCharacterManagerSource;
  private SkyrimCharacterHelperPlayerManager m_tCharacterManagerDestination;

    
  /**
   * Constructor
   * 
   */
  public SwingWorkerSaveGameCopy()
  {
   super();
   m_bMode                      = false;
   m_bDestinationMultiSelection = false;
   m_iAmountProcessedFiles      = 0    ;
   m_iThreadModeId              = 0    ;
   m_iWorkMode                  = 0    ;
   
   m_tSaveGames                   = null;
   m_tCharacterManagerSource      = null;
   m_tCharacterManagerDestination = null; 
   m_sDestinationSelection        = null;
  }

  
 /**
   * Stores the directory name and more
   * 
   * @param pa_iModeId           thread mode id
   * @param pa_iWorkMode         work mode (copy or move)
   * @param pa_tSaveGames        save games to delete
   * @param pa_bMode             copy mode, true = copy active save games, false = copy backup save games
   * @param pa_bDestinationMultiSelection  true if the destination list has a multiselection active, false otherwise
   * @param pa_sDestinationSelection       name of the selected destination list entry 
   */
  public void init(int pa_iModeId, int pa_iWorkMode, List<SkyrimCharacterHelperSaveGame> pa_tSaveGames, boolean pa_bMode, boolean pa_bDestinationMultiSelection, String pa_sDestinationSelection)
  {
   m_tCharacterManagerSource      = (true == pa_bMode ? m_tPlayerManagerActiveSaveGames   : m_tPlayerManagerBackupSaveGames);
   m_tCharacterManagerDestination = (true == pa_bMode ? m_tPlayerManagerBackupSaveGames   : m_tPlayerManagerActiveSaveGames);
   
   m_bDestinationMultiSelection   = pa_bDestinationMultiSelection;
   m_sDestinationSelection        = pa_sDestinationSelection     ;
   
   m_tSaveGames      = pa_tSaveGames;
   m_bMode           = pa_bMode     ;
   m_iThreadModeId   = pa_iModeId   ;
   m_iWorkMode       = pa_iWorkMode ;
  }

  
  /**
   * This is where the lengthy stuff happens
   * 
   * @return null
   */
  public Object doInBackground() 
  {
   int    lc_iCountTotal        = 0;
   int    lc_iAmountSaveGames   = 0;
   
   boolean lc_bDestinationHasPlayer  = false;
   boolean lc_bReselect              = false;
   boolean lc_bEmpty                 = false;

   
   String  lc_sSourcePlayerName      = null ;
   String  lc_sSourceFileName        = null ;
   String  lc_sDestinationFile       = null ; 
   String  lc_sDestinationDir        = (true == m_bMode ? m_tView.getBackupPath() : m_tView.getActivePath()); 

   SkyrimCharacterHelperPlayer    lc_tSourcePlayer = null; 
   SkyrimCharacterHelperSaveGame  lc_tNewSaveGame  = null;
   
   List<SkyrimCharacterHelperPlayer> lc_tDestinationPlayerOldSelection  = (true == m_bMode ? m_tView.getSelectedBackupPlayers() : m_tView.getSelectedActivePlayers()); 

   //
   // only continue, if everything is setup nicely
   //
   if (null != m_tCharacterManagerSource && null != m_tCharacterManagerDestination && null != m_tSaveGames)
   {
    //
    // only continue, if we have a valid list of savegames
    //
    if (0 < (lc_iAmountSaveGames = m_tSaveGames.size()))
    {
     m_tView.initTotalProgress(lc_iAmountSaveGames);   
     
     //
     // check all save-files. we do *not* allow thread cancellation while writing files, but only in gaps
     // between read and write operations
     // 

     for (SkyrimCharacterHelperSaveGame lc_tSaveGame : m_tSaveGames)
     {       
      if (true == isCancelled()) {break;}
      
      //
      // process a single savegame
      //
      if (null != lc_tSaveGame)
      {                   
       //
       // get name of player owning this savegame and the related player entry from source player manager
       //
       lc_sSourcePlayerName     = lc_tSaveGame.getPlayerName();
       lc_bDestinationHasPlayer = m_tCharacterManagerDestination.hasPlayer(lc_sSourcePlayerName);

       if (null != (lc_sSourceFileName = lc_tSaveGame.getFilePath()))
       {
        if (true == isCancelled()) {break;} 
        lc_sDestinationFile = lc_sDestinationDir + m_sSeparator + lc_tSaveGame.getFileName();
           
        //
        // copy and add to destination manager
        //
        if (true == copyFile(lc_sSourceFileName, lc_sDestinationFile, lc_tSaveGame))
        {
         //
         // update destination character manager
         //
         lc_tNewSaveGame = new SkyrimCharacterHelperSaveGame(lc_tSaveGame, lc_sDestinationFile); // store new path!         
         m_tCharacterManagerDestination.add(lc_tNewSaveGame);
         lc_iCountTotal++;

         //
         // add savegame to list, if visible
         //
         if (false == m_bDestinationMultiSelection && null != lc_sSourcePlayerName && null != m_sDestinationSelection) 
         {
          if (lc_sSourcePlayerName.equals(m_sDestinationSelection))
          {
           if (true == m_bMode) {m_tView.addBackupSaveGame(lc_tNewSaveGame, false);}
           else                 {m_tView.addActiveSaveGame(lc_tNewSaveGame, false);}
          }
         }
         
         //
         // check if a new player was added to the destination character manager. if so, add it to the destination player list
         //
         if (false == lc_bDestinationHasPlayer)
         {
          if (true == m_bMode) m_tView.addBackupPlayer(m_tCharacterManagerDestination.getPlayer(lc_sSourcePlayerName));
          else                 m_tView.addActivePlayer(m_tCharacterManagerDestination.getPlayer(lc_sSourcePlayerName));
          
          m_tView.reselectPlayers(lc_tDestinationPlayerOldSelection, !m_bMode); 
         }
         
         //
         // in move mode, delete source after copying. 
         //
         if (SkyrimCharacterHelperConstants.SKH_WORKMODE_MOVE == m_iWorkMode) 
         {
          //
          // update source manager and delete file
          //
          m_tCharacterManagerSource.removeSaveGame(lc_tSaveGame);
          this.deleteFile(lc_tSaveGame.getFilePath());
          
          // remove save game from source display
          //
          lc_bReselect = (lc_iCountTotal == lc_iAmountSaveGames);
          lc_bEmpty    = (0 == m_tCharacterManagerSource.getSaveGameAmount(lc_sSourcePlayerName)); 

          if (true == m_bMode) {m_tView.removeActiveSaveGame(lc_tSaveGame, (!lc_bEmpty & lc_bReselect));}  
          else                 {m_tView.removeBackupSaveGame(lc_tSaveGame, (!lc_bEmpty & lc_bReselect));}

          //
          //
          // check if the current source player's savegame is empty. if so, we need to remove the player
          //
          if (null != (lc_tSourcePlayer = m_tCharacterManagerSource.getPlayer(lc_sSourcePlayerName)))  // after removing save games, get the updated version from playermanager
          {
           if (0 == lc_tSourcePlayer.getSaveGameAmount())
           {
            m_tCharacterManagerSource.removePlayer(lc_sSourcePlayerName);
           
            if (true == m_bMode) {m_tView.removeActivePlayer(lc_tSourcePlayer, true, false);}
            else                 {m_tView.removeBackupPlayer(lc_tSourcePlayer, true, (m_iThreadModeId == SkyrimCharacterHelperConstants.SKH_WORKER_THREAD_LAUNCH_FROM_BACKUP));} 
           }
          }
         }
         
         //
         // update amount display
         //
         m_tView.setAmountActive(m_tPlayerManagerActiveSaveGames, m_tPlayerManagerActiveSaveGames.getSaveGameAmountForList(m_tView.getSelectedActivePlayers()));
         m_tView.setAmountBackup(m_tPlayerManagerBackupSaveGames, m_tPlayerManagerBackupSaveGames.getSaveGameAmountForList(m_tView.getSelectedBackupPlayers()));                   
        } // copyFile
       }
      } 
      if (false == isCancelled()) m_tView.setTotalProgress(lc_iCountTotal);                  
     } // for lc_tSaveGame
    }    
   }
   return null; 
  }    
  

  /**
   * Copies the given source file to the given destination file
   * 
   * @param pa_sSourceFileName          source file name
   * @param pa_sDestinationFileName      destination file name
   * @param pa_tSaveGame                source save game
   */
  private synchronized boolean copyFile(String pa_sSourceFileName, String pa_sDestinationFileName, SkyrimCharacterHelperSaveGame pa_tSaveGame)
  {
   int    lc_iCountCurrent      = 0;
   int    lc_iFileSize          = 0;
   int    lc_iBytesRead         = 0;
   int    lc_iBytesReadTotal    = 0;
   int    lc_iRatio             = 0;

   boolean lc_bResult = false;
   
   File             lc_tSourceFile       = null;
   File             lc_tDestinationFile  = null;
   FileInputStream  lc_tInputStream      = null;
   FileOutputStream lc_tOutputStream     = null;
   byte[]           lc_tBuffer           = new byte[SkyrimCharacterHelperConstants.SKH_FILE_BUFFER_SIZE];
   
   //
   // obtain source top copy
   //
   if (null != pa_sSourceFileName && null != pa_sDestinationFileName && null != pa_tSaveGame && null != lc_tBuffer)
   {
    if (null != (lc_tSourceFile = new File(pa_sSourceFileName))) 
    {
     lc_iFileSize       = (int) lc_tSourceFile.length();
     lc_iRatio          = lc_iFileSize / SkyrimCharacterHelperConstants.SKH_FILE_BUFFER_SIZE;
     lc_iBytesRead      = 0;
     lc_iBytesReadTotal = 0;
     lc_iCountCurrent   = 0; 

     m_tView.initCurrentProgress(lc_iRatio);
     m_tView.setStringCurrentProgress(pa_sSourceFileName);

     try
     {
      if (null != (lc_tInputStream = new FileInputStream(lc_tSourceFile))) 
      {
       //
       // obtain destination file
       //
       if (null != (lc_tOutputStream = new FileOutputStream(pa_sDestinationFileName)))
       {
        //
        // copy one file. no intermission accepted. instead of using copyFile(), we copy in byte-blocks to be able
        // of displaying the process with the progress bars.
        //
        try
        {
         while ((lc_iBytesRead = lc_tInputStream.read(lc_tBuffer)) > 0)
         {
          lc_tOutputStream.write(lc_tBuffer, 0, lc_iBytesRead);  

          lc_iBytesReadTotal += lc_iBytesRead;
          if (lc_iBytesReadTotal >= lc_iRatio) lc_iBytesReadTotal = 0;
          if (false == isCancelled()) m_tView.setCurrentProgress(++lc_iCountCurrent); 
         } // while
        }
        catch (Exception lc_tException) {}
          
        m_iAmountProcessedFiles++;
        lc_bResult = true;
        
        //
        // close output stream
        //
        if (null != lc_tOutputStream) try{lc_tOutputStream.close();} catch (Exception lc_tException){}

       }
       //                   
       // close input stream 
       //
       if (null != lc_tInputStream ) try{lc_tInputStream.close() ;} catch (Exception lc_tException){}       
      }
     }
     catch (Exception lc_tException) {}            
    }
   }
   
   if (true == lc_bResult)
   {
    //
    // use timestamp of source savegame !
    //
    if (null != (lc_tDestinationFile = new File(pa_sDestinationFileName))) {lc_tDestinationFile.setLastModified(pa_tSaveGame.getFileDate());}     
   }
   m_tView.setStringCurrentProgress(null);   
   return lc_bResult;
  }
  
  
  
 
  /**
   * Deletes the given file
   * 
   * @param pa_sFilePath   file path
   */
  private synchronized void deleteFile(String pa_sFilePath)
  {
   File lc_tFile = null;
   
   if (null != pa_sFilePath)
   {
    if (null != (lc_tFile = new File(pa_sFilePath))) 
    {
     lc_tFile.delete(); 
    }
   }
  }
 
 
 
  /**
   * When we're done, we notify the main thread about this
   * 
   */
  @Override
  protected void done()
  {  
   handleNonScanWorkerThreadFinished(m_iThreadModeId, m_iAmountProcessedFiles, m_bMode);
  }    
  
  
  /**
   * Returns the amount of proccesed files
   * 
   * @return   amount, may be 0
   */
  public final int getProcessedAmount() {return m_iAmountProcessedFiles;}
  
  
  /**
   * Returns the mode
   * 
   * @param pa_bMode             copy mode, true = copy active save games, false = copy backup save games
   */
  public final boolean getMode() {return m_bMode;}
  
  
  /**
   * Returns the work mode
   * 
   * @return   workmode
   */
  public final int getWorkMode() {return m_iWorkMode;}
  
 } // end of inner class SwingWorkerPlayerClear
 //------------------------------------------------------------------------------------------------------------------------------------------------------------------------------


 
 
 
    
 /**-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------
  * 
  * Internal class needed to update the progress bars properly when scanning directories. 
  * 
  * @author Ulf Wagemann
  */
 class SwingWorkerDirectoryScan extends    SwingWorker
                                implements SkyrimCharacterHelperProgressNotifier 
 {
  private File    m_tDirectory    ;
  private boolean m_bMode         ;
  private boolean m_bShowMessage  ;
  private boolean m_bIsRescan     ;
  private String  m_sMessageString;
  private SkyrimCharacterHelperPlayerManager m_tCharacterManager;
  
  
  
  /**
   * Constructor
   * 
   */
  public SwingWorkerDirectoryScan()
  {
   super();
   
   m_tCharacterManager = null ;
   m_tDirectory        = null ;
   m_sMessageString    = null ;
   m_bMode             = false;
   m_bIsRescan         = false;
   m_bShowMessage      = true ;
  }
  
  
  
  /**
   * Stores the directory name
   * 
   * @param pa_tCharacterManager char manager to use
   * @param pa_sDirectory        directory to scan  
   * @param pa_bMode             scanner mode, true = active save games, false = backup save games
   */
  public void init(String pa_sDirectory, boolean pa_bIsRescan, boolean pa_bMode, String pa_sMessageString, boolean pa_bShowMessage)
  {
   m_tDirectory        = new File(pa_sDirectory);
   m_tCharacterManager = (true == pa_bMode ? m_tPlayerManagerActiveSaveGames   : m_tPlayerManagerBackupSaveGames);   
   m_bMode             = pa_bMode         ;
   m_sMessageString    = pa_sMessageString;
   m_bShowMessage      = pa_bShowMessage  ;
   m_bIsRescan         = pa_bIsRescan     ;
  }
    
  
  
  /**
   * Returns the mode
   * 
   * @return scanner mode, true = save games, false = backup save games
   */
  public boolean getMode() {return m_bMode;}
  
  
  
  /**
   * This is where the lengthy stuff happens
   * 
   * @return 
   */
  public Object doInBackground() 
  {
   int lc_iCount = 0;

   File[]  lc_tSaveFiles         = null ;
   String  lc_sFileName          = null ;

   SkyrimCharacterHelperSaveGame lc_tSaveGame = null;
   
  
   if (null != m_tDirectory)
   {
    if (null != (lc_tSaveFiles = m_tDirectory.listFiles(new SkyrimCharacterHelperSaveFileFilter())))
    {
     if (0 < lc_tSaveFiles.length)
     {
      //
      // init progress display
      //
      m_tView.initTotalProgress(lc_tSaveFiles.length);
      m_tView.initCurrentProgress(100);
      
      if (true == m_bShowMessage) m_tView.setMessage(insertNumberString(m_sMessageString, lc_tSaveFiles.length)); 
      
      //
      // check all save-files
      // 
      for (File lc_tCurrentFile : lc_tSaveFiles)
      {       
       if (false == isCancelled())
       {
        try 
        {
         if (null != (lc_sFileName = lc_tCurrentFile.getCanonicalPath()))
         {
          m_tView.setStringCurrentProgress(lc_sFileName);
          
          if (null != (lc_tSaveGame = new SkyrimCharacterHelperSaveGame()))
          {
           if (true == lc_tSaveGame.read(lc_sFileName, (SkyrimCharacterHelperProgressNotifier) this, false)) // don't read screenshot data 
           {
            m_tCharacterManager.add(lc_tSaveGame); 
           }
          }
         }
        } 
        catch (IOException lc_tException) {}
       }
       else 
       {
        break;
       }
      
       //
       // update progress display
       //
       if (false == isCancelled()) m_tView.setTotalProgress(++lc_iCount);      
      } // for  
      m_tView.setStringCurrentProgress(null);
     }
    }
   }
   return null; 
  }  
  
  
  /**
   * Returns whether this thread is a resca thread
   * 
   * @return  true if it is, otherwiswe false
   */
  public final boolean isRescan() {return m_bIsRescan;}
  
  
  
  /**
   * Receives progress notification from reading a file and displays it
   * 
   * @param pa_iValue   new value
   */
  public void notifyProgress(int pa_iValue) 
  {
   if (false == isCancelled()) {m_tView.setCurrentProgress(pa_iValue);}
  }
  

  
  /**
   * When we're done, we notify the main thread about this
   * 
   */
  @Override
  protected void done() 
  {    
   handleScanWorkerThreadFinished(m_bMode, isCancelled());
  }  
  
  
  
 } // end of inner class SwingWorkerDirectoryScan
 //------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
 

  
  
 public SkyrimCharacterHelperApp()
 {
  super(); 

  m_bInitializing        = true ;
  m_bPrepareForLaunching           = false;
  m_bScanAll             = true ;
  
  m_tWorkerDirectoryScan   = null;
  m_tWorkerCopySaveGames   = null;
  m_tWorkerLoaderSaveGamesActive = null;
  m_tWorkerLoaderSaveGamesBackup = null;
  
  m_tLaunchThread          = null;
  
  m_tProperties          = new Properties();     
  m_sSeparator           = System.getProperty("file.separator");
  m_tLocalizedMessages   = ResourceBundle.getBundle(SkyrimCharacterHelperConstants.SKH_FILE_LOCALIZED_MESSAGES);  
  m_iSaveGameChanges     = 0;
  
  m_tPlayerManagerActiveSaveGames = new SkyrimCharacterHelperPlayerManager();
  m_tPlayerManagerBackupSaveGames = new SkyrimCharacterHelperPlayerManager();
 }
 
 
 
 /**
  * At startup create and show the main frame of the application.
  * 
  */
 @Override protected void startup() 
 {
  String lc_sActivePath             = "";
  String lc_sBackupPath             = "";
  String lc_sWorkModeString         = null;
  String lc_sConfirmationModeString = null;
  String lc_sOverwriteModeString    = null;
  String lc_sMinimizeString         = null;
  String lc_sImageModeString        = null;
  String lc_sLaunchBinary           = null;
  String lc_sAutoSaveString         = null;
  String lc_sQuickSaveString        = null;
  String lc_sPrefixString           = null;
  String lc_sLaunchParameters       = null;
  String lc_sInitModeString         = null;

  boolean lc_bFreshStart = true;
  
  if (null != m_tProperties)    
  {
   
   //
   // load properties
   //
   try
   {
    if (true == loadProperties())
    {
     lc_sActivePath             = checkPath(getPropertyString(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_ACTIVE_PATH  ));
     lc_sBackupPath             = checkPath(getPropertyString(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_BACKUP_PATH  ));
     lc_sLaunchBinary           = checkExecutable(getPropertyString(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_LAUNCH_BINARY));

     lc_sWorkModeString         = getPropertyString(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_WORK_MODE        );
     lc_sConfirmationModeString = getPropertyString(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_CONFIRMATION_MODE);
     lc_sOverwriteModeString    = getPropertyString(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_OVERWRITE_MODE   );
     lc_sInitModeString         = getPropertyString(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_INIT_MODE        );
     lc_sMinimizeString         = getPropertyString(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_MINIMIZE_MODE    );
     lc_sImageModeString        = getPropertyString(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_IMAGE_MODE       );
     lc_sAutoSaveString         = getPropertyString(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_AUTOSAVE_MODE    );
     lc_sQuickSaveString        = getPropertyString(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_QUICKSAVE_MODE   );
     lc_sPrefixString           = getPropertyString(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_PREFIX           );
     lc_sLaunchParameters       = getPropertyString(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_LAUNCH_PARAMETERS);
     lc_bFreshStart             = false;
    }
    m_tProperties.put(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_ACTIVE_PATH  , lc_sActivePath  );
    m_tProperties.put(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_BACKUP_PATH  , lc_sBackupPath  );
   }
   catch (Exception lc_tException) {} 
   
   //
   // check for scan mode
   //
   if (true == checkString(lc_sInitModeString))
   {
    m_bScanAll = (SkyrimCharacterHelperConstants.SKH_INIT_SCAN_ALL == Integer.valueOf(lc_sInitModeString));
   }
   
   //
   // check for valid lauch binary   
   //
   if (false == checkString(lc_sLaunchBinary))
   {
    lc_sLaunchBinary = getSkyrimInstallationDirectory(); 
   }      
   m_tProperties.put(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_LAUNCH_BINARY, lc_sLaunchBinary);     

   //
   // check for equal directories
   //
   if (null != lc_sActivePath && null != lc_sBackupPath)
   {
    if (lc_sActivePath.equals(lc_sBackupPath))
    {
     m_tProperties.put(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_BACKUP_PATH, ""); 
     lc_sBackupPath = null;
    }
   }

   //
   // open window, the windows restores the correct column header order for both savegame tables from the given
   // property strings
   //
   if (null != (m_tView = new SkyrimCharacterHelperView((SkyrimCharacterHelperGuiConnector) this, 
                                                        (WindowListener) this                   ,
                                                        getLocalizedComboBoxOptionsWorkMode()   , 
                                                        getLocalizedYesNoOptions()      ,
                                                        getLocalizedComboBoxOptionsLaunchMode() ,
                                                        getLocalizedComboBoxOptionsImageMode()  ,
                                                        getLocalizedComboBoxOptionsInitMode()   ,
                                                        getMonitoringDisplays()                 ,
                                                        getLocalizedTableHeaders()              ,
                                                        getPropertyString(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_HEADER_ACTIVE    ),
                                                        getPropertyString(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_HEADER_BACKUP    ),
                                                        getPropertyString(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_SORT_ORDER_ACTIVE),
                                                        getPropertyString(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_SORT_ORDER_BACKUP)
                                                        )))
   {
    show(m_tView);

    //
    // show fresh start requester, if needed
    //
    if (true == lc_bFreshStart)
    {
     m_tView.setUILockMode(true, false);
     JOptionPane.showMessageDialog(m_tView, 
                                   getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TEXT_FRESH_START), 
                                   getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TITLE_INFORMATION), JOptionPane.INFORMATION_MESSAGE);              
     m_tView.setUILockMode(false, false);
    }

    //
    // check if we need to set Skyrim's savegame folder
    //
    if (false == checkString(lc_sActivePath))
    {
     lc_sActivePath = tryGuessSkyrimSaveGameFolder();      
    }
    
    //
    // check for empty prefix
    //
    if (false == checkString(lc_sPrefixString))   
    {
     lc_sPrefixString = tryGuessPrefix(lc_sActivePath, true); 
    }  
    
    //
    // update ui according to properties
    //
    m_tView.setFileMode        (null != lc_sWorkModeString         ? Integer.valueOf(lc_sWorkModeString        ) : 0);
    m_tView.setConfirmationMode(null != lc_sConfirmationModeString ? Integer.valueOf(lc_sConfirmationModeString) : 0);
    m_tView.setOverwriteMode   (null != lc_sOverwriteModeString    ? Integer.valueOf(lc_sOverwriteModeString   ) : 0);
    m_tView.setMinimizeMode    (null != lc_sMinimizeString         ? Integer.valueOf(lc_sMinimizeString        ) : 0);
    m_tView.setAutoSaveMode    (null != lc_sAutoSaveString         ? Integer.valueOf(lc_sAutoSaveString        ) : 0);
    m_tView.setQuickSaveMode   (null != lc_sQuickSaveString        ? Integer.valueOf(lc_sQuickSaveString       ) : 0);
    m_tView.setImageMode       (null != lc_sImageModeString        ? Integer.valueOf(lc_sImageModeString       ) : 0);
    m_tView.setInitMode        (null != lc_sInitModeString         ? Integer.valueOf(lc_sInitModeString        ) : 0);
    
    m_tView.setActivePath(lc_sActivePath);
    m_tView.setBackupPath(lc_sBackupPath);
    
    m_tView.setPrefix       (lc_sPrefixString    );
    m_tView.setLaunchBinary (lc_sLaunchBinary    );
    m_tView.setLaunchOptions(lc_sLaunchParameters);
    
    m_tView.setOnSkyrim(false);
    
    //
    // start scanning 
    //
    if (true == checkString(lc_sActivePath))
    {
     //
     // scan directories. we first scan Skyrim's directory, when this thread returns, it will start
     // the scan of the destination directory, since m_bInitializing is false until both of them have
     // been executed. see method handleScanWorkerThreadFinished() for this handling.
     //
     scanActiveSaveGamesDirectory(lc_sActivePath, getLocalizedString(SkyrimCharacterHelperConstants.SKH_MESSAGE_SCANNING_ACTIVE_SAVEGAMES), true, false);
    }
    else
    {
     //
     // there is no source path, so there is no thread invoking the second scanning thread. to handle this,
     // we start the second thread manually and set the m_bInitializing to false, so that when this thread ends,
     // we're in sync again. 
     //
     if (true == checkString(lc_sBackupPath) && true == m_bScanAll) // maybe we are commanded not to scan... 
     {
      scanBackupSaveGamesDirectory(lc_sBackupPath, false);
     }
     else
     {
      m_bInitializing = false; // when initializing, we're through now 
      m_tView.updateUIElements();
     }
    }
  
   }
  }     
 }
 
 
 
 /**
  * This method is to initialize the specified window by injecting resources.
  * Windows shown in our application come fully initialized from the GUI
  * builder, so this additional configuration is not needed.
  */
 @Override protected void configureWindow(java.awt.Window root) {}

 
 
 /**
  * A convenient static getter for the application instance.
  * 
  * @return the instance of SkyrimCharacterHelperApp
  */
 public static SkyrimCharacterHelperApp getApplication() {return Application.getInstance(SkyrimCharacterHelperApp.class);}

  
  
 /**
  * Main method launching the application.
  * 
  */
  public static void main(String[] args) {launch(SkyrimCharacterHelperApp.class, args);}
  
  
  
  
  
 //
 // -------------------------------------------------------------------------------------------------------------------------------------------------- start of PUBLIC non-BSAF methods
 //

 
 
 
 /**
  * WindowListener.windowClosing()
  * 
  * @param pa_tEvent  Event
  */ 
  public void windowClosing(WindowEvent pa_tEvent) {handleProgramExit();}

 
 
 /**
  * WindowListener.windowClosed()
  * 
  * @param pa_tEvent  Event
  */ 
  public void windowClosed(WindowEvent pa_tEvent) 
  {
   System.exit((int) SkyrimCharacterHelperConstants.SKH_FILE_SYSTEM_COMMAND_OK);    
  }

 
 
  
 /**
  * WindowListener.windowOpened()
  *
  * As soon as the main window is open, start the intial scan(s)
  *
  * @param pa_tEvent  Event
  */ 
 public void windowOpened(WindowEvent pa_tEvent) {}


 
 /**
  * WindowListener.windowIconified()
  * 
  * @param pa_tEvent  Event
  */ 
 public void windowIconified(WindowEvent pa_tEvent) {}



 /**
  * WindowListener.windowDeiconified()
  * 
  * @param pa_tEvent  Event
  */ 
 public void windowDeiconified(WindowEvent pa_tEvent) {}



 /**
  * WindowListener.windowActivated()
  * 
  * @param pa_tEvent  Event
  */ 
 public void windowActivated(WindowEvent pa_tEvent) {}



 /**
  * WindowListener.windowDeactivated()
  * 
  * @param pa_tEvent  Event
  */ 
 public void windowDeactivated(WindowEvent pa_tEvent) {}
 
 

 /**
  * Handles clicking the exit button
  * 
  */
 public void handleButtonExit() {handleProgramExit();}

 
 
 /**
  * Handles clicking the cancel button and stops any ongoing processing 
  * TODO : threadid mit auswerten!
  */
 public void handleButtonProcessCancel()
 {
  int     lc_iAmount     = 0;
  String  lc_sMessage    = getLocalizedString(SkyrimCharacterHelperConstants.SKH_MESSAGE_INTERRUPTED);
  String  lc_sMessageKey = null;
  boolean lc_bMode       = false;
  
  //
  // cancel any init or launch process
  //
  m_bInitializing = false; 
  m_bPrepareForLaunching    = false;
  
  //
  // check scan thread. just kick it.
  //
  if (null != m_tWorkerDirectoryScan) 
  {
   m_tWorkerDirectoryScan.cancel(true); // do not null it, let it return!  
  } 
  
  try
  {
   //
   // check copy thread. first get information, then kick it
   //
   if (null != m_tWorkerCopySaveGames) 
   {
    lc_iAmount = m_tWorkerCopySaveGames.getProcessedAmount();
    lc_bMode   = m_tWorkerCopySaveGames.getMode();
    lc_sMessageKey = (true == lc_bMode ? SkyrimCharacterHelperConstants.SKH_MESSAGE_BACKUP_DONE : SkyrimCharacterHelperConstants.SKH_MESSAGE_RESTORE_DONE); 
    m_tWorkerCopySaveGames.cancel(true); // do not null it, let it return!
   }
  }
  catch (Exception lc_tException)
  {}
  
  try
  {
   //
   // check delete thread. first get information, then kick it
   //
   if (null != m_tWorkerDeleteSaveGames) 
   {
    lc_iAmount = m_tWorkerDeleteSaveGames.getProcessedAmount();
    lc_sMessageKey = SkyrimCharacterHelperConstants.SKH_MESSAGE_DELETE_DONE; 
    m_tWorkerDeleteSaveGames.cancel(true); // do not null it, let it return!
   }
  }
  catch (Exception lc_tException)
  {}
  
  try
  {
   //
   // 4. check launcher, reset thread and reflag UI
   //
   if (null != m_tLaunchThread) 
   {
    m_tView.setLaunched(false);
    m_tView.setOnSkyrim(false);
    
    m_tLaunchThread.terminate();
    m_tLaunchThread = null     ; // NULL IT!
    
    m_tView.setMessage(insertNumberString(getLocalizedString(SkyrimCharacterHelperConstants.SKH_MESSAGE_LAUNCH_ABORTED), m_iSaveGameChanges));  
   }
  }
  catch (Exception lc_tException)
  {}

  try
  {
   //
   // 5. check savegame workers, reset thread 
   //
   if (null != m_tWorkerLoaderSaveGamesActive) m_tWorkerLoaderSaveGamesActive.cancel(true); // do not null it, let it return!
   if (null != m_tWorkerLoaderSaveGamesBackup) m_tWorkerLoaderSaveGamesBackup.cancel(true); // do not null it, let it return!
  }
  catch (Exception lc_tException)
  {}

  //
  // if possible, display what happened until cancellation
  //
  if (0 < lc_iAmount)
  {
   if (null != lc_sMessageKey)
   {
    lc_sMessage += " " + insertNumberString(getLocalizedString(lc_sMessageKey), lc_iAmount);  
    m_tView.setMessage(lc_sMessage);
   }
  }
   
  //
  // update UI and cancel any intialization which may go on
  //
  m_tView.setUILockMode(false, false);       
  m_tView.resetProgress();
 }

 
 
 /**
  * Brings up the about dialog
  * 
  */
 public void handleButtonAbout()
 {
  SkyrimCharacterHelperAboutDialog lc_tAboutDialog = null;
  
  if (null != (lc_tAboutDialog = new SkyrimCharacterHelperAboutDialog(m_tView, true)))
  {
   m_tView.setUILockMode(true, true);
   lc_tAboutDialog.setVisible(true);
   m_tView.setUILockMode(false, false);
  }
 }

 
 
 /**
  * Handles clicking the Path button for the active player list
  * 
  */
 public void handleButtonSelectActivePath() {performSelectPath(true);}

 
 
 /**
  * Handles clicking the Path button for the backup player list
  * 
  */
 public void handleButtonSelectBackupPath() {performSelectPath(false);}

 
 
 /**
  * Handles a list selection in the active characters list
  * 
  */
 public void handleListSelectionActivePlayers() {performCharacterListSelection(true);}
 
 

 /**
  * Handles a list selection in the active characters list
  * 
  */
 public void handleListSelectionBackupPlayers() {performCharacterListSelection(false);}
 
 

 /**
  * Handles a list selection in the active save games list. In case of a multiselection,
  * nothing is displayed
  * 
  */
 public void handleListSelectionActiveSaveGames() {performSaveGameListSelection(true);}
 
 
 
 /**
  * Handles a list selection in the backup save games list. In case of a multiselection,
  * nothing is displayed
  * 
  */
 public void handleListSelectionBackupSaveGames()  {performSaveGameListSelection(false);} 


 
 /**
  * Handles clicking the rescan button for active savegames
  * 
  */
 public void handleButtonSelectActiveRescan(String pa_sMessage, boolean pa_bShowMessage)
 {
  scanActiveSaveGamesDirectory(checkPath(getPropertyString(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_ACTIVE_PATH)), pa_sMessage, pa_bShowMessage, true); 
 }
 
 
 
 /**
  * Handles clicking the rescan button for backup savegames
  * 
  */
 public void handleButtonSelectBackupRescan()
 {
  scanBackupSaveGamesDirectory(checkPath(getPropertyString(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_BACKUP_PATH)), true); 
 }


 
 /**
  * Handles clicking theclipboard button for the active savegame
  * 
  */
 public void handleButtonActiveClipboard()
 {
  copyImageToClipboard(m_tView.getActiveSaveGameScreenshot()); 
  m_tView.setMessage(getLocalizedString(SkyrimCharacterHelperConstants.SKH_MESSAGE_CLIPBOARD_DONE));    
 }
 
 
 
 /**
  * Handles clicking the clipboard button for the backup savegame
  * 
  */
 public void handleButtonBackupClipboard()
 {
  copyImageToClipboard(m_tView.getBackupSaveGameScreenshot()); 
  m_tView.setMessage(getLocalizedString(SkyrimCharacterHelperConstants.SKH_MESSAGE_CLIPBOARD_DONE));    
 }


 
 /**
  * Handles clicking delete button for the active player list
  * 
  */
 public void handleButtonActiveDeletePlayer()
 {
  performDeletePlayers(true);
 }

 
 
 /**
  * Handles clicking delete button for the backup player list
  * 
  */
 public void handleButtonBackupDeletePlayer()
 {
  performDeletePlayers(false);
 }

 
 
 /**
  * Handles clicking clear button for the active players list
  * 
  */
 public void handleButtonActiveClearPlayers()
 {
  performClearPlayers(true); 
 }

 
 /**
  * Handles clicking clear button for the backup players list
  * 
  */
 public void handleButtonBackupClearPlayers()
 {
  performClearPlayers(false); 
 }
 

 
 /**
  * Handles clicking delete button for the active savegame list
  * 
  */
 public void handleButtonActiveDeleteSaveGame()
 {
  performDeleteSaveGames(true);
 }
 
 

 /**
  * Handles clicking delete button for the backup savegame list
  * 
  */
 public void handleButtonBackupDeleteSaveGame() 
 {
  performDeleteSaveGames(false);
 }
 
 
 
 /**
  * Handles clicking the backup button in the active savegame list
  * 
  */
 public void handleButtonActiveBackupSaveGame() 
 {
  performCopySaveGames(true);
 }
 

    
/**
 * Handles clicking the restore button in the backup savegame list
 * 
 */
 public void handleButtonBackupSaveGameRestore()
 {
  performCopySaveGames(false);
 }
  
  
 
 /**
  * Handles clicking the backup button in the active player list
  * 
  */
 public void handleButtonActiveBackupPlayer() 
 {
  performCopyPlayers(true);
 }
  
 
 
 /**
  * Handles clicking the backup button in the active player list
  * 
  */
 public void handleButtonBackupPlayerRestore()
 {
  performCopyPlayers(false);
 }
 


 
 /**
  * Handles clicking the launch button for the backup player list
  * 
  * Step 1: move anything belonging to the currently selected backup player to Skyrim's savegame directory and select this player
  * Step 2: move anything which does not belong to the currently selected active player to the backup directory
  * Step 3: launch Skyrim
  * 
  */
 public void handleButtonBackupLaunchPlayer()
 {
  boolean lc_bContinue = true;
  int     lc_iCount    = 0   ;
  
  SkyrimCharacterHelperPlayer         lc_tSelectedBackupPlayer          = m_tView.getFirstSelectedBackupPlayer();  
  SkyrimCharacterHelperPlayer         lc_tSelectedDestinationPlayer     = null                                  ;  
  List<SkyrimCharacterHelperSaveGame> lc_tSaveGames                     = null;
  String                              lc_sSelectedDestinationPlayerName = null;
  
  if (null != lc_tSelectedBackupPlayer)
  {
   //
   // delete warning
   //
   if (SkyrimCharacterHelperConstants.SKH_CONFIRM_DELETE == m_tView.getConfirmationMode())    
   {
    lc_bContinue = (0 == JOptionPane.showOptionDialog(m_tView,  
                                                      insertString(getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TEXT_MOVE_FOR_LAUNCH_BACKUP_PLAYER) , lc_tSelectedBackupPlayer.getName()), 
                                                      getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TITLE_WARNING), 
                                                      JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE, null, 
                                                      getLocalizedOkCancelOptions()                                   , 
                                                      getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_OPTION_OK)));
   }
   if (true == lc_bContinue)
   {  
    //
    // get savegames of concerned players
    // 
    if (null != (lc_tSaveGames = getSaveGamesListFromPlayers(false, true)))
    {
     //
     // overwrite warning
     // 
     if (SkyrimCharacterHelperConstants.SKH_CONFIRM_YES == m_tView.getOverwriteMode() && 0 < (lc_iCount = getAmountOfFileCollisions(lc_tSaveGames, false)))
     {
      lc_bContinue = (0 == JOptionPane.showOptionDialog(m_tView,  
                                                   insertNumberString(getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TEXT_OVERWRITE_SAVEGAME), lc_iCount), 
                                                   getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TITLE_WARNING), 
                                                   JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE, null, 
                                                   getLocalizedOkCancelOptions()                                   , 
                                                   getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_OPTION_OK)));
     }
     
     //
     // on continue, set launch flag
     //
     if (true == lc_bContinue)
     {
      m_bPrepareForLaunching = true; 
      m_sLaunchPlayer = lc_tSelectedBackupPlayer.getName();
      
      //
      // backup the savegames. when this thread returns, the handler-routine will launch Skyrim.
      //
      if (null != (m_tWorkerCopySaveGames = new SwingWorkerSaveGameCopy()))
      {
       lc_tSelectedDestinationPlayer     = m_tView.getFirstSelectedActivePlayer();
       lc_sSelectedDestinationPlayerName = (null != lc_tSelectedDestinationPlayer ? lc_tSelectedDestinationPlayer.getName() : null);
        
       m_tView.setMessage(getLocalizedString(SkyrimCharacterHelperConstants.SKH_MESSAGE_RESTORING));          
       m_tView.setUILockMode(true, false);   
       
       m_tWorkerCopySaveGames.init(SkyrimCharacterHelperConstants.SKH_WORKER_THREAD_LAUNCH_FROM_BACKUP,
                                   SkyrimCharacterHelperConstants.SKH_WORKMODE_MOVE,
                                   lc_tSaveGames, 
                                   false, 
                                   m_tView.hasListMultiSelectionActivePlayers(),
                                   lc_sSelectedDestinationPlayerName);
       
       m_tWorkerCopySaveGames.execute(); 
      }     
     }
    }
   }
  }   
 }


  
  
 /**
  * Handles clicking the launch button for the active player list
  * 
  * Step 1: move anything which does not belong to the currently selected active player to the backup directory
  * Step 2: launch Skyrim
  * 
  */
 public void handleButtonActiveLaunchPlayer()
 {
  boolean lc_bContinue      = true;
  boolean lc_bJustOnePlayer = m_tView.hasOneActivePlayer();
  int     lc_iCount         = 0   ;
  
  SkyrimCharacterHelperPlayer         lc_tSelectedActivePlayer          = m_tView.getFirstSelectedActivePlayer();  
  SkyrimCharacterHelperPlayer         lc_tSelectedDestinationPlayer     = null                                  ;  
  List<SkyrimCharacterHelperSaveGame> lc_tSaveGames                     = null;
  String                              lc_sSelectedDestinationPlayerName = null;
  
  if (null != lc_tSelectedActivePlayer)
  {
   //
   // delete warning
   //
   if (false == lc_bJustOnePlayer && SkyrimCharacterHelperConstants.SKH_CONFIRM_DELETE == m_tView.getConfirmationMode())    
   {
    lc_bContinue = (0 == JOptionPane.showOptionDialog(m_tView,  
                                                      insertString(getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TEXT_MOVE_FOR_LAUNCH_ACTIVE_PLAYER), lc_tSelectedActivePlayer.getName()), 
                                                      getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TITLE_WARNING         ), 
                                                      JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE, null, 
                                                      getLocalizedOkCancelOptions()                                   , 
                                                      getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_OPTION_OK)));
   }
   if (true == lc_bContinue)
   {  
    if (null != (lc_tSaveGames = getSaveGamesExceptSelectedPlayer(true)))
    {
     //
     // overwrite warning
     // 
     if (SkyrimCharacterHelperConstants.SKH_CONFIRM_YES == m_tView.getOverwriteMode() && 0 < (lc_iCount = getAmountOfFileCollisions(lc_tSaveGames, true)))
     {
      lc_bContinue = (0 == JOptionPane.showOptionDialog(m_tView,  
                                                   insertNumberString(getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TEXT_OVERWRITE_SAVEGAME), lc_iCount), 
                                                   getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TITLE_WARNING), 
                                                   JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE, null, 
                                                   getLocalizedOkCancelOptions()                                   , 
                                                   getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_OPTION_OK)));
     }
     
     //
     // on continue, set launch flag
     //
     if (true == lc_bContinue)
     {
      m_bPrepareForLaunching    = true; 
      m_sLaunchPlayer = lc_tSelectedActivePlayer.getName();

      if (false == lc_bJustOnePlayer)
      {
       //
       // backup the savegames. when this thread returns, the handler-routine will launch Skyrim.
       //
       if (null != (m_tWorkerCopySaveGames = new SwingWorkerSaveGameCopy()))
       {
        lc_tSelectedDestinationPlayer     = m_tView.getFirstSelectedBackupPlayer();
        lc_sSelectedDestinationPlayerName = (null != lc_tSelectedDestinationPlayer ? lc_tSelectedDestinationPlayer.getName() : null);
         
        m_tView.setMessage(getLocalizedString(SkyrimCharacterHelperConstants.SKH_MESSAGE_BACKING_UP));          
        m_tView.setUILockMode(true, false);   
        
        m_tWorkerCopySaveGames.init(SkyrimCharacterHelperConstants.SKH_WORKER_THREAD_LAUNCH_FROM_ACTIVE,
                                    SkyrimCharacterHelperConstants.SKH_WORKMODE_MOVE,
                                    lc_tSaveGames, 
                                    true, 
                                    m_tView.hasListMultiSelectionBackupPlayers(),
                                    lc_sSelectedDestinationPlayerName);
       
        m_tWorkerCopySaveGames.execute(); 
       }
      }
      else
      {
       m_bPrepareForLaunching = false;
       m_tView.setMessage(getLocalizedString(SkyrimCharacterHelperConstants.SKH_MESSAGE_LAUNCH_SINGLE));       
       launchSkyrim(); 
      }
     }
    }
    else // fix 08.10.2012
    {
     m_bPrepareForLaunching = false;
     m_tView.setMessage(getLocalizedString(SkyrimCharacterHelperConstants.SKH_MESSAGE_LAUNCH_SINGLE));       
     launchSkyrim(); 
    }
   }
  }
 }



 /**
  * Handles clicking the choose binary button
  * 
  * 
  */
 public void handleButtonChooseBinary()
 {
  String lc_sFile      = getPropertyString(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_LAUNCH_BINARY);
  String lc_sNewFile   = null;
  
  if (null != m_tProperties)
  {
   if (null != (lc_sNewFile = handleFileRequesterFile(lc_sFile, getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TITLE_FILEREQUESTER_LAUNCH_BINARY))))
   {
    if (true == checkString(lc_sNewFile))
    {
     if (null != checkExecutable(lc_sNewFile))
     {
      m_tProperties.put(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_LAUNCH_BINARY, lc_sNewFile);
      m_tView.setLaunchBinary(lc_sNewFile);
     }
     else
     {
      m_tView.setMessage(getLocalizedString(SkyrimCharacterHelperConstants.SKH_MESSAGE_NOT_EXECUTABLE));    
     }
    }
   }
  }
 }
 
 

 /**
  * Handles clicking the restore button for the launch path
  * 
  */
 public void handleButtonRestoreLaunchPath()
 {
  String lc_sLaunchBinary = getSkyrimInstallationDirectory(); 
   
  m_tProperties.put(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_LAUNCH_BINARY, lc_sLaunchBinary);     
  m_tView.setLaunchBinary(lc_sLaunchBinary);     
  m_tView.setMessage(getLocalizedString(SkyrimCharacterHelperConstants.SKH_MESSAGE_LAUNCH_RESTORED));    
 }
 
 
 
 /**
  * Handles clicking the launch solo button
  * 
  */
 public void handleButtonLaunchSolo()
 {
  m_tView.setMessage(getLocalizedString(SkyrimCharacterHelperConstants.SKH_MESSAGE_LAUNCH_SINGLE)); 
  launchSkyrim(); 
 }
 
 
 
 
 //
 // -------------------------------------------------------------------------------------------------------------------------------------------------- start of PRIVATE non-BSAF methods
 // 
 
 
 
 /**
  * Checks whether the given path is a valid readable and writeable existing directory
  * 
  * @param pa_sPath   path
  * @return null or path
  */
 private String checkPath(String pa_sPath)
 {
  File   lc_tFile = null;
  String lc_sResult = null;
  
  try
  {
   if (null != pa_sPath)
   {
    if (null != (lc_tFile = new File(pa_sPath)))
    {
     if (true == lc_tFile.exists())
     {
      if (true == lc_tFile.isDirectory() && true == lc_tFile.canRead() && true == lc_tFile.canWrite()) 
      {
       lc_sResult = pa_sPath; 
      }
     }
    }
   }
  }
  catch (Exception lc_tException) {}
  return lc_sResult;
 }
 
 
 
 /**
  * Checks whether the given  path is a valid executable file
  * 
  * @param pa_sPath   path
  * @return "" or path
  */
 private String checkExecutable(String pa_sPath)
 {
  File   lc_tFile   = null;
  String lc_sResult = null;
  
  try
  {
   if (null != pa_sPath)
   {
    if (null != (lc_tFile = new File(pa_sPath)))
    {
     if (true == lc_tFile.exists())
     {
      if (true == lc_tFile.isFile() && true == lc_tFile.canExecute()) 
      {
       lc_sResult = pa_sPath; 
      }
     }
    }
   }
  }
  catch (Exception lc_tException) {}  
  return lc_sResult;
 }
 
 
 
 /**
  * Returns the Skyrim installation path retrieved through the Windows registry
  * 
  * @return   null in case of error, otherwise Skyrim's install path
  */
  private String getSkyrimInstallationDirectory()
  {
   String lc_sSkyrimBinaryNormal   = null;

   String lc_sSkyrimDirectory32Bit = getRegistryValue(SkyrimCharacterHelperConstants.SKH_REGISTRY_QUERY_STRING_32BIT);
   String lc_sSkyrimDirectory64Bit = getRegistryValue(SkyrimCharacterHelperConstants.SKH_REGISTRY_QUERY_STRING_64BIT);
   String lc_sSkyrimPath           = (null != lc_sSkyrimDirectory32Bit ? lc_sSkyrimDirectory32Bit : lc_sSkyrimDirectory64Bit);
   
   if (null != lc_sSkyrimPath)
   {
    lc_sSkyrimBinaryNormal = checkSkyrimFile(lc_sSkyrimPath, SkyrimCharacterHelperConstants.SKH_SKYRIM_BINARY_LAUNCHER, true); // changed 08.02.2012
   }   
   return lc_sSkyrimBinaryNormal;
  }
  
  
 
 /**
  * Checks whether the give path is a valid readable and writeable existing directory
  * 
  * @param pa_sPath   path
  * @return null or path
  */
 private String checkSkyrimFile(String pa_sPath, String pa_sFile, boolean pa_bCheckExecute)
 {
  File         lc_tFile    = null;
  String       lc_sResult  = null;
  StringBuffer lc_tNewPath = new StringBuffer();
  
  boolean lc_bOk = false;
  
  if (null != lc_tNewPath && null != pa_sFile && null != pa_sPath)
  {
   lc_tNewPath.append(pa_sPath);
   if (false == pa_sPath.endsWith(m_sSeparator)) lc_tNewPath.append(m_sSeparator);
   lc_tNewPath.append(pa_sFile);
   
   if (null != (lc_tFile = new File(lc_tNewPath.toString())))
   {
    if (true == lc_tFile.exists())
    {
     lc_bOk  = lc_tFile.isFile();
     lc_bOk &= (true == pa_bCheckExecute ? true == lc_tFile.canExecute() : lc_tFile.canRead());

     if (true ==  lc_bOk)
     {
      lc_sResult = lc_tNewPath.toString(); 
     }
    }
   }
  }
  return (true == lc_bOk ? lc_sResult : null);
 }
 
 
 
 /**
  * Checks whether the given string is non-null and non-empty
  * 
  * @param pa_sValue   string
  * @return  true, if he given string is non-null and non-empty, false otherwise
  */
 private boolean checkString(String pa_sValue)
 {
  if (null != pa_sValue)
  {
   if (0 < pa_sValue.length()) 
   {
    return true; 
   }
  }
  return false; 
 }
  
 
 
 /**
  * Returns the value for the given Skyrim registry entry
  * 
  * @param pa_sEntry  32bit version value or 64 bit version value
  * @return  Skyrim installation path, may be null
  */
 private String getRegistryValue(String pa_sEntry)
 {
  InputStreamReader lc_tInputStreamReader = null;
  ProcessBuilder    lc_tProcessBuilder    = null;
  Process           lc_tRegistryQuery     = null;
  StringBuffer      lc_sRegReply          = new StringBuffer();   
  String            lc_sResult            = null;
  int               lc_iChar              = 0;
  int               lc_iPos               = -1;
  
  if (null != (lc_tProcessBuilder = new ProcessBuilder(SkyrimCharacterHelperConstants.SKH_REGISTRY_REG_COMMAND,
                                                       SkyrimCharacterHelperConstants.SKH_REGISTRY_QUERY      ,
                                                       pa_sEntry, 
                                                       SkyrimCharacterHelperConstants.SKH_REGISTRY_QUERY_OPTION,
                                                       SkyrimCharacterHelperConstants.SKH_REGISTRY_KEY_PATH)))
  {
   try 
   {
    if (null != lc_sRegReply )
    {
     if (null != (lc_tRegistryQuery = lc_tProcessBuilder.start()))
     {
      if (null != (lc_tInputStreamReader = new InputStreamReader(lc_tRegistryQuery.getInputStream())))
      {
       //
       // create on single string from the answer by ignoring control codes
       //
       // example answer: HKEY_LOCAL_MACHINE\Software\Wow6432Node\Bethesda Softworks\Skyrim    Installed Path    REG_SZ    c:\program files (x86)\steam\steamapps\common\skyrim\
       //
       while (-1 != (lc_iChar = lc_tInputStreamReader.read()))
       {
        if (lc_iChar >= 32)
          lc_sRegReply.append(Character.toChars(lc_iChar)); // respect unicode 
       }
       lc_tInputStreamReader.close();

       if (-1 != (lc_iPos = lc_sRegReply.indexOf(SkyrimCharacterHelperConstants.SKH_REGISTRY_VALUE_TYPE)))
       {
        lc_sResult = lc_sRegReply.substring(lc_iPos + SkyrimCharacterHelperConstants.SKH_REGISTRY_VALUE_TYPE.length());
       }
      }
     }
    }
   }
   catch (IOException lc_tException) {}
  }
  return (null != lc_sResult ? lc_sResult.trim() : null);
 }
  
 
 
 /**
  * Launches Skyrim
  * 
  */ 
 private void launchSkyrim()
 {
  String lc_sLaunchBinary = getPropertyString(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_LAUNCH_BINARY);     
  
  if (null == m_tLaunchThread)
  {
   if (null != (m_tLaunchThread = new SkyrimCharacterHelperLaunchThread((SkyrimCharacterHelperThreadNotifier) this, 
                                                                        lc_sLaunchBinary, 
                                                                        m_tView.getPrefix(), 
                                                                        m_tView.getActivePath(),
                                                                        m_tView.getLaunchOptions(),
                                                                        (SkyrimCharacterHelperConstants.SKH_AUTOSAVE_YES  == m_tView.getAutoSaveMode ()),
                                                                        (SkyrimCharacterHelperConstants.SKH_QUICKSAVE_YES == m_tView.getQuickSaveMode()) ))) 
   {
    m_iSaveGameChanges = 0; 
    m_tLaunchThread.start(); 
    m_tView.setLaunched(true);
    m_tView.setOnSkyrim(true);
    m_tView.setUILockMode(true, false);
   }
  
   //
   // minimize, if needed
   //
   if (SkyrimCharacterHelperConstants.SKH_CONFIRM_YES == m_tView.getMinimizeMode()) 
   {
    m_tView.setState(Frame.ICONIFIED);
   }
  }
 }
 
 
 
 /**
  * Tries guessing a prefix based on two skyrim ini files.
  * v1.15: savegame check added, code restructured 
  * 
  * @param pa_sSkyrimSaveGameFolder  Skyrims savegame fodler
  * @param pa_bUseSkyrimLocale       true = use Skyrim locale, false = don't 
  * 
  * @return  guess, may be null
  */
 private String tryGuessPrefix(String pa_sSkyrimSaveGameFolder, boolean pa_bUseSkyrimLocale)
 {
  String lc_sResult = null;
  
  String lc_sSkyrimDirectory32Bit = getRegistryValue(SkyrimCharacterHelperConstants.SKH_REGISTRY_QUERY_STRING_32BIT);
  String lc_sSkyrimDirectory64Bit = getRegistryValue(SkyrimCharacterHelperConstants.SKH_REGISTRY_QUERY_STRING_64BIT);
  String lc_sSkyrimPath           = (null != lc_sSkyrimDirectory32Bit ? lc_sSkyrimDirectory32Bit : lc_sSkyrimDirectory64Bit);
  String lc_sDefaultIniPath       = null;
  String lc_sLocaleIniValue       = null;
  String lc_sLocaleIniFile        = null;
  String lc_sSaveValue            = null;
  
  Locale       lc_tDefaultLocale = Locale.getDefault();
  StringBuffer lc_tResult = null;
  
  //
  // Use Skyrim's ini-files to find out the prefix?
  //
  if (true == pa_bUseSkyrimLocale)
  {
   if (null != (lc_sDefaultIniPath = checkSkyrimFile(lc_sSkyrimPath, SkyrimCharacterHelperConstants.SKH_FILE_DEFAULT_INI, false))) // check for read-access
   {
    //
    // first: read result1 =  <skyrimdir>\Skyrim_default.ini and search for sLanguage=
    //
    if (null != (lc_sLocaleIniValue = searchInFile(SkyrimCharacterHelperConstants.SKH_FILE_DEFAULT_INI_LANGUAGE_KEY, SkyrimCharacterHelperConstants.SKH_FILE_DEFAULT_INI_LANGUAGE_SEPARATOR, lc_sDefaultIniPath, false))) 
    {
     if (null != (lc_sLocaleIniFile = lc_sSkyrimPath + SkyrimCharacterHelperConstants.SKH_FILE_INTERFACE_FOLDER 
                                      + SkyrimCharacterHelperConstants.SKH_FILE_TRANSLATION + lc_sLocaleIniValue +  SkyrimCharacterHelperConstants.SKH_FILE_EXTENSION_TXT))
     {
      //
      // second : read <skyrimdir>\Data\Interface\Translate_<result1>.txt and search for $SAVE followes by a tabulator
      //
      if (null != (lc_sSaveValue = searchInFile(SkyrimCharacterHelperConstants.SKH_FILE_LANGUAGE_SAVE_KEY, SkyrimCharacterHelperConstants.SKH_FILE_LANGUAGE_SAVE_SEPARATOR, lc_sLocaleIniFile, true))) 
      {
       //
       //
       // 
       if (1 < lc_sSaveValue.length()) 
       {
        if (null != (lc_tResult = new StringBuffer()))         
        {
         lc_tResult.append(lc_sSaveValue.charAt(0)); 
         lc_tResult.append(lc_sSaveValue.substring(1).toLowerCase());
         lc_sResult = lc_tResult.toString();
        }
       }
      }    
     }
    }
   }
  }
  
  //
  // still no result? then we try to find any savegame and retrieve the prefix from its name
  //
  if (false == checkString(lc_sResult))
  {
   lc_sResult = tryFindPrefixBySaveGame(pa_sSkyrimSaveGameFolder);
  }
  
  //
  // still no result? oh oh...
  //
  if (false == checkString(lc_sResult) || false ==  pa_bUseSkyrimLocale)
  {
   if (null != lc_tDefaultLocale)
   {
    if      (lc_tDefaultLocale == Locale.GERMAN)                                        {return SkyrimCharacterHelperConstants.SKH_PREFIX_GERMAN ;}
    else if (lc_tDefaultLocale == Locale.CANADA || lc_tDefaultLocale == Locale.ENGLISH) {return SkyrimCharacterHelperConstants.SKH_PREFIX_ENGLISH;}
   }
  }
  return lc_sResult;
 }

 
 
 /**
  * Tries to find savegames in the given folder and return the prefix 
  * 
  * @param pa_sSkyrimSaveGameFolder   savegame folder
  * @return prefix, may be null
  */
 private String tryFindPrefixBySaveGame(String pa_sSkyrimSaveGameFolder)
 {
  File   lc_tDirectory    = new File(pa_sSkyrimSaveGameFolder); 
  File[] lc_tFiles        = null;
  String lc_sPrefix       = null;
  String lc_sSaveGameName = null;
  int    lc_iEnd          = -1;
  
  if (null != lc_tDirectory)
  {
   if (true == lc_tDirectory.exists() && true == lc_tDirectory.isDirectory())
   {
    if (null != (lc_tFiles = lc_tDirectory.listFiles()))
    {
     for (File lc_tFile : lc_tFiles) 
     {
      if (null != lc_tFile)
      {
       if (null != (lc_sSaveGameName = lc_tFile.getName()))
       {
        if (true == lc_sSaveGameName.toLowerCase().endsWith(SkyrimCharacterHelperConstants.SKH_FILE_EXTENSION_ESS))
        {
         if (-1 != (lc_iEnd =  lc_sSaveGameName.indexOf(SkyrimCharacterHelperConstants.SKH_FILE_PREFIX_SEPARATOR)))
         {
          try
          {
           lc_sPrefix = lc_sSaveGameName.substring(0, lc_iEnd);
           if (true == checkString(lc_sPrefix))
           {
            break;
           }
          }
          catch (Exception lc_tException)
          {
           lc_sPrefix = null;
          }      
         }
        }
       }
      } 
     } // for
    } 
   }
  }
  return lc_sPrefix;
 }
 
 
 
 
 /**
  * Reads the given textfile line by line, until the line with the given key appears. then, the remainder of that line is returned.
  * 
  * @param pa_sKey            key
  * @param pa_sFilePath       full file path
  * @return  null or value string
  */
 private String searchInFile(String pa_sKey, String pa_sSeparator, String pa_sFilePath, boolean pa_bLittleEndian)
 {
  File         lc_tFile     = null;
  String       lc_sResult   = null;
  String       lc_sLine     = null;
  String       lc_sWorkLine = null;
  
  BufferedReader    lc_tInputStringReader = null;
  InputStreamReader lc_tInputStreamReader = null;
  FileInputStream   lc_InputStream        = null;
  
  int     lc_iPos      = -1;
  int     lc_iStartPos = -1;
  boolean lc_bDone = false;
  
          
  
  if (null != pa_sKey && null != pa_sFilePath)
  {
   if (null != (lc_tFile = new File(pa_sFilePath)))
   {
    if (true == lc_tFile.exists())
    {
     if (true == lc_tFile.isFile() && true == lc_tFile.canRead())
     {
      try 
      {
       if (null != (lc_InputStream = new FileInputStream(lc_tFile)))
       {
        try 
        {
         if (null != (lc_tInputStreamReader = new InputStreamReader(lc_InputStream, (true == pa_bLittleEndian ? "UTF-16LE" : "UTF-8")))) // convert UCS-2 little endian to UTF using UTF-16LE in case of that language file
         {
          if (null != (lc_tInputStringReader = new BufferedReader(lc_tInputStreamReader)))
          {
           try 
           {
            while (false == lc_bDone && (null != (lc_sLine = lc_tInputStringReader.readLine())))          
            {
             if (null != (lc_sWorkLine = lc_sLine.trim()))
             {
              if (-1 != (lc_iPos = lc_sWorkLine.indexOf(pa_sKey)))
              {
               if (-1 != (lc_iStartPos = lc_sWorkLine.indexOf(pa_sSeparator, lc_iPos + pa_sKey.length())))
               {
                lc_sResult = lc_sWorkLine.substring(lc_iStartPos+1);
                lc_bDone = true;
               }
              }
             }
            } // while
           } 
           catch (IOException lc_tException) {}
          }
         }
        } 
        catch (UnsupportedEncodingException lc_tException) {}
       }
      } 
      catch (FileNotFoundException lc_tException) {}
     }
    }
   }
  }
  
  if (null != lc_tInputStringReader) try {lc_tInputStringReader.close(); lc_tInputStringReader = null;} catch (Exception lc_tException) {}
  if (null != lc_tInputStreamReader) try {lc_tInputStreamReader.close(); lc_tInputStreamReader = null;} catch (Exception lc_tException) {}
  if (null != lc_InputStream       ) try {lc_InputStream.close()       ; lc_InputStream        = null;} catch (Exception lc_tException) {}

  return lc_sResult;
 }
  

 
 /**
  * Scans the active savegame directory
  * 
  */
 private void scanActiveSaveGamesDirectory(String pa_sPath, String pa_sMessage, boolean pa_bShowMessage, boolean pa_bIsRescan)
 {
  String lc_sPath = getPropertyString(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_ACTIVE_PATH);
  
  //
  // always set pat, if it is null, it has to be reset! store current selections.
  //
  m_tView.setActivePath(pa_sPath); 
  m_tView.storePlayerSelection(true);
  m_tView.storeSaveGameSelection(true);
  m_tView.setActiveScanned(false);
  
  //
  // make sure we have a message
  //
  if (null == pa_sMessage)
  {
   pa_sMessage = getLocalizedString(SkyrimCharacterHelperConstants.SKH_MESSAGE_SCANNING_ACTIVE_SAVEGAMES);
  } 
  
  //
  // go!
  //
  if (null != lc_sPath && null != m_tPlayerManagerActiveSaveGames)
  {
   if (null != (m_tWorkerDirectoryScan = new SwingWorkerDirectoryScan()))
   {
    m_tPlayerManagerActiveSaveGames.clear();
    m_tView.resetAmountActive();

    m_tView.setActiveSaveGameScreenshot(null);
    m_tView.setActivePlayerRace(null);
    m_tView.clearActiveSaveGameTable();
    m_tView.clearActivePlayerList();
    m_tView.setUILockMode(true, false);   

    m_tWorkerDirectoryScan.init(lc_sPath, pa_bIsRescan, true, pa_sMessage, pa_bShowMessage); 
    m_tWorkerDirectoryScan.execute();      
   }
  }
 }
 
 
 /**
  * Scans the backup savegame directory
  * 
  */
 private void scanBackupSaveGamesDirectory(String pa_sPath, boolean pa_bIsRescan)
 {
  String lc_sPath = getPropertyString(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_BACKUP_PATH);

  //
  // always set path. if it is null, it has to be reset! store current selections.
  //
  m_tView.setBackupPath(pa_sPath);    
  m_tView.storePlayerSelection(false);
  m_tView.storeSaveGameSelection(false);
  m_tView.setBackupScanned(false);

  //
  // go!
  //
  if (null != lc_sPath && null != m_tPlayerManagerBackupSaveGames)
  { 
   if (null != (m_tWorkerDirectoryScan = new SwingWorkerDirectoryScan()))
   {
    m_tPlayerManagerBackupSaveGames.clear();
    m_tView.resetAmountBackup();

    m_tView.setBackupSaveGameScreenshot(null);
    m_tView.setBackupPlayerRace(null);
    m_tView.clearBackupSaveGameTable();
    m_tView.clearBackupPlayerList();
    m_tView.setUILockMode(true, false);   

    m_tWorkerDirectoryScan.init(lc_sPath, pa_bIsRescan, false, getLocalizedString(SkyrimCharacterHelperConstants.SKH_MESSAGE_SCANNING_BACKUP_SAVEGAMES), true); 
    m_tWorkerDirectoryScan.execute();      
   }
  }
  else
  {
   m_bInitializing = false; // if we're initializing and don't work on a path, we're through 
  }
 }

 
 
 /**
  * Handles finishing a non-directory-scan-worker thread
  * 
  * @param pa_iThreadModeId   thread mode id
  * @param pa_iNumberOfFiles  number of successfully processed files
  * @param pa_bMode             copy mode, true = copy active save games, false = copy backup save games
  */
 protected synchronized void handleNonScanWorkerThreadFinished(int pa_iThreadModeId, int pa_iNumberOfFiles, boolean pa_bMode)
 {
  String  lc_sMessageKey          = null ;
  boolean lc_bReloadScreenhotTest = false;
  boolean lc_bReloadScreenhotMode = false;
  boolean lc_bScrollToSelection   = false;
  boolean lc_bWasMoving           = false;
  
  switch(pa_iThreadModeId)
   {
   case  SkyrimCharacterHelperConstants.SKH_WORKER_THREAD_MODE_DELETE        : 
                                                                               lc_sMessageKey = SkyrimCharacterHelperConstants.SKH_MESSAGE_DELETE_DONE; 
                                                                               lc_bReloadScreenhotTest = true;
                                                                               lc_bScrollToSelection   = true;
                                                                               lc_bReloadScreenhotMode = pa_bMode;
                                                                               break;
   case  SkyrimCharacterHelperConstants.SKH_WORKER_THREAD_MODE_RESTORE       : 
                                                                               lc_sMessageKey = SkyrimCharacterHelperConstants.SKH_MESSAGE_RESTORE_DONE; 
                                                                               lc_bWasMoving           = (null != m_tWorkerCopySaveGames ? (m_tWorkerCopySaveGames.getWorkMode() == SkyrimCharacterHelperConstants.SKH_WORKMODE_MOVE) : false);
                                                                               lc_bScrollToSelection   = lc_bWasMoving;
                                                                               lc_bReloadScreenhotTest = true;
                                                                               lc_bReloadScreenhotMode = true;               // check destination image at end 
                                                                               m_tView.restoreSaveGameSelection(true, true); // restore destination selection
                                                                               break;
     
   case  SkyrimCharacterHelperConstants.SKH_WORKER_THREAD_MODE_BACKUP        : 
                                                                               lc_sMessageKey = SkyrimCharacterHelperConstants.SKH_MESSAGE_BACKUP_DONE; 
                                                                               lc_bWasMoving           = (null != m_tWorkerCopySaveGames ? (m_tWorkerCopySaveGames.getWorkMode() == SkyrimCharacterHelperConstants.SKH_WORKMODE_MOVE) : false);
                                                                               lc_bScrollToSelection   = lc_bWasMoving;
                                                                               lc_bReloadScreenhotTest = true;
                                                                               lc_bReloadScreenhotMode = false;               // check destination image at end 
                                                                               m_tView.restoreSaveGameSelection(false, true); // restore destination selection
                                                                               break;
     
   case  SkyrimCharacterHelperConstants.SKH_WORKER_THREAD_LAUNCH_FROM_ACTIVE : lc_sMessageKey = SkyrimCharacterHelperConstants.SKH_MESSAGE_LAUNCH_ACTIVE; break;
   case  SkyrimCharacterHelperConstants.SKH_WORKER_THREAD_LAUNCH_FROM_BACKUP : lc_sMessageKey = SkyrimCharacterHelperConstants.SKH_MESSAGE_LAUNCH_BACKUP; break;
   default: lc_sMessageKey = null;
  }

  //
  // null-out the thread which terminated to know which one is active when cancelling anyhting. 
  // 
  if (null != m_tWorkerCopySaveGames  ) {m_tWorkerCopySaveGames   = null;}
  if (null != m_tWorkerDeleteSaveGames) {m_tWorkerDeleteSaveGames = null;}

  //
  // update UI 
  //
  m_tView.setUILockMode(false, false);   
  m_tView.setMessage(null != lc_sMessageKey ? insertNumberString(getLocalizedString(lc_sMessageKey), pa_iNumberOfFiles) : "");    
  
  m_tView.setAmountActive(m_tPlayerManagerActiveSaveGames, m_tPlayerManagerActiveSaveGames.getSaveGameAmountForList(m_tView.getSelectedActivePlayers()));
  m_tView.setAmountBackup(m_tPlayerManagerBackupSaveGames, m_tPlayerManagerBackupSaveGames.getSaveGameAmountForList(m_tView.getSelectedBackupPlayers()));
  m_tView.resetProgress();
  
  //
  // final check: if we're launching an active player character, here we call Skyrim :)
  //
  if (pa_iThreadModeId == SkyrimCharacterHelperConstants.SKH_WORKER_THREAD_LAUNCH_FROM_ACTIVE)
  {
   m_bPrepareForLaunching = false;
   m_sLaunchPlayer = null;
   
   launchSkyrim(); 
  }
  if (pa_iThreadModeId == SkyrimCharacterHelperConstants.SKH_WORKER_THREAD_LAUNCH_FROM_BACKUP)
  {
   //
   // this means, the first step - restorig the selected backup toon - is finished. now character toon is contained
   // in Skyrim's player list and needs to be selected and then to be launched (which in turn means moving
   // anything not belonging to this character to the backup directory. we simulate that by just selecting the character
   // and then pretending a click on the Launch button :) 
   //
   //SKH_WORKER_THREAD_LAUNCH_FROM_BACKUP 
   if (null != m_tPlayerManagerActiveSaveGames)
   {
    m_tView.selectActivePlayer(m_tPlayerManagerActiveSaveGames.getPlayer(m_sLaunchPlayer));
    handleButtonActiveLaunchPlayer();
   }
  }
  
  //
  // make sure that the selection of the source is visible in case of moving savegames.we start invokaLater() to make sure that this request is handled after the previously
  // triggered selection change!
  //
  if (true == lc_bScrollToSelection)
  {
   final boolean lc_bFugly = pa_bMode;
   
   javax.swing.SwingUtilities.invokeLater(new Runnable()
   {
    public void run() 
    {
     m_tView.ensureSaveGameSelectionIsVisible(lc_bFugly); 
    }
   });       
  }
  
  //
  // check if we need to reload the screenshot. we start invokaLater() to make sure that this request is handled after the previously
  // triggered selection change! 
  //
  if (true == lc_bReloadScreenhotTest)
  {
   final boolean lc_bFugly  = lc_bReloadScreenhotMode;
   final boolean lc_bFugly2 = lc_bWasMoving;
   
   javax.swing.SwingUtilities.invokeLater(new Runnable()
   {
    public void run() 
    { 
     performSaveGameListSelection(lc_bFugly);   
     
     if (true == lc_bFugly2) // if we have moved stuff, we also need to restore the new source selection!
     {
      performSaveGameListSelection(!lc_bFugly);   
     }
    }
   });       
  }
 }

 
 
 /**
  * Handles a returning savegame thread
  * 
  * @param pa_bMode           mode, true = save games, false = backup save games
  * @param pa_bWasCancelled   true if the thread was cancelled
  */
 private synchronized void handleSaveGameWorkerThreadFinished(boolean pa_bMode, boolean pa_bWasCancelled)
 {
  SkyrimCharacterHelperSaveGame lc_tSaveGame = null;
  BufferedImage lc_tImage = null;
  boolean lc_bHasMultiSelection = (true == pa_bMode ? m_tView.hasListMultiSelectionActiveSaveGames() : m_tView.hasListMultiSelectionBackupSaveGames());
  
  //
  // set image
  //
  if (false == lc_bHasMultiSelection)
  {
   if ((null != m_tWorkerLoaderSaveGamesActive && true == pa_bMode) || (null != m_tWorkerLoaderSaveGamesBackup && false == pa_bMode))
   {
    if (null != (lc_tSaveGame = (true == pa_bMode ? m_tWorkerLoaderSaveGamesActive.getSaveGame() : m_tWorkerLoaderSaveGamesBackup.getSaveGame())))
    { 
     if (null != lc_tSaveGame.getScreenshot()) 
     {
      lc_tImage = lc_tSaveGame.getScreenshot().getImage();
     }
    }
   }
  }
  if (true == pa_bMode) m_tView.setActiveSaveGameScreenshot(lc_tImage);
  else                  m_tView.setBackupSaveGameScreenshot(lc_tImage); 
  
  m_tView.setUILockMode(false, false);   
  m_tView.setStringCurrentProgress(null);
  m_tView.resetProgress();    
  m_tView.setMessage(getLocalizedString(SkyrimCharacterHelperConstants.SKH_MESSAGE_READY));
  
  if (true == pa_bMode) m_tWorkerLoaderSaveGamesActive = null; // NULL it!
  else                  m_tWorkerLoaderSaveGamesBackup = null; // NULL it!
 }
 
 
 
 /**
  * Handles finishing a directory scan worker thread. This part implements the startup scan logic, it especially
  * cares for executing the two scanner threads sequentially to avoid messing up the progress display.
  * 
  * @param pa_bMode           scanner mode, true = save games, false = backup save games
  * @param pa_bWasCancelled   true if the thread was cancelled
  */
 private synchronized void handleScanWorkerThreadFinished(boolean pa_bMode, boolean pa_bWasCancelled)
 {
  boolean lc_bRescan = (null != m_tWorkerDirectoryScan ? m_tWorkerDirectoryScan.isRescan() : false);
   
  //
  // if a backup scan was cancelled, memorize it!
  //
  if (false == pa_bMode)
  {
   m_tView.setBackupScanned(!pa_bWasCancelled);
   
   if (false == pa_bWasCancelled) m_tView.setAmountBackup(m_tPlayerManagerBackupSaveGames, m_tPlayerManagerBackupSaveGames.getSaveGameAmountForList(m_tView.getSelectedBackupPlayers()));
   else 
   {
    m_tView.resetAmountBackup();
    m_tPlayerManagerBackupSaveGames.clear();
   }
  }
  //
  // if an active scan was cancelled, memorize it!
  //
  else
  {
   m_tView.setActiveScanned(!pa_bWasCancelled);
   
   if (false == pa_bWasCancelled) m_tView.setAmountActive(m_tPlayerManagerActiveSaveGames, m_tPlayerManagerActiveSaveGames.getSaveGameAmountForList(m_tView.getSelectedActivePlayers()));
   else
   {
    m_tView.resetAmountActive();
    m_tPlayerManagerActiveSaveGames.clear();
   }
  }
  
  m_tView.setMessage("");
  m_tView.resetProgress();
  m_tView.setUILockMode(false, false);   

  //
  // null-out thread to know which one is active when cancelling anyhting
  //
  if (null != m_tWorkerDirectoryScan) {m_tWorkerDirectoryScan = null;}
  
  //
  // only continue, if the thread was not interrupted or cancelled
  //
  if (false == pa_bWasCancelled)
  {
   //
   // if pa_bMode is true, this is the end of the active directory scan
   //
   if (true == pa_bMode)
   {   
    //
    // init player list, restore eventually existing previous selections
    //
    m_tView.initPlayerList(m_tPlayerManagerActiveSaveGames, true);
   
    if (true == m_bInitializing && true == m_bScanAll) // maybe we are commanded not to scan
    {
     scanBackupSaveGamesDirectory(checkPath(getPropertyString(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_BACKUP_PATH)), false);
    }
   } 
   //
   // if pa_bMode is false, this is the end of the backup directory scan or we're done with initializing (backup scan ist the last step of that)
   //
   else
   {
    if (true == m_bInitializing) m_tView.setMessage(getLocalizedString(SkyrimCharacterHelperConstants.SKH_MESSAGE_READY));
    m_bInitializing = false; 
   
    //
    // init player list, restore eventually existing previous selections. but only, if the scan was completed.
    //
    m_tView.initPlayerList(m_tPlayerManagerBackupSaveGames, false);
   }
  }
  
  //
  // make sure we display an image if we have a restored selection after rescanning. to make sure that this is invoked after the previously
  // triggered changes to the table, we use invokeLater
  //
  if (true == lc_bRescan)
  {
   final boolean lc_bFugly = pa_bMode;
   
   javax.swing.SwingUtilities.invokeLater(new Runnable()
   {
    public void run() 
    { 
     performSaveGameListSelection(lc_bFugly);    
    }
   });      
  }
 }


    
 
 /**
  * Copies the given image to the system clipboard
  * 
  * @param pa_tImage   image
  */
 private void copyImageToClipboard(BufferedImage pa_tImage)
 {
  SkyrimCharacterHelperClipboardImage lc_tClipBoardImage = new SkyrimCharacterHelperClipboardImage(pa_tImage);
  
  if (null != lc_tClipBoardImage)
  {
   Toolkit.getDefaultToolkit().getSystemClipboard().setContents(lc_tClipBoardImage, null); 
   Toolkit.getDefaultToolkit().beep();
  }
 }

 
  
 /**
  * Loads the properties file as defined in constants
  * 
  */
 private boolean loadProperties()
 {
  File             lc_tPropertiesFile = new File(SkyrimCharacterHelperConstants.SKH_FILE_PROPERTIES);
  FileInputStream  lc_tInputStream    = null;       
  boolean          lc_bResult         = false;
  
  if (null != lc_tPropertiesFile && null != m_tProperties)
  {
   if (true == lc_tPropertiesFile.exists() && true == lc_tPropertiesFile.isFile()) 
   {
    try
    {
     if (null != (lc_tInputStream = new FileInputStream(lc_tPropertiesFile)))
     {
      m_tProperties.load(lc_tInputStream);    
      lc_tInputStream.close();
      lc_bResult = true;
     }
    }
    catch (IOException lc_tException) {}    
   }
  }
  return lc_bResult;
 }

 
 
  
 /**
  * This routine saves the settings 
  * 
  * @param pa_sAnnotation Annotation
  */
 private void saveProperties(String pa_sAnnotation)
 {
  File              lc_tPropertiesFile = new File(SkyrimCharacterHelperConstants.SKH_FILE_PROPERTIES);
  FileOutputStream  lc_tOutputStream   = null;       
        
  if (null != lc_tPropertiesFile && null != m_tProperties)
  {
   m_tProperties.put(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_MINIMIZE_MODE    , Integer.toString(m_tView.getMinimizeMode())    );  
   m_tProperties.put(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_WORK_MODE        , Integer.toString(m_tView.getFileMode())        );  
   m_tProperties.put(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_CONFIRMATION_MODE, Integer.toString(m_tView.getConfirmationMode()));  
   m_tProperties.put(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_OVERWRITE_MODE   , Integer.toString(m_tView.getOverwriteMode())   );  
   m_tProperties.put(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_IMAGE_MODE       , Integer.toString(m_tView.getImageMode())       );  
   m_tProperties.put(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_AUTOSAVE_MODE    , Integer.toString(m_tView.getAutoSaveMode())    );  
   m_tProperties.put(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_QUICKSAVE_MODE   , Integer.toString(m_tView.getQuickSaveMode())   );  
   m_tProperties.put(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_INIT_MODE        , Integer.toString(m_tView.getInitMode())        );  
   
   m_tProperties.put(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_HEADER_ACTIVE    , m_tView.getHeaderArrangement(true )            );
   m_tProperties.put(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_HEADER_BACKUP    , m_tView.getHeaderArrangement(false)            );
   m_tProperties.put(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_SORT_ORDER_ACTIVE, m_tView.getSortOrder(true )                    );
   m_tProperties.put(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_SORT_ORDER_BACKUP, m_tView.getSortOrder(false)                    );
   m_tProperties.put(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_PREFIX           , m_tView.getPrefix()                            );
   m_tProperties.put(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_LAUNCH_PARAMETERS, m_tView.getLaunchOptions()                     );
   
   try
   {
    if (false == lc_tPropertiesFile.exists()) 
    {
     lc_tPropertiesFile.createNewFile();
    }
    
    if (true == lc_tPropertiesFile.exists() && true == lc_tPropertiesFile.isFile()) 
    {
     if (null != (lc_tOutputStream = new FileOutputStream(lc_tPropertiesFile)))
     {
      m_tProperties.store(lc_tOutputStream, pa_sAnnotation);  
      lc_tOutputStream.close();
     }
    }
   }
   catch (IOException lc_tException) {}
  }   
 }
 
 
 
 /**
  * Returns the localized string for the given key
  * 
  * @param pa_sKey
  * @return  localized string
  */
 private String getLocalizedString(String pa_sKey) 
 {
  return (null != m_tLocalizedMessages ? m_tLocalizedMessages.getString(pa_sKey) : ""); 
 }
 

 
  /**
  * Returns localized move/copy combobox options
  * 
  */
 private String[] getMonitoringDisplays()
 {
  String[] lc_tResult = new String[2];
  
  if (null != lc_tResult)
  {
   lc_tResult[0] = getLocalizedString(SkyrimCharacterHelperConstants.SKH_JLABEL_TEXT_NOT_MONITORING); 
   lc_tResult[1] = getLocalizedString(SkyrimCharacterHelperConstants.SKH_JLABEL_TEXT_MONITORING    ); 
  }
  return lc_tResult;
 }
 
 

 /**
  * Returns localized yes/no buttons texts
  * 
  */
 private String[] getLocalizedOkCancelOptions()
 {
  String[] lc_tResult = new String[2];
  
  if (null != lc_tResult)
  {
   lc_tResult[0] = getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_OPTION_OK    ); 
   lc_tResult[1] = getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_OPTION_CANCEL); 
  }
  return lc_tResult;
 }
 
 
 /**
  * Returns localized move/copy combobox options
  * 
  */
 private String[] getLocalizedComboBoxOptionsWorkMode()
 {
  String[] lc_tResult = new String[2];
  
  if (null != lc_tResult)
  {
   lc_tResult[0] = getLocalizedString(SkyrimCharacterHelperConstants.SKH_COMBOBOX_OPTION_COPY); 
   lc_tResult[1] = getLocalizedString(SkyrimCharacterHelperConstants.SKH_COMBOBOX_OPTION_MOVE); 
  }
  return lc_tResult;
 }
 
 
 
  /**
  * Returns localized move/copy combobox options
  * 
  */
 private String[] getLocalizedComboBoxOptionsImageMode()
 {
  String[] lc_tResult = new String[2];
  
  if (null != lc_tResult)
  {
   lc_tResult[0] = getLocalizedString(SkyrimCharacterHelperConstants.SKH_COMBOBOX_OPTION_SHOW); 
   lc_tResult[1] = getLocalizedString(SkyrimCharacterHelperConstants.SKH_COMBOBOX_OPTION_HIDE); 
  }
  return lc_tResult;
 }
 
 

 
 
  /**
  * Returns localized move/copy combobox options
  * 
  */
 private String[] getLocalizedComboBoxOptionsInitMode()
 {
  String[] lc_tResult = new String[2];
  
  if (null != lc_tResult)
  {
   lc_tResult[0] = getLocalizedString(SkyrimCharacterHelperConstants.SKH_COMBOBOX_OPTION_SCAN_ALL        ); 
   lc_tResult[1] = getLocalizedString(SkyrimCharacterHelperConstants.SKH_COMBOBOX_OPTION_SCAN_SKYRIM_ONLY); 
  }
  return lc_tResult;
 }
 
 
  /**
  * Returns localized lauch mode combobox options
  * 
  */
 private String[] getLocalizedComboBoxOptionsLaunchMode()
 {
  String[] lc_tResult = new String[2];
  
  if (null != lc_tResult)
  {
   lc_tResult[0] = getLocalizedString(SkyrimCharacterHelperConstants.SKH_COMBOBOX_OPTION_LAUNCH_MINIMIZED); 
   lc_tResult[1] = getLocalizedString(SkyrimCharacterHelperConstants.SKH_COMBOBOX_OPTION_LAUNCH_NORMAL   ); 
  }
  return lc_tResult;
 }
 
 
 
 
  /**
  * Returns localized yes/no combobox options
  * 
  */
 private String[] getLocalizedYesNoOptions()
 {
  String[] lc_tResult = new String[2];
  
  if (null != lc_tResult)
  {
   lc_tResult[0] = getLocalizedString(SkyrimCharacterHelperConstants.SKH_COMBOBOX_OPTION_YES); 
   lc_tResult[1] = getLocalizedString(SkyrimCharacterHelperConstants.SKH_COMBOBOX_OPTION_NO ); 
  }
  return lc_tResult;
 }
 
 

  /**
  * Returns localized table headers fpr the savegame tables
  * 
  */
 private String[] getLocalizedTableHeaders()
 {
  String[] lc_tResult = new String[SkyrimCharacterHelperConstants.SKH_TABLE_COLUMNS];
  
  if (null != lc_tResult)
  {
   lc_tResult[0] = getLocalizedString(SkyrimCharacterHelperConstants.SKH_TABLE_HEADER_LEVEL    ); 
   lc_tResult[1] = getLocalizedString(SkyrimCharacterHelperConstants.SKH_TABLE_HEADER_LOCATION ); 
   lc_tResult[2] = getLocalizedString(SkyrimCharacterHelperConstants.SKH_TABLE_HEADER_FILETIME ); 
   lc_tResult[3] = getLocalizedString(SkyrimCharacterHelperConstants.SKH_TABLE_HEADER_FILENAME ); 
  }
  return lc_tResult;
 }
 
 
 
 /**
  * Returns a property as string
  * 
  * @param pa_sKey   key
  */
 private String getPropertyString(String pa_sKey)
 {
  Object lc_tObject = null;
  
  if (null != m_tProperties)
  {
   if (null != (lc_tObject = m_tProperties.get(pa_sKey)))
   {
    return lc_tObject.toString(); 
   }
  }
  return null;
 }
 
 
 
 /**
  * Exists the application
  * 
  */
 private void handleProgramExit()
 {
  //
  // kill any still running threads
  //
  handleButtonProcessCancel();
  
  //
  // save properties
  //
  if (null != m_tProperties)
  {          
   saveProperties(SkyrimCharacterHelperConstants.SKH_PROPERTIES_ANNOTATION); 
  } 
  
  if (null != m_tPlayerManagerActiveSaveGames) m_tPlayerManagerActiveSaveGames.clear();
  if (null != m_tPlayerManagerBackupSaveGames) m_tPlayerManagerBackupSaveGames.clear();
  if (null != m_tLaunchThread                ) m_tLaunchThread.terminate();
 
  if (null != m_tWorkerDirectoryScan  ) {if (false == m_tWorkerDirectoryScan.isDone()  ) m_tWorkerDirectoryScan.cancel(true)  ;}
  if (null != m_tWorkerDeleteSaveGames) {if (false == m_tWorkerDeleteSaveGames.isDone()) m_tWorkerDeleteSaveGames.cancel(true);}
  if (null != m_tWorkerCopySaveGames  ) {if (false == m_tWorkerCopySaveGames.isDone()  ) m_tWorkerCopySaveGames.cancel(true)  ;}
  if (null != m_tWorkerLoaderSaveGamesActive) {if (false == m_tWorkerLoaderSaveGamesActive.isDone()) m_tWorkerLoaderSaveGamesActive.cancel(true);}
  if (null != m_tWorkerLoaderSaveGamesBackup) {if (false == m_tWorkerLoaderSaveGamesBackup.isDone()) m_tWorkerLoaderSaveGamesBackup.cancel(true);}
          
  m_tPlayerManagerActiveSaveGames = null;
  m_tPlayerManagerBackupSaveGames = null;
  m_tLaunchThread                 = null;

  m_tWorkerDirectoryScan   = null;
  m_tWorkerDeleteSaveGames = null;
  m_tWorkerCopySaveGames   = null;
  m_tWorkerLoaderSaveGamesActive = null;
  m_tWorkerLoaderSaveGamesBackup = null;
  
  //
  // exit. invoke the dispose later, because some drawing stuff may still be on its way. closedown after the
  // windowClosed event shows that it worked.
  //
  javax.swing.SwingUtilities.invokeLater(new Runnable()
  {
   public void run() 
   { 
    try{m_tView.dispose();} catch (Exception lc_tException){} // for saftey, try/catch
   }
  });   
 }
 


 /**
  * Handles a file requester
  * 
  * param pa_sPath                    path to dire or file
  * @param pa_sTitle                  dialog title
  * @param pa_iFileSelectionMode      file seleciton mode
  * @param pa_bCheckRead              true = check if file is readable
  * @param pa_bCheckExecute           true = check if file is execurablke
  * @return  null in case of error, otherwise valid dir or file path
  */
 private String handleFileRequester(String pa_sPath, String pa_sTitle, int pa_iFileSelectionMode, boolean pa_bCheckRead, boolean pa_bCheckExecute)
 {
  JFileChooser     lc_tFileChooser = new JFileChooser();
  File             lc_tFile        = null;
  String           lc_sResult      = null;
  
  if (null != lc_tFileChooser && null != m_tProperties)
  {
   if (null != pa_sPath)
   {
    lc_tFileChooser.setCurrentDirectory(new File(pa_sPath));       
   }                 
   lc_tFileChooser.setFileSelectionMode(pa_iFileSelectionMode);
   lc_tFileChooser.setAcceptAllFileFilterUsed(true);
   lc_tFileChooser.setMultiSelectionEnabled(false); 
   lc_tFileChooser.setDialogTitle(pa_sTitle);   
   lc_tFileChooser.setApproveButtonText(getLocalizedString(SkyrimCharacterHelperConstants.SKH_CHOOSER_OPEN_TEXT));
    
   if (JFileChooser.APPROVE_OPTION == lc_tFileChooser.showOpenDialog(m_tView))
   {
    if (null != (lc_tFile = lc_tFileChooser.getSelectedFile()))
    {
     if ( (true == pa_bCheckRead && true == lc_tFile.canRead()) || (true == pa_bCheckExecute && true == lc_tFile.canExecute()) )
     {
      try 
      {
       lc_sResult = lc_tFile.getCanonicalPath();
      } 
      catch (IOException lc_tException) {}
     }
     else
     {
      JOptionPane.showMessageDialog(m_tView, 
                                    getLocalizedString(true == pa_bCheckRead ? SkyrimCharacterHelperConstants.SKH_DIALOG_TEXT_NOT_READABLE_ERROR 
                                                                             : SkyrimCharacterHelperConstants.SKH_DIALOG_TEXT_NOT_EXECUTABLE_ERROR), 
                                    getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TITLE_ERROR), JOptionPane.ERROR_MESSAGE);                 
       
     }
    }
   }
  }
  return lc_sResult;   
 }

 
 
 /**
  * Brings up a filerequester with the given title and path for choosing a directory
  * 
  * @param pa_sPath    path
  * @param pa_sTitle   title
  * @return            result directory path
  */
 private String handleFileRequesterDirectory(String pa_sPath, String pa_sTitle)
 {
  return handleFileRequester(pa_sPath, pa_sTitle, JFileChooser.DIRECTORIES_ONLY, true, false);
 }

 
 
 /**
  * Brings up a filerequester with the given title and path for choosing a file
  * 
  * @param pa_sPath    path
  * @param pa_sTitle   title
  * @return            result file path
  */
 private String handleFileRequesterFile(String pa_sPath, String pa_sTitle)
 {
  return handleFileRequester(pa_sPath, pa_sTitle, JFileChooser.FILES_ONLY, false, true);
 }

 
 
 
 /**
  * Creates a list of save games from the player selection
  * 
  * @param   pa_bMode  mode, true = active save games, false = backup save games
  * @return  list, may be null
  */
 private List<SkyrimCharacterHelperSaveGame> getSaveGamesListFromPlayers(boolean pa_bMode, boolean pa_bUseSelectedPlayers)
 {
  ArrayList<SkyrimCharacterHelperSaveGame> lc_tResult          = new ArrayList<>();
  List<SkyrimCharacterHelperSaveGame>      lc_tPlayerSaveGames = null;
  List<SkyrimCharacterHelperPlayer>        lc_tSelectedPlayers = (true == pa_bMode ? (true == pa_bUseSelectedPlayers ? m_tView.getSelectedActivePlayers() : m_tView.getActivePlayers())
                                                                                   : (true == pa_bUseSelectedPlayers ? m_tView.getSelectedBackupPlayers() : m_tView.getBackupPlayers()));
 
  int lc_iCount = 0;
  
  if (null != lc_tSelectedPlayers && null != lc_tResult)
  {
   //
   // iterate over selected players
   //
   for (SkyrimCharacterHelperPlayer lc_tPlayer : lc_tSelectedPlayers)
   {
    if (null != lc_tPlayer) 
    {
     if (null != (lc_tPlayerSaveGames = lc_tPlayer.getSaveGamesList()))
     {
      //
      // iterate over current the current list entry's savegames
      //
      for (SkyrimCharacterHelperSaveGame lc_tSaveGame : lc_tPlayerSaveGames)
      {
       if (null != lc_tSaveGame) 
       {
        lc_tResult.add(lc_tSaveGame); lc_iCount++;
       }
      } // for 
     }
    }
   } // for
  }
  return (0 < lc_iCount ? lc_tResult : null);
 }
 
 
 
 
 /**
  * Creates a list of save games from the player selection
  * 
  * @param   pa_bMode  mode, true = active save games, false = backup save games
  * @return  list, may be null
  */
 private List<SkyrimCharacterHelperSaveGame> getSaveGamesExceptSelectedPlayer(boolean pa_bMode)
 {
  ArrayList<SkyrimCharacterHelperSaveGame> lc_tResult           = new ArrayList<>();
  List<SkyrimCharacterHelperSaveGame>      lc_tPlayerSaveGames  = null;
  List<SkyrimCharacterHelperPlayer>        lc_tConcernedPlayers = (true == pa_bMode ? m_tView.getActivePlayers()             : m_tView.getBackupPlayers()            );
  SkyrimCharacterHelperPlayer              lc_tSelectedPlayer   = (true == pa_bMode ? m_tView.getFirstSelectedActivePlayer() : m_tView.getFirstSelectedBackupPlayer());
  int lc_iCount = 0;
  
  if (null != lc_tConcernedPlayers && null != lc_tSelectedPlayer && null != lc_tResult)
  {
   //
   // iterate over selected players
   //
   for (SkyrimCharacterHelperPlayer lc_tPlayer : lc_tConcernedPlayers)
   {
    if (null != lc_tPlayer) 
    {
     //
     // do not include the player to be launched !
      //
     if (false == lc_tPlayer.equals(lc_tSelectedPlayer))
     {
      if (null != (lc_tPlayerSaveGames = lc_tPlayer.getSaveGamesList()))
      {
       //
       // iterate over current the current list entry's savegames
       //
       for (SkyrimCharacterHelperSaveGame lc_tSaveGame : lc_tPlayerSaveGames)
       {
        if (null != lc_tSaveGame) 
        {
         lc_tResult.add(lc_tSaveGame); lc_iCount++;
        }
       } // for 
      }
     }
    }
   } // for
  }
  return (0 < lc_iCount ? lc_tResult : null);
 }
 
 
 
 /**
  * Creates a list of save games from the player selection
  * 
  * @param   pa_bMode  mode, true = active save games, false = backup save games
  * @return  list, may be null
  */
 private List<SkyrimCharacterHelperSaveGame> getSaveGamesExceptSelectedSavegame(boolean pa_bMode)
 {
  ArrayList<SkyrimCharacterHelperSaveGame> lc_tResult           = new ArrayList<>();
  List<SkyrimCharacterHelperSaveGame>      lc_tPlayerSaveGames  = null;
  List<SkyrimCharacterHelperPlayer>        lc_tConcernedPlayers = (true == pa_bMode ? m_tView.getActivePlayers()               : m_tView.getBackupPlayers()              );
  SkyrimCharacterHelperPlayer              lc_tSelectedPlayer   = (true == pa_bMode ? m_tView.getFirstSelectedActivePlayer()   : m_tView.getFirstSelectedBackupPlayer()  );
  SkyrimCharacterHelperSaveGame            lc_tSelectedSaveGame = (true == pa_bMode ? m_tView.getFirstSelectedActiveSaveGame() : m_tView.getFirstSelectedBackupSaveGame());
  
  int lc_iCount = 0;
  
  if (null != lc_tConcernedPlayers && null != lc_tSelectedPlayer && null != lc_tResult)
  {
   //
   // iterate over selected players
   //
   for (SkyrimCharacterHelperPlayer lc_tPlayer : lc_tConcernedPlayers)
   {
    if (null != lc_tPlayer) 
    {
     //
     // do not include the player to be launched !
      //
     if (false == lc_tPlayer.equals(lc_tSelectedPlayer))
     {
      if (null != (lc_tPlayerSaveGames = lc_tPlayer.getSaveGamesList()))
      {
       //
       // iterate over current the current list entry's savegames
       //
       for (SkyrimCharacterHelperSaveGame lc_tSaveGame : lc_tPlayerSaveGames)
       {
        if (null != lc_tSaveGame) 
        {
         lc_tResult.add(lc_tSaveGame); lc_iCount++;
        }
       } // for 
      }
     }
    }
   } // for
   
   //
   // now, get any savegame of the selected player except the selected savegame
   //
   if (null != lc_tSelectedSaveGame)
   {
    if (null != (lc_tPlayerSaveGames = lc_tSelectedPlayer.getSaveGamesList()))
    {
     for (SkyrimCharacterHelperSaveGame lc_tSaveGame : lc_tPlayerSaveGames)
     {
      if (null != lc_tSaveGame) 
      {
       if (!lc_tSelectedSaveGame.equals(lc_tSaveGame)) 
       {
        lc_tResult.add(lc_tSaveGame); lc_iCount++; 
       }
      }   
     }
    }
   }   
  }
  return (0 < lc_iCount ? lc_tResult : null);
 }
 
 
 
 
 /**
  * Pre-computes how many files would be overwritten
  * 
  * @param pa_bMode           true = Skyrim > Backup, false = Backup > Skyrim
  * @return 
  */
 private int getAmountOfFileCollisions(List<SkyrimCharacterHelperSaveGame> pa_tSaveGames, boolean pa_bMode)
 {
  int lc_iCollisions = 0;
  SkyrimCharacterHelperPlayerManager lc_tDestinationPlayerManager = (true == pa_bMode ? m_tPlayerManagerBackupSaveGames : m_tPlayerManagerActiveSaveGames);
  
  if (null != pa_tSaveGames && null != lc_tDestinationPlayerManager)
  {
   for (SkyrimCharacterHelperSaveGame lc_tSaveGame : pa_tSaveGames)
   {
    if (null != lc_tSaveGame) 
    {
     if (true == lc_tDestinationPlayerManager.hasSaveGame(lc_tSaveGame)) 
     {
      lc_iCollisions++; 
     }
    }
   }
  }
  return lc_iCollisions;
 }
  
  
 /**
  * Replaces a substring with a string
  * 
  * @param pa_sSource        source string
  * @param pa_sReplace       string to replace
  * @param pa_sReplaceWith   string to replace with
  * @return                  new string or ""
  */ 
 private String replaceString(String pa_sSource, String pa_sReplace, String pa_sReplaceWith)
 {
  StringBuffer lc_sResult = null;
  
  try
  {
   if (null != pa_sSource && null != pa_sReplace && null != pa_sReplaceWith)
   {
    if (null != (lc_sResult = new StringBuffer()))
    {
     if (-1 != pa_sSource.indexOf(pa_sReplace))
     {
      lc_sResult.append(pa_sSource.substring(0, pa_sSource.indexOf(pa_sReplace)));
      lc_sResult.append(pa_sReplaceWith);
      lc_sResult.append(pa_sSource.substring(pa_sSource.indexOf(pa_sReplace) + pa_sReplace.length()));
     }
     else
     {
      lc_sResult.append(pa_sSource); 
     }
    }
   }
  }
  catch (Exception lc_tException) {}
  return (null != lc_sResult ? lc_sResult.toString() : "");   
 }
 
 
 /**
  * Replaces a substring with a number
  * 
  * @param pa_sSource    source string
  * @param pa_iNumber    number to insert
  * @return              new string
  */
 private String insertNumberString(String pa_sSource, int pa_iNumber)
 {
  return replaceString(pa_sSource, SkyrimCharacterHelperConstants.SKH_MESSAGE_NUM_REPLACER, Integer.toString(pa_iNumber));
 }
 
 
 
 /**
  * Replaces a substring with a string
  * 
  * @param pa_sSource    source string
  * @param pa_sToInsert  string to insert
  * @return              new string
  */
 private String insertString(String pa_sSource, String pa_sToInsert)
 {
  return replaceString(pa_sSource, SkyrimCharacterHelperConstants.SKH_MESSAGE_STR_REPLACER, pa_sToInsert);
 }
 
 
 
  /**
  * Handles a list selection in the save games lists. In case of a multiselection,
  * nothing is displayed
  * 
  */
 private synchronized void performSaveGameListSelection(boolean pa_bMode)
 {    
  java.awt.image.BufferedImage  lc_tImage             = null;
  String                        lc_sFileName          = null;

  boolean                       lc_bHasMultiSelection = (true == pa_bMode ? m_tView.hasListMultiSelectionActiveSaveGames() : m_tView.hasListMultiSelectionBackupSaveGames());
  SkyrimCharacterHelperSaveGame lc_tSelectedSaveGame  = (true == pa_bMode ? m_tView.getFirstSelectedActiveSaveGame()       : m_tView.getFirstSelectedBackupSaveGame()      );

  if (m_tView.getImageMode() == SkyrimCharacterHelperConstants.SKH_IMAGEMODE_SHOW)
  {
   if (true == pa_bMode) m_tView.setActiveSaveGameScreenshot(null);
   else                  m_tView.setBackupSaveGameScreenshot(null);

   if (null != lc_tSelectedSaveGame) 
   {
    if (false == lc_bHasMultiSelection)
    {
     if (null != lc_tSelectedSaveGame.getScreenshot()) 
     {
      if (null != lc_tSelectedSaveGame.getScreenshot().getImage()) 
      { 
       lc_tImage = lc_tSelectedSaveGame.getScreenshot().getImage();
      }
     }
    }
   }
   if (null == lc_tImage)
   {
    //
    // check if we need to load the screenshot data
    //
    if (false == lc_bHasMultiSelection && null != lc_tSelectedSaveGame)
    {
     //
     // start savegame thread
     // 
     if (null != (lc_sFileName = lc_tSelectedSaveGame.getFilePath())) 
     {
      if (true == pa_bMode)
      {
       if (null == m_tWorkerLoaderSaveGamesActive)
       {
        if (null != (m_tWorkerLoaderSaveGamesActive = new SwingWorkerSaveGameLoader(m_tWorkerLoaderSaveGamesBackup))) 
        {
         m_tView.setUILockMode(true, true);   
         m_tWorkerLoaderSaveGamesActive.init(lc_sFileName, lc_tSelectedSaveGame, pa_bMode, insertString(getLocalizedString(SkyrimCharacterHelperConstants.SKH_MESSAGE_READING_SAVEGAME), lc_sFileName)); 
         m_tWorkerLoaderSaveGamesActive.execute();
        }
       }
      }
      else
      {
       if (null == m_tWorkerLoaderSaveGamesBackup)
       {
        if (null != (m_tWorkerLoaderSaveGamesBackup = new SwingWorkerSaveGameLoader(m_tWorkerLoaderSaveGamesActive))) 
        {
         m_tView.setUILockMode(true, true);   
         m_tWorkerLoaderSaveGamesBackup.init(lc_sFileName, lc_tSelectedSaveGame, pa_bMode, insertString(getLocalizedString(SkyrimCharacterHelperConstants.SKH_MESSAGE_READING_SAVEGAME), lc_sFileName)); 
         m_tWorkerLoaderSaveGamesBackup.execute();
        }
       }
      }
     }
    }
   }
   if (null != lc_tImage)
   {
    if (true == pa_bMode) m_tView.setActiveSaveGameScreenshot(lc_tImage);
    else                  m_tView.setBackupSaveGameScreenshot(lc_tImage);
   }
  }
  m_tView.updateUIElements();
 } 
 
 
 
 /**
  * Performs a list selection in a character list
  * 
  * @param pa_bMode           true = active player list, false = backup player list
  * @param pa_bCheckButtons   true = check buttons
  */
 private void performCharacterListSelection(boolean pa_bMode)
 {
  SkyrimCharacterHelperPlayer   lc_tSelectedCharacter = null ;
  boolean                       lc_bHasMultiSelection = (true == pa_bMode ? m_tView.hasListMultiSelectionActivePlayers() : m_tView.hasListMultiSelectionBackupPlayers());
  
  if (true == pa_bMode) 
  {
   m_tView.setActiveSaveGameScreenshot(null);
   m_tView.setAmountActive(m_tPlayerManagerActiveSaveGames, -1); 
   m_tView.setActivePlayerRace(null);
   m_tView.clearActiveSaveGameTable();
  } 
  else
  {
   m_tView.setBackupSaveGameScreenshot(null);
   m_tView.setAmountBackup(m_tPlayerManagerBackupSaveGames, -1); 
   m_tView.setBackupPlayerRace(null);
   m_tView.clearBackupSaveGameTable();
  }
  
  if (false == lc_bHasMultiSelection)
  {   
   if (null != (lc_tSelectedCharacter =  (true == pa_bMode ? m_tView.getFirstSelectedActivePlayer() : m_tView.getFirstSelectedBackupPlayer())))
   {
    if (true == pa_bMode)
    {
     m_tView.initSaveGameList(lc_tSelectedCharacter.getSaveGames(), true , false);
     m_tView.setAmountActive(m_tPlayerManagerActiveSaveGames, lc_tSelectedCharacter.getSaveGameAmount());
     m_tView.setActivePlayerRace(lc_tSelectedCharacter.getRace());
    }
    else
    {
     m_tView.initSaveGameList(lc_tSelectedCharacter.getSaveGames(), false, false);
     m_tView.setAmountBackup(m_tPlayerManagerBackupSaveGames, lc_tSelectedCharacter.getSaveGameAmount());
     m_tView.setBackupPlayerRace(lc_tSelectedCharacter.getRace());
    }
   }
  }
  else
  {
   if (true == pa_bMode) m_tView.setAmountActive(m_tPlayerManagerActiveSaveGames, m_tPlayerManagerActiveSaveGames.getSaveGameAmountForList(m_tView.getSelectedActivePlayers()));
   else                  m_tView.setAmountBackup(m_tPlayerManagerBackupSaveGames, m_tPlayerManagerBackupSaveGames.getSaveGameAmountForList(m_tView.getSelectedBackupPlayers()));
  }
 }

    
 
 /**
  * Peforms selecting a path
  * 
  */
 private void performSelectPath(boolean pa_bMode)
 {
  String lc_sPath      = getPropertyString(true == pa_bMode ? SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_ACTIVE_PATH : SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_BACKUP_PATH);
  String lc_sOldValue  = (true == pa_bMode ? m_tView.getActivePath() : m_tView.getBackupPath());
  String lc_sDirectory = null;
  
  if (null != m_tProperties)
  {
   if (null != (lc_sDirectory = handleFileRequesterDirectory(lc_sPath, getLocalizedString(true == pa_bMode ? SkyrimCharacterHelperConstants.SKH_DIALOG_TITLE_FILEREQUESTER_ACTIVE_DIR 
                                                                                                  : SkyrimCharacterHelperConstants.SKH_DIALOG_TITLE_FILEREQUESTER_BACKUP_DIR))))
   {
    if (true == pa_bMode) m_tView.setActivePath(lc_sDirectory);
    else                  m_tView.setBackupPath(lc_sDirectory);

    //
    // check for equal directories
    //
    if (false == lc_sDirectory.equals(lc_sOldValue))
    {
     if (true == m_tView.hasEqualDirectories())
     {
      if (true == pa_bMode) m_tView.setActivePath(lc_sOldValue);
      else                  m_tView.setBackupPath(lc_sOldValue);
      m_tView.setMessage(getLocalizedString(SkyrimCharacterHelperConstants.SKH_MESSAGE_EQUAL_DIRS));
     }
     else
     {
      //
      // remove selections
      //
      m_tView.removeSaveGameSelections(pa_bMode);
      
      //
      // scan directory and fill list
      //  
      if (true == pa_bMode) m_tView.setActivePath(lc_sDirectory);
      else                  m_tView.setBackupPath(lc_sDirectory);
     
      m_tProperties.put(true == pa_bMode ? SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_ACTIVE_PATH : SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_BACKUP_PATH, lc_sDirectory);
  
      if (true == pa_bMode) scanActiveSaveGamesDirectory(lc_sDirectory, getLocalizedString(SkyrimCharacterHelperConstants.SKH_MESSAGE_SCANNING_ACTIVE_SAVEGAMES), true, false);
      else                  scanBackupSaveGamesDirectory(lc_sDirectory, false);
     }
    }
   }
  }
  m_tView.updateUIElements();
 }
 
 
 
 /**
  * Deletes a player
  * 
  */
 private void performDeletePlayers(boolean pa_bMode)
 {
  boolean lc_bContinue   = true;
  
  SkyrimCharacterHelperPlayerManager lc_tManager  = (true == pa_bMode ? m_tPlayerManagerActiveSaveGames : m_tPlayerManagerBackupSaveGames);
  List<SkyrimCharacterHelperPlayer>  lc_tPlayers  = (true == pa_bMode ? m_tView.getSelectedActivePlayers() : m_tView.getSelectedBackupPlayers());
  
  if (null != lc_tManager && null != lc_tPlayers)
  {
   if (SkyrimCharacterHelperConstants.SKH_CONFIRM_DELETE == m_tView.getConfirmationMode())
   {
    lc_bContinue = (0 == JOptionPane.showOptionDialog(m_tView,  
                                                      (1 == lc_tPlayers.size() ? getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TEXT_DELETE_PLAYER  )
                                                                               : getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TEXT_DELETE_PLAYERS)),  
                                                      getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TITLE_WARNING      ), 
                                                      JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE, null, 
                                                      getLocalizedOkCancelOptions()                                   , 
                                                      getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_OPTION_OK)));
   }
   if (true == lc_bContinue)
   {
    if (null != (m_tWorkerDeleteSaveGames = new SwingWorkerSaveGameDelete()))
    {
     m_tView.setMessage(getLocalizedString(SkyrimCharacterHelperConstants.SKH_MESSAGE_DELETE_PLAYERS));    
     m_tView.setUILockMode(true, false);     
     
     m_tWorkerDeleteSaveGames.init(SkyrimCharacterHelperConstants.SKH_WORKER_THREAD_MODE_DELETE, getSaveGamesListFromPlayers(pa_bMode, true), pa_bMode);                                 
     m_tWorkerDeleteSaveGames.execute(); 
    }    
   }
  }
 }
 
 
 
 /**
  * Deletes save games
  * 
  */
 private void performDeleteSaveGames(boolean pa_bMode)
 {
  boolean lc_bContinue = true;  
  SkyrimCharacterHelperPlayerManager  lc_tManager   = (true == pa_bMode ? m_tPlayerManagerActiveSaveGames : m_tPlayerManagerBackupSaveGames);
  List<SkyrimCharacterHelperSaveGame> lc_tSaveGames = (true == pa_bMode ? m_tView.getSelectedActiveSaveGames() : m_tView.getSelectedBackupSaveGames());

  if (null != lc_tManager && null != lc_tSaveGames)
  {
   if (SkyrimCharacterHelperConstants.SKH_CONFIRM_DELETE == m_tView.getConfirmationMode())
   {
    lc_bContinue = (0 == JOptionPane.showOptionDialog(m_tView,  
                                                      (1 == lc_tSaveGames.size() ? getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TEXT_DELETE_SAVEGAME )
                                                                                 : getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TEXT_DELETE_SAVEGAMES)),  
                                                      getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TITLE_WARNING     ), 
                                                      JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE, null, 
                                                      getLocalizedOkCancelOptions()                                   , 
                                                      getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_OPTION_OK)));
   }
   if (true == lc_bContinue)
   {
    if (null != lc_tManager)
    { 
     if (null != (m_tWorkerDeleteSaveGames = new SwingWorkerSaveGameDelete()))
     {
      m_tView.setMessage(getLocalizedString(SkyrimCharacterHelperConstants.SKH_MESSAGE_DELETE_SAVEGAMES));          
      m_tView.setUILockMode(true, false);  

      m_tWorkerDeleteSaveGames.init(SkyrimCharacterHelperConstants.SKH_WORKER_THREAD_MODE_DELETE, lc_tSaveGames, pa_bMode);
      m_tWorkerDeleteSaveGames.execute(); 
     }
    }
   }
  }   
 }

 
 
 
 /**
  * Clears all players
  * 
  */
 private void performClearPlayers(boolean pa_bMode)
 {
  boolean lc_bContinue = true;
  
  if (SkyrimCharacterHelperConstants.SKH_CONFIRM_DELETE == m_tView.getConfirmationMode())
  {
   lc_bContinue = (0 == JOptionPane.showOptionDialog(m_tView,  
                                                     getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TEXT_CLEAR_PLAYERS), 
                                                     getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TITLE_WARNING     ), 
                                                     JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE, null, 
                                                     getLocalizedOkCancelOptions()                                   , 
                                                     getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_OPTION_OK)));
  }
  if (true == lc_bContinue)
  {
   if (null != (m_tWorkerDeleteSaveGames = new SwingWorkerSaveGameDelete()))
   {
    m_tView.setMessage(getLocalizedString(SkyrimCharacterHelperConstants.SKH_MESSAGE_DELETE_ALL_PLAYERS));          
    m_tView.setUILockMode(true, false);

    m_tWorkerDeleteSaveGames.init(SkyrimCharacterHelperConstants.SKH_WORKER_THREAD_MODE_DELETE, getSaveGamesListFromPlayers(pa_bMode, false), pa_bMode);                                 
    m_tWorkerDeleteSaveGames.execute();      
   }
  }      
 }

 
 
 /**
  * Performs a savegame copy
  * 
  * @param pa_bMode  true = Skyrim > Backup, false = Backup > Skyrim
  * 
  */
 private void performCopySaveGames(boolean pa_bMode) 
 {
  String  lc_sSelectedDestinationPlayerName = null;
  boolean lc_bContinue   = true;
  int     lc_iCount      = 0   ;
   
  SkyrimCharacterHelperPlayerManager lc_tManager                      = (true == pa_bMode ? m_tPlayerManagerActiveSaveGames   : m_tPlayerManagerBackupSaveGames  );
  SkyrimCharacterHelperPlayer        lc_tSelectedDestinationPlayer    = null;  
  
  List<SkyrimCharacterHelperSaveGame> lc_tSaveGames = null;
  
  m_tView.storeSaveGameSelection(pa_bMode);
  m_tView.storeSaveGameSelection(!pa_bMode);
  
  if (null != lc_tManager)
  {
   if (null != (lc_tSaveGames = (true == pa_bMode ? m_tView.getSelectedActiveSaveGames() : m_tView.getSelectedBackupSaveGames())))
   {
    if (SkyrimCharacterHelperConstants.SKH_CONFIRM_DELETE == m_tView.getConfirmationMode() && SkyrimCharacterHelperConstants.SKH_WORKMODE_MOVE == m_tView.getFileMode())
    {
     //
     // delete warning
     //
     lc_bContinue = (0 == JOptionPane.showOptionDialog(m_tView,  
                                                       getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TEXT_DELETE_AFTER_COPY), 
                                                       getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TITLE_WARNING     ), 
                                                       JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE, null, 
                                                       getLocalizedOkCancelOptions()                                   , 
                                                       getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_OPTION_OK)));
    }
    if (true == lc_bContinue)
    {
     if (SkyrimCharacterHelperConstants.SKH_CONFIRM_YES == m_tView.getOverwriteMode() && (0 < (lc_iCount = getAmountOfFileCollisions(lc_tSaveGames, pa_bMode))))
     {
      //
      // overwrite warning
      // 
      lc_bContinue = (0 == JOptionPane.showOptionDialog(m_tView,  
                                                   insertNumberString(getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TEXT_OVERWRITE_SAVEGAME), lc_iCount), 
                                                   getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TITLE_WARNING          ), 
                                                   JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE, null, 
                                                   getLocalizedOkCancelOptions()                                   , 
                                                   getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_OPTION_OK))); 

     }
     if (true == lc_bContinue)
     {
      //
      // backup the save games
      //
      if (null != (m_tWorkerCopySaveGames = new SwingWorkerSaveGameCopy()))
      {
       lc_tSelectedDestinationPlayer     = (true == pa_bMode ? m_tView.getFirstSelectedBackupPlayer() : m_tView.getFirstSelectedActivePlayer());
       lc_sSelectedDestinationPlayerName = (null != lc_tSelectedDestinationPlayer ? lc_tSelectedDestinationPlayer.getName() : null);

       m_tView.setMessage(getLocalizedString(true == pa_bMode ? SkyrimCharacterHelperConstants.SKH_MESSAGE_BACKING_UP : SkyrimCharacterHelperConstants.SKH_MESSAGE_RESTORING));          
       m_tView.setUILockMode(true, false);   
 
       m_tWorkerCopySaveGames.init((true == pa_bMode ? SkyrimCharacterHelperConstants.SKH_WORKER_THREAD_MODE_BACKUP : SkyrimCharacterHelperConstants.SKH_WORKER_THREAD_MODE_RESTORE),
                                   m_tView.getFileMode(),
                                   lc_tSaveGames,
                                   pa_bMode,
                                   (true == pa_bMode ? m_tView.hasListMultiSelectionBackupPlayers() : m_tView.hasListMultiSelectionActivePlayers()),
                                   lc_sSelectedDestinationPlayerName);
       
       m_tWorkerCopySaveGames.execute(); 
      }    
     }
    }
   }
  }
 }

 
 
 
 /**
  * Performs copying 1 to n players
  * 
  * @param pa_bMode  true = Skyrim > Backup, false = Backup > Skyrim
  * 
  */
 private void performCopyPlayers(boolean pa_bMode) 
 {
  String  lc_sSelectedDestinationPlayerName  = null;
  boolean lc_bContinue    = true;
  int     lc_iCount       = 0   ;
  
  SkyrimCharacterHelperPlayer         lc_tSelectedDestinationPlayer = null;  
  List<SkyrimCharacterHelperSaveGame> lc_tSaveGames                 = null;
  
  m_tView.storeSaveGameSelection(pa_bMode);
  m_tView.storeSaveGameSelection(!pa_bMode);
  
  if (SkyrimCharacterHelperConstants.SKH_CONFIRM_DELETE == m_tView.getConfirmationMode() && SkyrimCharacterHelperConstants.SKH_WORKMODE_MOVE == m_tView.getFileMode())
  {
   //
   // delete warning
   //
   lc_bContinue = (0 == JOptionPane.showOptionDialog(m_tView,  
                                                     getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TEXT_DELETE_AFTER_COPY), 
                                                     getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TITLE_WARNING         ), 
                                                     JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE, null, 
                                                     getLocalizedOkCancelOptions()                                   , 
                                                     getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_OPTION_OK)));
  }
  if (true == lc_bContinue)
  {
   if (null != (lc_tSaveGames = getSaveGamesListFromPlayers(pa_bMode, true)))
   {
    if (SkyrimCharacterHelperConstants.SKH_CONFIRM_YES == m_tView.getOverwriteMode() && (0 < (lc_iCount = getAmountOfFileCollisions(lc_tSaveGames, pa_bMode))))
    {
     //
     // overwrite warning
     // 
     lc_bContinue = (0 == JOptionPane.showOptionDialog(m_tView,  
                                                  insertNumberString(getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TEXT_OVERWRITE_SAVEGAME), lc_iCount), 
                                                  getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TITLE_WARNING), 
                                                  JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE, null, 
                                                  getLocalizedOkCancelOptions()                                   , 
                                                  getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_OPTION_OK)));
    }
    if (true == lc_bContinue)
    {
     //
     // backup the save games
     //
     if (null != (m_tWorkerCopySaveGames = new SwingWorkerSaveGameCopy()))
     {
      lc_tSelectedDestinationPlayer     = (true == pa_bMode ? m_tView.getFirstSelectedBackupPlayer() : m_tView.getFirstSelectedActivePlayer());
      lc_sSelectedDestinationPlayerName = (null != lc_tSelectedDestinationPlayer ? lc_tSelectedDestinationPlayer.getName() : null);
            
      m_tView.setMessage(getLocalizedString(true == pa_bMode ? SkyrimCharacterHelperConstants.SKH_MESSAGE_BACKING_UP : SkyrimCharacterHelperConstants.SKH_MESSAGE_RESTORING));          
      m_tView.setUILockMode(true, false);   
      
      m_tWorkerCopySaveGames.init((true == pa_bMode ? SkyrimCharacterHelperConstants.SKH_WORKER_THREAD_MODE_BACKUP : SkyrimCharacterHelperConstants.SKH_WORKER_THREAD_MODE_RESTORE),
                                  m_tView.getFileMode(),
                                  lc_tSaveGames, 
                                  pa_bMode     , 
                                  (true == pa_bMode ? m_tView.hasListMultiSelectionBackupPlayers() : m_tView.hasListMultiSelectionActivePlayers()),
                                  lc_sSelectedDestinationPlayerName);
      
      m_tWorkerCopySaveGames.execute(); 
     }     
    }
   }
  }  
 } 
 
 
 
  
  /**
   * SkyrimCharacterHelperThreadNotifier.notifySkyrimTerminated
   * 
   * Receives the notification that Skyrim has terminated. As a result, we now rescan
   * the Skyrim savegame directory.
   * 
   * @param pa_iValue   new value
   */
 @Override
 public void notifySkyrimTerminated(int pa_iExitValue) 
 {
  //
  // reset launcher thread and reflag UI
  //
  m_tLaunchThread = null; 

  m_tView.setOnSkyrim(false);
  m_tView.setLaunched(false);
  m_tView.setUILockMode(false, false);
  
  //
  // un-minimize, if needed
  //
  if (Frame.ICONIFIED == m_tView.getState())
  {
   m_tView.setState(m_tView.getExtendedState() & ~Frame.ICONIFIED);
  }
  m_tView.toFront();
  m_tView.setMessage(insertNumberString(getLocalizedString(SkyrimCharacterHelperConstants.SKH_MESSAGE_LAUNCH_DONE), m_iSaveGameChanges));  
 }

 
 
 /**
  * Notification that a Skyrim savegames was deleted
  * 
  * @param pa_sFileName  file name without path
  */
 public synchronized void notifySkyrimSavegameDeleted(String pa_sFileName)
 {
  SkyrimCharacterHelperSaveGame lc_tSaveGame = null;
  SkyrimCharacterHelperPlayer       lc_tSourceFirstSelection = m_tView.getFirstSelectedActivePlayer();

  String  lc_sCharacter                = null;
  boolean lc_bSourceHasSingleSelection = (null != m_tView.getSelectedActivePlayers() && false == m_tView.hasListMultiSelectionActivePlayers());
  boolean lc_bSourceHasPlayer          = false;
  boolean lc_bEmpty                    = false;
  boolean lc_bWasDisplayed             = false;
  
  if (null != m_tPlayerManagerActiveSaveGames && null != pa_sFileName)
  {   
   //
   // get character name for deleted savegame from charactermanager
   //
   if (null != (lc_sCharacter = m_tPlayerManagerActiveSaveGames.hasSaveGame(pa_sFileName)))
   {
    //
    // store current character selection and determine whether that character is new
    //
    m_tView.storePlayerSelection(true);
    lc_bSourceHasPlayer = m_tPlayerManagerActiveSaveGames.hasPlayer(lc_sCharacter);
    
    //
    // remove savegame from manager and get amount of remaining savegames for that character
    //
    m_iSaveGameChanges++;
    lc_tSaveGame = m_tPlayerManagerActiveSaveGames.removeSaveGame(pa_sFileName); 
    lc_bEmpty    = (0 == m_tPlayerManagerActiveSaveGames.getSaveGameAmount(lc_sCharacter));

    //
    // check if that character is currently displayed. if so, we need to remove a savegame entry from the table
    //
    if (null != lc_tSourceFirstSelection)
    {
     if (true == lc_bSourceHasSingleSelection && true == lc_bSourceHasPlayer && lc_sCharacter.equals(lc_tSourceFirstSelection.getName()))
     {
      m_tView.removeActiveSaveGame(lc_tSaveGame, (false == lc_bEmpty));
      lc_bWasDisplayed = true;
     }
    }
   
    //
    // check, if we removed a character's last savegame and therefore, need to remove the character from the character manager
    //
    if (true == lc_bEmpty)
    {
     m_tPlayerManagerActiveSaveGames.removePlayer(lc_sCharacter);
     
     //
     // if that character was displayed, we also need to remove the character from the list
     //
     if (true == lc_bWasDisplayed)
     {
      m_tView.setActiveSaveGameScreenshot(null);
      m_tView.setActivePlayerRace(null);
      m_tView.removeActivePlayer(lc_tSourceFirstSelection, true, false);       
     }
    }

    //
    // in either case, the amount of savegames has changed, recompute and redisplay
    //
    m_tView.setAmountActive(m_tPlayerManagerActiveSaveGames, m_tPlayerManagerActiveSaveGames.getSaveGameAmountForList(m_tView.getSelectedActivePlayers()));
   }
  }
 }

 
  
 



 /**
  * Notification that a Skyrim savegames was modified or created
  * 
  * @param pa_sFileName  file name without path
  */
 public synchronized void notifySkyrimSavegameModified(String pa_sFileName)
 {
  SkyrimCharacterHelperSaveGame lc_tSaveGame = null;
  String lc_sFullPath  = null;  
  String lc_sCharacter = null;
  
  List<SkyrimCharacterHelperPlayer> lc_tSourcePlayerOldSelection = m_tView.getSelectedActivePlayers();
  SkyrimCharacterHelperPlayer       lc_tSourceFirstSelection     = m_tView.getFirstSelectedActivePlayer();
  boolean                           lc_bSourceHasSingleSelection = (null != m_tView.getSelectedActivePlayers() && false == m_tView.hasListMultiSelectionActivePlayers());
  boolean                           lc_bSourceHasPlayer          = false;
  
  if (null != m_tPlayerManagerActiveSaveGames && null != pa_sFileName)
  {
   //
   // in either case, we do the same
   //
   if (null != (lc_tSaveGame = new SkyrimCharacterHelperSaveGame()))
   {
    if (null != (lc_sFullPath = getPropertyString(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_ACTIVE_PATH)))
    {            
     //
     // read savegame
     //
     if (!lc_sFullPath.endsWith(m_sSeparator)) lc_sFullPath += m_sSeparator;
     lc_sFullPath += pa_sFileName;

     if (true == lc_tSaveGame.read(lc_sFullPath, null, true)) // include screenshot data
     {
      //
      // get character name and check, whether we already have such a player in the character manager
      //
      lc_sCharacter       = lc_tSaveGame.getPlayerName();  
      lc_bSourceHasPlayer = m_tPlayerManagerActiveSaveGames.hasPlayer(lc_sCharacter);
      
      //
      // add savegame to character manager. already existing savegames with equal filenames are overwritten.
      //
      m_tPlayerManagerActiveSaveGames.add(lc_tSaveGame);
      m_iSaveGameChanges++;

      //
      // add savegame to display, if that user exists and is currently selected. already existing savegames with equal filenames are overwritten in the display table.
      //
      if (null != lc_tSourceFirstSelection)
      {
       if (true == lc_bSourceHasSingleSelection && true == lc_bSourceHasPlayer && lc_sCharacter.equals(lc_tSourceFirstSelection.getName()))
       {
        m_tView.addActiveSaveGame(lc_tSaveGame, true);
       }
      }
        
      //
      // check, if we need to add a new player to the list. if so, we also need to restore the selection
      //
      if (false == lc_bSourceHasPlayer)
      {
       m_tView.addActivePlayer(m_tPlayerManagerActiveSaveGames.getPlayer(lc_sCharacter));
       m_tView.reselectPlayers(lc_tSourcePlayerOldSelection, true); 
      }
       
      //
      // in either case, the amount of savegames has changed, recompute and redisplay
      //
      m_tView.setAmountActive(m_tPlayerManagerActiveSaveGames, m_tPlayerManagerActiveSaveGames.getSaveGameAmountForList(m_tView.getSelectedActivePlayers()));
     }
    }
   }
  }
 }


 
 /**
  * Checks for a folder My Games\Skyrim\Saves in the current user's home directory
  * 
  */
 private String tryGuessSkyrimSaveGameFolder()
 {
  String lc_sHomeDirectory = System.getenv     (SkyrimCharacterHelperConstants.SKH_USER_PROFILE);
  String lc_sOsName        = System.getProperty(SkyrimCharacterHelperConstants.SKH_OS_NAME     );
  String lc_sOsVersion     = System.getProperty(SkyrimCharacterHelperConstants.SKH_OS_VERSION  );
  String lc_sPath          = null;
   
  boolean lc_bWinXp = false;
  boolean lc_bWin2K = false;
  boolean lc_bWin7  = false;
  
  if (null != lc_sOsName && null != lc_sOsVersion && null != m_tProperties)
  {
   if (lc_sOsName.toUpperCase().startsWith(SkyrimCharacterHelperConstants.SKH_OS_WINDOWS.toUpperCase()))
   {
    lc_bWinXp = lc_sOsVersion.startsWith(SkyrimCharacterHelperConstants.SKH_OS_WINDOWS_XP);
    lc_bWin2K = lc_sOsVersion.startsWith(SkyrimCharacterHelperConstants.SKH_OS_WINDOWS_2K);
    lc_bWin7  = lc_sOsVersion.startsWith(SkyrimCharacterHelperConstants.SKH_OS_WINDOWS_7 );

    //
    // ofc, vista does everything in a different manner then XP and 7... sigh...
    //
    if (true == lc_bWinXp) lc_sHomeDirectory += SkyrimCharacterHelperConstants.SKH_FILE_DOCUMENT_FOLDER_2K_XP + SkyrimCharacterHelperConstants.SKH_FILE_DOCUMENT_SAVEGAME_FOLDER_LATER  ; // funny, but true: different to vista
    if (true == lc_bWin2K) lc_sHomeDirectory += SkyrimCharacterHelperConstants.SKH_FILE_DOCUMENT_FOLDER_2K_XP + SkyrimCharacterHelperConstants.SKH_FILE_DOCUMENT_SAVEGAME_FOLDER_2K_XP;
    if (true == lc_bWin7 ) lc_sHomeDirectory += SkyrimCharacterHelperConstants.SKH_FILE_DOCUMENT_FOLDER_LATER + SkyrimCharacterHelperConstants.SKH_FILE_DOCUMENT_SAVEGAME_FOLDER_LATER  ;

    lc_sHomeDirectory += SkyrimCharacterHelperConstants.SKH_FILE_SKYRIM_SAVEGAME_FOLDER;
   
    if (null != (lc_sPath = checkPath(lc_sHomeDirectory)))
    {
     if (0 == JOptionPane.showOptionDialog(m_tView,  
                                           insertString(getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TEXT_FRESH_START_SAVEGAME), lc_sPath), 
                                           getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TITLE_INFORMATION), 
                                           JOptionPane.YES_NO_OPTION, JOptionPane.INFORMATION_MESSAGE, null, 
                                           getLocalizedYesNoOptions(), 
                                           getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_OPTION_YES)))
     {
      //
      // use it!
      //
      m_tProperties.put(SkyrimCharacterHelperConstants.SKH_PROPERTIES_KEY_ACTIVE_PATH, lc_sPath); 
      return lc_sPath;
     }
    }
   }
  }
  return null;
 }
 
 

 /**
  * Handles changing the image mode
  * 
  */
 public void handleComboBoxImageMode()
 {
  int lc_iMode = m_tView.getImageMode();
  
  if (SkyrimCharacterHelperConstants.SKH_IMAGEMODE_SHOW == lc_iMode)
  {
   javax.swing.SwingUtilities.invokeLater(new Runnable()
   {
    public void run() 
    { 
     if (null == m_tView.getActiveSaveGameScreenshot()) performSaveGameListSelection(true );    
     if (null == m_tView.getBackupSaveGameScreenshot()) performSaveGameListSelection(false);    
    }
   });      
  }
 }

 
 
 
 /**
  * Handles clicking the launch button for the active savegame list
  * 
  * Step 1: move anything besides the currently selected active savegame to the backup directory
  * Step 2: launch Skyrim
  * 
  */
 public void handleButtonActiveLaunchSaveGame()
 {
  boolean lc_bContinue         = true;
  boolean lc_bJustOneSaveGame  = m_tView.hasOneActiveSaveGame();
  int     lc_iCount            = 0   ;
  
  SkyrimCharacterHelperPlayer         lc_tSelectedActivePlayer          = m_tView.getFirstSelectedActivePlayer();  
  SkyrimCharacterHelperPlayer         lc_tSelectedDestinationPlayer     = null                                  ;  
  List<SkyrimCharacterHelperSaveGame> lc_tSaveGames                     = null;
  String                              lc_sSelectedDestinationPlayerName = null;
  
  if (null != lc_tSelectedActivePlayer)
  {
   //
   // delete warning
   //
   if (false == lc_bJustOneSaveGame && SkyrimCharacterHelperConstants.SKH_CONFIRM_DELETE == m_tView.getConfirmationMode())    
   {
    lc_bContinue = (0 == JOptionPane.showOptionDialog(m_tView,  
                                                      getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TEXT_MOVE_FOR_LAUNCH_ACTIVE_SAVEGAME), 
                                                      getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TITLE_WARNING         ), 
                                                      JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE, null, 
                                                      getLocalizedOkCancelOptions()                                   , 
                                                      getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_OPTION_OK)));
   }
   if (true == lc_bContinue)
   {  
    //
    // get savegames of concerned players
    // 
    lc_tSaveGames = getSaveGamesExceptSelectedSavegame(true);
    
    if (true == lc_bJustOneSaveGame || null != lc_tSaveGames)
    {
     if (false == lc_bJustOneSaveGame)
     {
      //
      // overwrite warning
      // 
      if (SkyrimCharacterHelperConstants.SKH_CONFIRM_YES == m_tView.getOverwriteMode() && 0 < (lc_iCount = getAmountOfFileCollisions(lc_tSaveGames, true)))
      {
       lc_bContinue = (0 == JOptionPane.showOptionDialog(m_tView,  
                                                    insertNumberString(getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TEXT_OVERWRITE_SAVEGAME), lc_iCount), 
                                                    getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TITLE_WARNING), 
                                                    JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE, null, 
                                                    getLocalizedOkCancelOptions()                                   , 
                                                    getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_OPTION_OK)));
      }
     }
     
     //
     // on continue, set launch flag
     //
     if (true == lc_bContinue)
     {
      m_bPrepareForLaunching    = true; 
      m_sLaunchPlayer = lc_tSelectedActivePlayer.getName();    

      if (false == lc_bJustOneSaveGame)
      {
       //
       // backup the savegames. when this thread returns, the handler-routine will launch Skyrim.
       //
       if (null != (m_tWorkerCopySaveGames = new SwingWorkerSaveGameCopy()))
       {
        lc_tSelectedDestinationPlayer     = m_tView.getFirstSelectedBackupPlayer();
        lc_sSelectedDestinationPlayerName = (null != lc_tSelectedDestinationPlayer ? lc_tSelectedDestinationPlayer.getName() : null);
         
        m_tView.setMessage(getLocalizedString(SkyrimCharacterHelperConstants.SKH_MESSAGE_BACKING_UP));          
        m_tView.setUILockMode(true, false);   
        
        m_tWorkerCopySaveGames.init(SkyrimCharacterHelperConstants.SKH_WORKER_THREAD_LAUNCH_FROM_ACTIVE,
                                    SkyrimCharacterHelperConstants.SKH_WORKMODE_MOVE,
                                    lc_tSaveGames, 
                                    true, 
                                    m_tView.hasListMultiSelectionBackupPlayers(),
                                    lc_sSelectedDestinationPlayerName);
       
        m_tWorkerCopySaveGames.execute(); 
       }
      }
      else
      {
       m_bPrepareForLaunching = false;
       m_tView.setMessage(getLocalizedString(SkyrimCharacterHelperConstants.SKH_MESSAGE_LAUNCH_SINGLE));       
       launchSkyrim(); 
      }
     }
    }
   }
  }
 }
 
 
 
 /**
  * Handles clicking the launch button for the backup savegame list
  * 
  * Step 1: move the currently selected savegame to Skyrim's savegame directory and select this player
  * Step 2: move anything which does not belong to the currently selected active player to the backup directory
  * Step 3: launch Skyrim
  * 
  */ 
 public void handleButtonBackupLaunchSaveGame()
 {
  boolean lc_bContinue = true;
  int     lc_iCount    = 0   ;
  
  SkyrimCharacterHelperPlayer         lc_tSelectedBackupPlayer          = m_tView.getFirstSelectedBackupPlayer()  ;  
  SkyrimCharacterHelperSaveGame       lc_tSelectedBackupSavegame        = m_tView.getFirstSelectedBackupSaveGame();  
  SkyrimCharacterHelperPlayer         lc_tSelectedDestinationPlayer     = null                                    ;  
  
  ArrayList<SkyrimCharacterHelperSaveGame> lc_tSaveGames                = new ArrayList<>();
  String                              lc_sSelectedDestinationPlayerName = null;
  
  if (null != lc_tSelectedBackupPlayer && null != lc_tSelectedBackupSavegame && null != lc_tSaveGames)
  {
   // 
   // append selected savegame to list for later processing
   //
   lc_tSaveGames.add(lc_tSelectedBackupSavegame);
           
   //
   // delete warning
   //
   if (SkyrimCharacterHelperConstants.SKH_CONFIRM_DELETE == m_tView.getConfirmationMode())    
   {
    lc_bContinue = (0 == JOptionPane.showOptionDialog(m_tView,  
                                                      getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TEXT_MOVE_FOR_LAUNCH_BACKUP_SAVEGAME), 
                                                      getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TITLE_WARNING), 
                                                      JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE, null, 
                                                      getLocalizedOkCancelOptions()                                   , 
                                                      getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_OPTION_OK)));
   }
   if (true == lc_bContinue)
   {  
    //
    // overwrite warning
    // 
    if (SkyrimCharacterHelperConstants.SKH_CONFIRM_YES == m_tView.getOverwriteMode() && 0 < (lc_iCount = getAmountOfFileCollisions(lc_tSaveGames, false)))
    {
     lc_bContinue = (0 == JOptionPane.showOptionDialog(m_tView,  
                                                  insertNumberString(getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TEXT_OVERWRITE_SAVEGAME), lc_iCount), 
                                                  getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_TITLE_WARNING), 
                                                  JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE, null, 
                                                  getLocalizedOkCancelOptions()                                   , 
                                                  getLocalizedString(SkyrimCharacterHelperConstants.SKH_DIALOG_OPTION_OK)));
    }
     
    //
    // on continue, set launch flag
    //
    if (true == lc_bContinue)
    {
     m_bPrepareForLaunching = true; 
     m_sLaunchPlayer = lc_tSelectedBackupPlayer.getName();
     
     //
     // restore the selected savegame. when this thread returns, the handler-routine will launch Skyrim.
     //
     if (null != (m_tWorkerCopySaveGames = new SwingWorkerSaveGameCopy()))
     {
      lc_tSelectedDestinationPlayer     = m_tView.getFirstSelectedActivePlayer();
      lc_sSelectedDestinationPlayerName = (null != lc_tSelectedDestinationPlayer ? lc_tSelectedDestinationPlayer.getName() : null);
       
      m_tView.setMessage(getLocalizedString(SkyrimCharacterHelperConstants.SKH_MESSAGE_RESTORING));          
      m_tView.setUILockMode(true, false);   
       
      m_tWorkerCopySaveGames.init(SkyrimCharacterHelperConstants.SKH_WORKER_THREAD_LAUNCH_FROM_BACKUP,
                                  SkyrimCharacterHelperConstants.SKH_WORKMODE_MOVE,
                                  lc_tSaveGames, 
                                  false, 
                                  m_tView.hasListMultiSelectionActivePlayers(),
                                  lc_sSelectedDestinationPlayerName);
       
      m_tWorkerCopySaveGames.execute(); 
     }     
    }
   }
  }   
 }
 

 

 
} // eoc
   
