/**
 * SkyrimCharacterHelper
 * 
 * UI based tool to save and backup skyrim character save-files
 * 
 */
package skyrimcharacterhelper;



/**
 * Imports
 * 
 */
import java.awt.Toolkit;
import java.util.List             ;
import java.io.File               ;
import java.io.InputStream        ;  
import java.io.BufferedInputStream;
import java.io.IOException        ;
import java.io.FileInputStream    ;
import java.io.FileOutputStream   ;
import java.io.ByteArrayInputStream;

import java.nio.file.Path                   ; 
import java.nio.file.WatchService           ;
import java.nio.file.WatchEvent             ;
import java.nio.file.WatchKey               ;
import java.nio.file.StandardWatchEventKinds;

import javax.sound.sampled.Clip       ;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.AudioInputStream;
import javax.sound.sampled.LineListener    ;
import javax.sound.sampled.LineEvent       ;



/**
 * Directory sniffer thread, which listens for updates on savegames and autosaves and copies those
 * to ordinary savegames
 * 
 * @author Ulf Wagemann
 */
public class SkyrimCharacterHelperDirectorySnifferThread extends    Thread
                                                         implements LineListener
{
 private String  m_sDirectory ;
 private String  m_sSeparator ;
 private String  m_sPrefix    ;
 
 private File         m_tDirectory   ;
 private WatchService m_tWatchService;
 private WatchKey     m_tWatchKey    ;
 
 private boolean m_bCopyAutoSaves ;  
 private boolean m_bCopyQuickSaves; 
 private boolean m_bRunning       ;
 private boolean m_bSoundLoaded   ;

 private SkyrimCharacterHelperThreadNotifier m_tNotifier;
 
 private byte[]            m_tSoundClipArray     ;
 private Clip              m_tClip               ;
 private AudioInputStream  m_tAudioInputStream   ;
 private InputStream       m_tInputStream        ;
 private InputStream       m_tBufferedInputStream;
 
 
 /**
  * Constructor
  * 
  * @param pa_sPrefix          First part of a savegame name, i.e. "Speichern" for the German game client
  * @param pa_sDirectory       directory to surveil
  * @param pa_bCopyAutoSaves   true =  copy autosaves to savegames
  * @param pa_bCopyQuickSaves  true =  copy quicksaves to savegames
  */
 public SkyrimCharacterHelperDirectorySnifferThread(SkyrimCharacterHelperThreadNotifier pa_tNotifier, String pa_sPrefix, String pa_sDirectory, boolean pa_bCopyAutoSaves, boolean pa_bCopyQuickSaves)
 {
  m_sDirectory    = pa_sDirectory;
  m_bRunning      = false        ;
  m_sPrefix       = pa_sPrefix   ;
  
  m_sSeparator           = System.getProperty("file.separator");
  m_tDirectory           = new File(pa_sDirectory);
  m_tWatchKey            = null;
  m_tWatchService        = null;
  m_tSoundClipArray      = null;
  m_tClip                = null;
  m_tAudioInputStream    = null;
  m_tInputStream         = null;
  m_tBufferedInputStream = null;
  
  m_bCopyAutoSaves  = pa_bCopyAutoSaves ;
  m_bCopyQuickSaves = pa_bCopyQuickSaves;
  
  m_tNotifier = pa_tNotifier;
 }              
  
 
  
 /**
  * Returns whether this thread is running
  * 
  * @return   true if it is executing, false otherwise
  */
 public final boolean isRunning() {return  m_bRunning;}
 
 
  
 /**
  * Smoothly cancels the thread
  * 
  */
 public final void terminate() 
 {   
  m_bRunning = false;
 }
 
 
  
 /**
  * Sleeps for the given amount of millis
  * 
  * @param pa_lTime  millis
  */
 private void goToSleep(long pa_lTime) // my first goto since 25 years.... ;-)
 {
  try 
  {
   sleep(pa_lTime);
  } 
  catch (InterruptedException lc_tException) {} 
  catch (Exception lc_tException) {} 
 }

 
 
 /**
  * Thread.run : sniffes for directory changes on autosaves and quicksaves
  * 
  */
 @Override
 public final void run()
 {
  boolean      lc_bContinue     = false;
  Path         lc_tPath         = null ;
  Path         lc_tPathContext  = null ;
  String       lc_sFileName     = null ;
  
  List<WatchEvent<?>>   lc_tEvents = null;
  WatchEvent.Kind<Path> lc_tKind   = null;
      
  //
  // check stuff
  //
  if (null != m_tDirectory)
  {
   if (true == m_tDirectory.isDirectory()) 
   {
    if (true == m_tDirectory.canWrite()  && true ==  m_tDirectory.canRead())
    {
     lc_bContinue = true;
    }
   }
  }
  
  //
  // setup listener
  //
  if (true == lc_bContinue)
  {
   lc_bContinue = false;
   
   if (null != (lc_tPath = m_tDirectory.toPath()))
   {
    try 
    {
     if (null != (m_tWatchService = lc_tPath.getFileSystem().newWatchService()))
     {
      lc_tPath.register(m_tWatchService, StandardWatchEventKinds.ENTRY_DELETE, StandardWatchEventKinds.ENTRY_MODIFY);
      lc_bContinue = true;
     }
    } 
    catch (IOException lc_tException) 
    {
     lc_tPath = null;
    }
   }
  }

  //
  // read sound into byte buffer to avoid repeated loading of the sound clip
  //
  m_bSoundLoaded = readSoundFile();
  
  //
  // now we can start listening
  //
  if (true == lc_bContinue)
  {
   m_bRunning = true;
   
   //
   // loop until none of the two known processes are active anymore
   //
   while (true == m_bRunning)
   {
    //
    // listen
    // 
    try
    {
     if (null != m_tWatchService) 
     {
      m_tWatchKey = m_tWatchService.take();
     }
    }
    catch (InterruptedException lc_tException) {}
    catch (Exception lc_tException) {}
    
    //
    // something has happened
    //
    if (null != m_tWatchKey)
    {
     if (true == m_tWatchKey.isValid()) // 02.05.2012 check validity before proceeding
     {
      lc_tEvents = null;
      
      if (null != (lc_tEvents = m_tWatchKey.pollEvents()))
      {
       m_tWatchKey.reset();      
       
       if (0 < lc_tEvents.size())
       {
        for (WatchEvent<?> lc_tWatchEvent : lc_tEvents) 
        {
         if (null != lc_tWatchEvent)
         {
          lc_tKind        = (WatchEvent.Kind<Path>) lc_tWatchEvent.kind();
	      lc_tPathContext = (Path) lc_tWatchEvent.context();
          lc_sFileName    = lc_tPathContext.toString();
        
          //
          // just .ess and .ess.bak files
          //
          if (true == lc_sFileName.toLowerCase().endsWith(SkyrimCharacterHelperConstants.SKH_FILE_EXTENSION_ESS    ) || 
              true == lc_sFileName.toLowerCase().endsWith(SkyrimCharacterHelperConstants.SKH_FILE_EXTENSION_ESS_BAK)  )
          {
           //
           // any file creation triggers two events:
           //
           // 1. create
           // 2. modify
           // 
           // So we *MUST* even listen to the modify-event to ensure that Skyirm is done with that file! the main program is responsible to find out
           // whether it was a pure creation process or a file creation
           //
           if (lc_tKind.equals(StandardWatchEventKinds.ENTRY_MODIFY)) 
           {
            //
            // always notify main
            //
            if (null != m_tNotifier) m_tNotifier.notifySkyrimSavegameModified(lc_sFileName);  

            //
            // for autosaves and quicksaves only, we create new savegames, if needed. but not for backups of autosaves and quicksaves.
            // 
            if (false == lc_sFileName.endsWith(SkyrimCharacterHelperConstants.SKH_FILE_EXTENSION_ESS_BAK))
            {
             if ((true == m_bCopyAutoSaves  && true == lc_sFileName.startsWith(SkyrimCharacterHelperConstants.SKH_FILE_TEMP_SLOT_AUTO)) ||
                 (true == m_bCopyQuickSaves && true == lc_sFileName.startsWith(SkyrimCharacterHelperConstants.SKH_FILE_TEMP_SLOT_QUICK)))
             { 
              //
              // if something has happened, load it into a savegame for further analyzation. if s savegame is created, this will trigger a MODIFY event soon,
              // which then is propagated to main using the default mechanism :)
              //
              checkCreateSavegame(lc_sFileName);
             }
            }
           }
           // 
           // any file deletion triggers one event:
           // 
           // 1. delete
           //           
           else if (lc_tKind.equals(StandardWatchEventKinds.ENTRY_DELETE)) 
           {
            if (null != m_tNotifier) m_tNotifier.notifySkyrimSavegameDeleted(lc_sFileName);
           }
          }
         }
        } // for
       }
      }
     }
    }
    
   } // while 
  }
  
  //
  // security check for shutting down watch service. hardended on 02.05.2012.
  //
  if (null != m_tWatchKey    ) {try {if (true == m_tWatchKey.isValid())  m_tWatchKey.cancel();} catch (Exception lc_tException) {}}
  if (null != m_tWatchService) {try {m_tWatchService.close();                                 } catch (Exception lc_tException) {}} // close down service, too
 }
 

 
 /**
  * Scans the savegame just created by Skyrim
  * 
  * @param pa_sFileName   filename without any path
  * */
 private void checkCreateSavegame(String pa_sFileName)
 {
  SkyrimCharacterHelperSaveGame lc_tSaveGame = null;
  
  String lc_sFullPath        = null;
  String lc_sSaveGamePath    = null;
  String lc_sNewSaveGamePath = null;
  String lc_sPlayerName      = null;
  String lc_sPlayerLocation  = null;
  String lc_sGameDate        = null;
  String lc_sGameNumber      = null;
  String lc_sNewGameNumber   = null;
  
  File[] lc_tFiles    = null;
  
  int    lc_iMaxNum      = 0   ;
  int    lc_iSubStartPos = -1  ;
  int    lc_iSubEndPos   = -1  ;
  int    lc_iStartPos    = -1  ;
  int    lc_iEndPos      = -1  ;
  
  int    lc_iGameNumber    = 0;
  int    lc_iNewGameNumber = 0;
  
  if (null != pa_sFileName)
  {
   lc_sFullPath = m_sDirectory + m_sSeparator + pa_sFileName;
 
   //
   // analyze savegame
   //
   if (null != (lc_tSaveGame = new SkyrimCharacterHelperSaveGame()))
   {
    if (true == lc_tSaveGame.read(lc_sFullPath, null, true)) // here we can read including the screenshot
    {
     lc_sPlayerName     = lc_tSaveGame.getPlayerName()    ;
     lc_sPlayerLocation = lc_tSaveGame.getPlayerLocation();
     lc_sGameDate       = lc_tSaveGame.getGameDate()      ;
     
     //
     // now get all files which belong to this player
     //
     if (null != (lc_tFiles = m_tDirectory.listFiles()))
     {
      for (File lc_tFile : lc_tFiles) 
      {
       if (null != lc_tFile)
       {
        if (null != (lc_sSaveGamePath = lc_tFile.getName()))
        {
         //
         // only consider player savegames
         //
         if (-1 != (lc_iEndPos = lc_sSaveGamePath.indexOf(lc_sPlayerName)))
         {
          if (-1 != (lc_iStartPos = lc_sSaveGamePath.indexOf(m_sPrefix)))
          {
           lc_iSubStartPos = lc_iStartPos + m_sPrefix.length() + 1; // " " behind prefix
           lc_iSubEndPos   = lc_iEndPos - 3                       ; // " - " behind game number
           
           if (0 <= lc_iStartPos && 0 <= lc_iSubEndPos && lc_iSubStartPos <  lc_sSaveGamePath.length() && lc_iSubEndPos <  lc_sSaveGamePath.length())    
           {
            try
            {
             if (null != (lc_sGameNumber = lc_sSaveGamePath.substring(lc_iSubStartPos, lc_iSubEndPos)))
             {
              if (null == lc_sGameNumber          ) lc_sGameNumber = SkyrimCharacterHelperConstants.SKH_FILE_DEFAULT_NUMBER;
              if (true == lc_sGameNumber.isEmpty()) lc_sGameNumber = SkyrimCharacterHelperConstants.SKH_FILE_DEFAULT_NUMBER;    

              //
              // and memorize highest number
              //
              lc_iGameNumber = Integer.valueOf(lc_sGameNumber);
              if (lc_iGameNumber > lc_iMaxNum)
              {
               lc_iMaxNum = lc_iGameNumber;
              }
             }
            }
            catch (Exception lc_tException) {}
           }  
          }
         }
        }
       }
      } // for
      
      //
      // increase game number
      //
      try
      {
       lc_iNewGameNumber = (Integer.MAX_VALUE-1 > lc_iMaxNum ? lc_iMaxNum+1 : 1);
       lc_sNewGameNumber = Integer.toString(lc_iNewGameNumber);
      }
      catch (Exception lc_tException)
      {
       lc_iNewGameNumber = 1; 
      }
      
      //
      // check game time for leading "000", which is not allowed. no idea why they did it this way, but well... we want to follow the rules...
      //
      if (null != lc_sGameDate)
      {
       if (!lc_sGameDate.startsWith("00.") &&  lc_sGameDate.startsWith("00")) {lc_sGameDate = lc_sGameDate.substring(1);} //000x is not allowed, 00 is
       if ( lc_sGameDate.startsWith("0"  ) && !lc_sGameDate.startsWith("00")) {lc_sGameDate = lc_sGameDate.substring(1);} //reduce 0xx to xx, but keep 00
      }
      //
      // create new name and copy file
      //
      lc_sNewSaveGamePath = m_sDirectory + m_sSeparator + m_sPrefix + " " + lc_sNewGameNumber + " - " + lc_sPlayerName + " " + lc_sPlayerLocation + " " + lc_sGameDate + SkyrimCharacterHelperConstants.SKH_FILE_EXTENSION_ESS;
      
      if (true == copyFile(lc_sFullPath, lc_sNewSaveGamePath))
      {
       playSoundFromByteArray();
      }
     }     
    }
   }
  }
 }
 
 
 
 /**
  * Copies a quicksave or autosave file to a new savegame following the current numbering for that character :-)
  * 
  * @param pa_sSource            source path
  * @param pa_sDestination       destination path
  */
 private boolean copyFile(String pa_sSource, String pa_sDestination)
 {
  int    lc_iBytesRead      = 0;
  int    lc_iBytesReadTotal = 0;

  File             lc_tSourceFile       = null;
  File             lc_tDestinationFile  = null;
  FileInputStream  lc_tInputStream      = null;
  FileOutputStream lc_tOutputStream     = null;
  byte[]           lc_tBuffer           = new byte[SkyrimCharacterHelperConstants.SKH_FILE_BUFFER_SIZE];
   
  boolean lc_bResult = false;
  
  if (null != pa_sSource && null != pa_sDestination && null != lc_tBuffer) 
  {
   if (false == pa_sSource.isEmpty() && false == pa_sDestination.isEmpty()) 
   {
    //
    // obtain source top copy
    //
    if (null != (lc_tSourceFile = new File(pa_sSource))) 
    {
     lc_iBytesRead = 0;

     try
     {
      if (null != (lc_tInputStream = new FileInputStream(lc_tSourceFile))) 
      {
       //
       // obtain destination file
       //
       if (null != (lc_tOutputStream = new FileOutputStream(pa_sDestination)))
       {
        //
        // copy one file. 
        //
        try
        {
         while ((lc_iBytesRead = lc_tInputStream.read(lc_tBuffer)) > 0)
         {
          lc_tOutputStream.write(lc_tBuffer, 0, lc_iBytesRead);
          lc_iBytesReadTotal += lc_iBytesRead;
         } // while
        }
        catch (Exception lc_tException) {}
        
        //
        // close output stream
        //
        if (null != lc_tOutputStream) try{lc_tOutputStream.close();} catch (Exception lc_tException){}

        //
        // use timestamp of source savegame !
        //
        if (null != (lc_tDestinationFile = new File(pa_sDestination))) {lc_tDestinationFile.setLastModified(lc_tSourceFile.lastModified());}
        
        //
        // result
        //
        lc_bResult = true;
       }
       //                   
       // close input stream 
       //
       if (null != lc_tInputStream ) try{lc_tInputStream.close() ;} catch (Exception lc_tException){}
      }
     }
     catch (Exception lc_tException) {}            
    }
   }
  }
  return lc_bResult;
 }
 
 

 /**
  * Plays a .wav-file once
  * 
  */
 private void playSoundFromFile()
 {
  //
  // if we don't have a valid soud, we just beep at least
  //
  if (false == m_bSoundLoaded) 
  {
   Toolkit.getDefaultToolkit().beep(); 
   return;
  }
  

  //
  // Setup stream. Note: 
  //
  // The documentation for AudioSystem.getAudioInputStream(InputStream) says: 
  //
  // The implementation of this method may require multiple parsers to examine the stream to determine whether they support it. 
  // These parsers must be able to mark the stream, read enough data to determine whether they support the stream, and, if not, 
  // reset the stream's read pointer to its original position. If the input stream does not support these operation, this method 
  // may fail with an IOException.
  //
  // Therefore, we decorate the input stream with a buffered input stream to provide this mark/reset functionality. 
  //
  try
  {
   if (null != (m_tClip = AudioSystem.getClip()))
   {
    if (null != SkyrimCharacterHelperDirectorySnifferThread.class.getClassLoader())
    {
     if (null != (m_tInputStream = SkyrimCharacterHelperDirectorySnifferThread.class.getClassLoader().getResourceAsStream(SkyrimCharacterHelperConstants.SKH_FILE_CLICK_SOUND)))
     {
      if (null != (m_tBufferedInputStream = new BufferedInputStream(m_tInputStream)))
      {
       if (null != (m_tAudioInputStream = AudioSystem.getAudioInputStream(m_tBufferedInputStream)))
       {
        try
        {
         m_tClip.addLineListener((LineListener) this); 
         m_tClip.open(m_tAudioInputStream); 
         m_tClip.loop(0);
        }
        catch (Exception lc_tException) {}
       }
      }
     }
    }
   }
  }
  catch (Exception lc_tException) {}  
 }
 
 
 
 /**
  * Plays the sound from the byte array
  * 
  */
 private void playSoundFromByteArray()
 {
  //
  // if we don't have a valid soud, we just beep at least
  //
  if (false == m_bSoundLoaded) 
  {
   Toolkit.getDefaultToolkit().beep(); 
   return;
  }

  //
  // Setup stream. Note: 
  //
  // The documentation for AudioSystem.getAudioInputStream(InputStream) says: 
  //
  // The implementation of this method may require multiple parsers to examine the stream to determine whether they support it. 
  // These parsers must be able to mark the stream, read enough data to determine whether they support the stream, and, if not, 
  // reset the stream's read pointer to its original position. If the input stream does not support these operation, this method 
  // may fail with an IOException.
  //
  // Therefore, we decorate the input stream with a buffered input stream to provide this mark/reset functionality. 
  //
  try
  {
   if (null != m_tSoundClipArray && null != (m_tClip = AudioSystem.getClip()))
   {
    if (null != SkyrimCharacterHelperDirectorySnifferThread.class.getClassLoader())
    {
     if (null != (m_tInputStream = new ByteArrayInputStream(m_tSoundClipArray)))
     {
      if (null != (m_tBufferedInputStream = new BufferedInputStream(m_tInputStream)))
      {
       if (null != (m_tAudioInputStream = AudioSystem.getAudioInputStream(m_tBufferedInputStream)))
       {
        try
        {
         m_tClip.addLineListener((LineListener) this); 
         m_tClip.open(m_tAudioInputStream); 
         m_tClip.loop(0);
        }
        catch (Exception lc_tException) {}
       }
      }
     }
    }
   }
  }
  catch (Exception lc_tException) {}  
  
 }
 
 
 
 /**
  * Reads the sound file into a byte array
  * 
  * @return  true on success, false otherwise
  */
 private boolean readSoundFile()
 {
  FileInputStream  lc_tFileInputStream     = null;
  FileOutputStream lc_tFileOutputStream    = null;
  InputStream      lc_tInputStream         = null;
  InputStream      lc_tBufferedInputStream = null;
  File             lc_tFile                = null;
  int              lc_iBytesRead           = 0;
  int              lc_iSize                = 0;
  
  byte[]  lc_tBytes   = new byte[100000];
  boolean lc_bHasTemp = false;
  boolean lc_bResult  = false;
  
  //
  // step one: create temp-file from sound file
  //
  if (null != (lc_tInputStream = SkyrimCharacterHelperDirectorySnifferThread.class.getClassLoader().getResourceAsStream(SkyrimCharacterHelperConstants.SKH_FILE_CLICK_SOUND)))
  {
   if (null != (lc_tBufferedInputStream = new BufferedInputStream(lc_tInputStream)))
   {
    try
    {
     if (null != (lc_tFileOutputStream = new FileOutputStream(SkyrimCharacterHelperConstants.SKH_FILE_TEMP_SOUND_FILE)))
     {
      try
      {
       while (0 < (lc_iBytesRead = lc_tBufferedInputStream.read(lc_tBytes)))
       {
        lc_tFileOutputStream.write(lc_tBytes, 0, lc_iBytesRead);
       } // while     
      }
      catch (Exception lc_tException) {}
      
      //
      // close temp file
      //
      try {lc_tFileOutputStream.close();} catch (Exception lc_tException) {}
      lc_bHasTemp = true;
     }
    }
    catch (Exception lc_tException) {}
    
    //
    // close buffered input stream
    //
    try {lc_tBufferedInputStream.close();} catch (Exception lc_tException) {}
   } 
   //
   // close input stream
   //
   try {lc_tInputStream.close();} catch (Exception lc_tException) {}
  }
  
  //
  // temp two: get file size, read temp file and delete it
  //
  if (true == lc_bHasTemp)
  {
   if (null != (lc_tFile = new File(SkyrimCharacterHelperConstants.SKH_FILE_TEMP_SOUND_FILE)))
   {
    if (0 < (lc_iSize = (int) lc_tFile.length()))
    {
     if (null != (m_tSoundClipArray = new byte[lc_iSize]))
     {
      try
      {
       if (null != (lc_tFileInputStream = new FileInputStream(lc_tFile)))
       {
        lc_iBytesRead = lc_tFileInputStream.read(m_tSoundClipArray, 0, lc_iSize);
        lc_bResult    = (lc_iBytesRead == lc_iSize);
        
        //
        // close temp file
        //
        try {lc_tFileInputStream.close();} catch (Exception lc_tException) {}
       }
      }
      catch (Exception lc_tException) {}
     }
    }
    //
    // delete temp file
    //
    try {lc_tFile.delete();} catch (Exception lc_tException) {}
   }
  }
  return lc_bResult;
 }


 /**
  * LineListener.update, closes stuff on close event
  * 
  * @param pa_tLineEvent 
  */
 public void update(LineEvent pa_tLineEvent) 
 {
  if (null != pa_tLineEvent)
  {
   if (pa_tLineEvent.getType() == LineEvent.Type.STOP) 
   {
    try {if (null != m_tClip               ) {m_tClip.close()               ; m_tClip                = null;}} catch (Exception lc_tException) {}
    try {if (null != m_tAudioInputStream   ) {m_tAudioInputStream.close()   ; m_tAudioInputStream    = null;}} catch (Exception lc_tException) {}       
    try {if (null != m_tBufferedInputStream) {m_tBufferedInputStream.close(); m_tBufferedInputStream = null;}} catch (Exception lc_tException) {}       
    try {if (null != m_tInputStream        ) {m_tInputStream.close()        ; m_tInputStream         = null;}} catch (Exception lc_tException) {}       
   }
  }
 }
 
} // eoc
  