/**
 * SkyrimCharacterHelper
 * 
 * UI based tool to save and backup skyrim character save-files
 * 
 */
package skyrimcharacterhelper;



/**
 * Imports
 * 
 */
import java.io.IOException      ;
import java.io.InputStream      ;
import java.io.InputStreamReader;
import java.io.BufferedReader   ;

import java.util.List           ;
import java.util.ArrayList      ;



/**
 * Thread for launching Skyrim andthe directory sniffer.
 * 
 * @author Ulf Wagemann
 */
public class SkyrimCharacterHelperLaunchThread extends Thread
{
 private SkyrimCharacterHelperThreadNotifier   m_tNotifier; 
 private SkyrimCharacterHelperDirectorySnifferThread m_tDirectorySniffer;
 
 private String  m_sLaunchFile      ;
 private String  m_sLaunchParameters;
 private String  m_sSeparator       ;
 private boolean m_bRunning         ;
 private boolean m_bCopyAutoSaves   ;  
 private boolean m_bCopyQuickSaves  ; 
 private boolean m_bNotify          ; 
 
 
 
 /**--------------------------------------------------------------------------------------------------------------------
  * 
  * Class for reading in- and outputstreams properly when starting external applications
  * 
  * based on www.javaworld.com/javaworld/jw-12-2000/jw-1229-traps.html?page=1. Old, but true...
  */
 class StreamGobbler extends Thread
 {
  InputStream m_tInputStream;
    
  
  
  /**
   * Constructor
   * 
   * @param pa_tInputStream input stream to gobble
   */
  StreamGobbler(InputStream pa_tInputStream)
  {
   m_tInputStream = pa_tInputStream;
  }
    
  
  
  /**
   * Thread.run()
   * 
   * gobble!
   */
  public void run()
  {
   InputStreamReader lc_tInputStreamReader = (null != m_tInputStream        ? new InputStreamReader(m_tInputStream       ) : null);
   BufferedReader    lc_tBufferedReader    = (null != lc_tInputStreamReader ? new BufferedReader   (lc_tInputStreamReader) : null);

   String lc_sLine = null;
   
   //
   // read until nothing is left
   //
   if (null != lc_tBufferedReader)
   {
    try
    {
     while (null !=  (lc_sLine = lc_tBufferedReader.readLine()))
     {}
    }
    catch (IOException lc_tException){}
   }
   
   //
   // close stuff
   //
   try
   {
    if (null != lc_tBufferedReader   ) lc_tBufferedReader.close()   ; 
    if (null != lc_tInputStreamReader) lc_tInputStreamReader.close(); 
   }
   catch (IOException lc_tException){}
  }
 } // eoc --------------------------------------------------------------------------------------------------------------
 
 


 /**
  * Constructor
  * 
  * @param pa_sFileName        name of launchfile
  * @param pa_sPrefix          savegame prefix
  * @param pa_sDirectory       directory to monitor
  * @param pa_sParameters      command line options
  * @param pa_bCopyAutoSaves   true = convert autosaves to savegames
  * @param pa_bCopyQuickSaves  true = convert quicksaves to savegames
  */
 public SkyrimCharacterHelperLaunchThread(SkyrimCharacterHelperThreadNotifier pa_tNotifier, 
                                          String  pa_sLaunchFile   , 
                                          String  pa_sPrefix       , 
                                          String  pa_sDirectory    ,
                                          String  pa_sParameters   ,
                                          boolean pa_bCopyAutoSaves,
                                          boolean pa_bCopyQuickSaves)
 {
  m_sLaunchFile       = pa_sLaunchFile;
  m_tNotifier         = pa_tNotifier  ;
  m_sLaunchParameters = pa_sParameters;
  
  m_bRunning        = false         ;
  m_bNotify         = true          ;
  
  m_bCopyAutoSaves  = pa_bCopyAutoSaves ;
  m_bCopyQuickSaves = pa_bCopyQuickSaves;
  
  m_sSeparator        = System.getProperty("file.separator");
  m_tDirectorySniffer = new SkyrimCharacterHelperDirectorySnifferThread(pa_tNotifier, pa_sPrefix, pa_sDirectory, pa_bCopyAutoSaves, pa_bCopyQuickSaves);
 }              
  
 
  
 /**
  * Returns whether this thread is running
  * 
  * @return   true if it is executing, false otherwise
  */
 public final boolean isRunning() {return  m_bRunning;}
 
 
  
 /**
  * Smoothly cancels the thread
  * 
  */
 public final void terminate() 
 {
  if (null != m_tDirectorySniffer) m_tDirectorySniffer.terminate();
  
  m_bNotify  = false; // no notification, if cancelled from outside!
  m_bRunning = false;
 }
 
 
 
 /**
  * Thread.run : launches Skyrim
  * 
  */
 @Override
 public final void run()
 {
  int lc_iLaunchResult   = -1;
  int lc_iProcessResult  = -1;
  
  //
  // Perform the launch. Only head on, if launching was successful.
  //
  m_bRunning = true;
  if (SkyrimCharacterHelperConstants.SKH_FILE_SYSTEM_COMMAND_OK == (lc_iLaunchResult = launch()))
  {
   //
   // wait max60s or until a Skyrim-related process appears. -1 indicates that we had problems obtaining the process list.
   //
   if (-1 != waitForSkyrimProcesses())
   {
    //
    // always start the siffer
    //
    if (null != m_tDirectorySniffer) m_tDirectorySniffer.start();

    //
    // loop until none of the two known processes are active anymore
    //
    try
    {
     while (true == m_bRunning)
     { 
      goToSleep(SkyrimCharacterHelperConstants.SKH_FILE_PROCESS_DELAY_TIME); // sleep, baby, sleep!

      //
      // obtain process list and search for standard binary, SKSE binary and launcher binary
      //   
      lc_iProcessResult = checkForSkyrimProcesses();
      m_bRunning        = (0 < lc_iProcessResult)  ;
     } // while
    }
    catch (Exception lc_tException) {}
    
    //
    // finally, terminate sniffer and notify main thread
    //  
    if (null != m_tDirectorySniffer) m_tDirectorySniffer.terminate();
   }
  }
  //
  // notify main thread
  
  if (true == m_bNotify)
  {
   if (null != m_tNotifier) m_tNotifier.notifySkyrimTerminated(lc_iLaunchResult);
  }
 }
 


 /**
  * Launches the defined launch target
  * 
  * @return   exit code
  */
 private int launch()
 {
  Process       lc_tExecuteProcess = null;
  StreamGobbler lc_tErrorGobbler   = null;
  StreamGobbler lc_tOutputGobbler  = null;
  int           lc_iExitValue      = (int) SkyrimCharacterHelperConstants.SKH_FILE_SYSTEM_COMMAND_OK;
 
  String lc_sLaunchTarget = m_sLaunchFile;
  
  //
  // add parameters, if needed
  //
  if (null != m_sLaunchParameters && null != lc_sLaunchTarget)
  {
   if (0 < m_sLaunchParameters.length())
   {
    if (false == m_sLaunchParameters.startsWith(" "))  lc_sLaunchTarget += " ";
    lc_sLaunchTarget += m_sLaunchParameters;
   }
  }
          
  if (null != lc_sLaunchTarget)
  {
   try 
   {
    if (false == isProcessRunning(m_sLaunchFile)) // do not double-start
    {
     //
     // get process from start via runtime
     // 
     if (null != (lc_tExecuteProcess = Runtime.getRuntime().exec(lc_sLaunchTarget)))
     {
      //
      // prepare stream gobblers
      //
      lc_tErrorGobbler  = new StreamGobbler(lc_tExecuteProcess.getErrorStream());            
      lc_tOutputGobbler = new StreamGobbler(lc_tExecuteProcess.getInputStream());               

      //
      // just kick them off. no need to scan their data
      //
      if (null != lc_tErrorGobbler ) lc_tErrorGobbler.start();
      if (null != lc_tOutputGobbler) lc_tOutputGobbler.start();
          
      //
      //  sync on the object to start so that we own the process monitor! if you don't do this,
      //  you'll be in exception hell.
      //
      synchronized(lc_tExecuteProcess) 
      {
       try   {lc_iExitValue = lc_tExecuteProcess.waitFor();} 
       catch (InterruptedException lc_tException) {}
      }         
     }
    }
   }
   catch (IOException lc_tException) {}   
  }
  return lc_iExitValue;
 }
 
 
 
 
 /**
  * Checks whether the given process is already running
  * 
  * @param pa_sProcess
  * @return  true, if it's already running, false otherwise
  */
 private boolean isProcessRunning(String pa_sProcess)
 {
  String  lc_sProcess = null ;
  boolean lc_bResult  = false;
  int     lc_iPos     = -1   ;
  
  List<String>  lc_tProcesses = null;
  
  //
  // extract last part of that name
  //
  if (null != pa_sProcess)
  {  
   lc_iPos     = pa_sProcess.lastIndexOf(m_sSeparator); 
   lc_sProcess = (-1 == lc_iPos ? pa_sProcess : pa_sProcess.substring(lc_iPos + m_sSeparator.length()));
  }
  
  //
  // check
  //
  if (null != lc_sProcess)
  {
   if (null != (lc_tProcesses = getProcesses()))
   {
    if (0 < lc_tProcesses.size())
    {
     lc_bResult  = lc_tProcesses.contains(lc_sProcess.toLowerCase()); 
    }
   }
  }
  return lc_bResult;
 }
 
 
 
 /**
  * Retrieves a list of process names with the help of the Windows tool tasklist.exe
  * 
  * @return  list of process names
  */
  private List<String> getProcesses() 
  {
   List<String>   lc_tProcesses   = null;
   Process        lc_tProcess     = null;
   String         lc_sLine        = null;
   String         lc_sWorkLine    = null;
   
   BufferedReader    lc_tInputReader       = null;
   InputStreamReader lc_tInputStreamReader = null;
   
   int lc_iPos = -1;
   
   try 
   {
    if (null != (lc_tProcess = Runtime.getRuntime().exec(SkyrimCharacterHelperConstants.SKH_SKYRIM_TASKLIST_CMD)))
    {
     if (null != lc_tProcess.getInputStream()) 
     {
      if (null != (lc_tInputStreamReader = new InputStreamReader(lc_tProcess.getInputStream())))
      {
       if (null != (lc_tInputReader = new BufferedReader(lc_tInputStreamReader)))
       {
        if (null != (lc_tProcesses = new ArrayList<>()))
        {
         while (null != (lc_sLine = lc_tInputReader.readLine())) 
         {
          if (false == lc_sLine.isEmpty()) 
          {
           if (null != (lc_sWorkLine = lc_sLine.substring(1).toLowerCase().trim())) // each output line starts with "
           {
            if (false == lc_sWorkLine.isEmpty())
            {
             if (-1 != (lc_iPos = lc_sWorkLine.indexOf("\""))) // get trailing "
             {
              lc_tProcesses.add(lc_sWorkLine.substring(0, lc_iPos));
             }
            }
           }
          }
         } // while
        }
        lc_tInputReader.close();
       }
       lc_tInputStreamReader.close();
      }
     }
    }
   }
   catch (Exception lc_tException) {}

   return lc_tProcesses;
  }
  
  
  
 /**
  * Sleeps for the given amount of millis
  * 
  * @param pa_lTime  millis
  */
 private void goToSleep(long pa_lTime) // my first goto since 25 years.... ;-)
 {
  try 
  {
   sleep(pa_lTime);
  } 
  catch (InterruptedException lc_tException) {} 
  catch (Exception lc_tException) {} 
 }

 
 
 /**
  * Checks whether any Skyrim-related process is running.
  * 
  * @return   -1 = error, 0 = no, 1 = yes
  */
 private int checkForSkyrimProcesses()
 {
  List<String>  lc_tProcesses      = null;
  boolean       lc_bProcessesFound = true;    
   
  if (null != (lc_tProcesses = getProcesses()))
  {
   if (0 < lc_tProcesses.size())
   {
    lc_bProcessesFound  = lc_tProcesses.contains(SkyrimCharacterHelperConstants.SKH_SKYRIM_BINARY_NORMAL  .toLowerCase()); // TESV.exe
    lc_bProcessesFound |= lc_tProcesses.contains(SkyrimCharacterHelperConstants.SKH_SKYRIM_BINARY_LAUNCHER.toLowerCase()); // SkyrimLauncher.exe
    lc_bProcessesFound |= lc_tProcesses.contains(SkyrimCharacterHelperConstants.SKH_SKYRIM_BINARY_SKSE.toLowerCase()    ); // skse_loader.exe
    
    return (true == lc_bProcessesFound ? 1 : 0);
   }
   return 0;
  }
  return -1;
 }
 
 
 
 /**
  * Waits max.60s for the first Skyrim-related process to appear
  * 
  */
 private int waitForSkyrimProcesses()
 {
  long    lc_lEndTime       = System.currentTimeMillis() + SkyrimCharacterHelperConstants.SKH_FILE_PROCESS_MAX_WAIT_TIME;
  int     lc_iProcessResult = -1   ;
  boolean lc_bDone          = false;
  
  while (false == lc_bDone && System.currentTimeMillis()  < lc_lEndTime)
  {
   lc_iProcessResult = checkForSkyrimProcesses();
   lc_bDone = (1 == lc_iProcessResult || -1 == lc_iProcessResult); // exit on success and on error
   
   if (false == lc_bDone) 
   {
    goToSleep(1000); 
   }
  }
  return lc_iProcessResult;
 }
 
 
 
} // eoc
