/**
 * SkyrimCharacterHelper
 * 
 * UI based tool to save and backup skyrim character save-files
 * 
 */
package skyrimcharacterhelper;



/**
 * Imports
 * 
 */
import java.util.Arrays     ;
import java.util.ArrayList  ;
import java.util.Collections;
import java.util.Comparator ;
import java.util.List       ;
import java.util.Objects    ;
import java.text.Collator   ;    



/**
 * Class for the data of a Skyrim character. Implements Comparable for the sorted lists :)
 * 
 * @author Ulf Wagemann
 */
public class SkyrimCharacterHelperPlayer implements Comparable<SkyrimCharacterHelperPlayer>
        
{
 private String m_sName;
 private String m_sRace;
 
 private ArrayList<SkyrimCharacterHelperSaveGame> m_tSaveGames;
 private static Collator                          m_tCollator ;
 
 
 
 /**
  * Constructor
  * 
  */
 public SkyrimCharacterHelperPlayer()
 {
  m_sName      = null;
  m_sRace      = null;
  m_tSaveGames = new ArrayList<>();
  
  m_tCollator = java.text.Collator.getInstance(); 
  m_tCollator.setStrength(java.text.Collator.TERTIARY);    
 }
 
 
 /**
  * Clears all
  * 
  */
public void clear()
{
 if (null != m_tSaveGames)  m_tSaveGames.clear();
}


 /**
  * Constructor
  * 
  * @param pa_tSaveGame  save game to add and intialize from
  */
 public SkyrimCharacterHelperPlayer(SkyrimCharacterHelperSaveGame pa_tSaveGame)
 {
  m_sName      = null;
  m_tSaveGames = new ArrayList<>();

  m_tCollator = java.text.Collator.getInstance(); 
  m_tCollator.setStrength(java.text.Collator.TERTIARY);  
  
  if (null != pa_tSaveGame)
  {
   m_sRace = pa_tSaveGame.getPlayerRace(); 
   m_sName = pa_tSaveGame.getPlayerName(); 
   addSaveGame(pa_tSaveGame);
  }
 }
 
 
 
 /**
  * Returns the amount of savegames for this character
  * 
  * @return    amount ofd savegames
  */
 public final int getSaveGameAmount() {return (null != m_tSaveGames ? m_tSaveGames.size() : 0);}
 
 
 
  
 /**
  * Returns the player race
  * 
  * @return   player race
  */
 public final String getRace() {return m_sRace;}

  
 
 
 /**
  * Returns the player name
  * 
  * @return   player name
  */
 public final String getName() {return m_sName;}

  
  
 /**
  * Sets the player name
  * 
  * @param pa_sName   new player name
  */
 public final void setName(String pa_sName) {m_sName = pa_sName;}

  

 
 /**
  * Adds a savegame to this character
  * 
  * @param pa_tSaveGame    save game
  * @rteturn  true, if a new record was added, false in case of overwriting or error
  */
 public final boolean addSaveGame(SkyrimCharacterHelperSaveGame pa_tSaveGame)
 {
  int     lc_iPos = -1;
  boolean lc_bFreshAdd = false;
  
  if (null != m_tSaveGames && null != pa_tSaveGame)
  {
   if (null == m_sRace) m_sRace = pa_tSaveGame.getPlayerRace(); 
 
   //
   // if we have such a savegame already, we need to overwrite it
   //
   if (-1 != (lc_iPos = m_tSaveGames.indexOf(pa_tSaveGame)))
   {
    m_tSaveGames.set(lc_iPos, pa_tSaveGame); 
   }
   else
   {
    m_tSaveGames.add(pa_tSaveGame); 
    lc_bFreshAdd = true;
   }
  }
  return lc_bFreshAdd;
 }
 
 
 
 /**
  * Returns the save games
  * 
  * @return   save games
  */
 public final SkyrimCharacterHelperSaveGame[] getSaveGames()
 {
  SkyrimCharacterHelperSaveGame[] lc_tArray = null;
  
  if (null != m_tSaveGames)
  {
   Collections.sort(m_tSaveGames, new Comparator<SkyrimCharacterHelperSaveGame>() 
                                  {
                                   public int compare(SkyrimCharacterHelperSaveGame o1, SkyrimCharacterHelperSaveGame o2)
                                   { 
                                    return m_tCollator.compare(o1.getFileName(), o2.getFileName()); // compare NAME and not PATH !!
                                   }
                                  });
   if (null != (lc_tArray = new SkyrimCharacterHelperSaveGame[m_tSaveGames.size()]))
   {
    return m_tSaveGames.toArray(lc_tArray);
   }
  }
  return null;   
 }

 
 /**
  * Retrieves the savegames of the character with the given name in form of a list
  * 
  * @return           list, null in case of error
  */
 public final List<SkyrimCharacterHelperSaveGame> getSaveGamesList()
 {
  SkyrimCharacterHelperSaveGame[]          lc_tSaveGames = null;
  ArrayList<SkyrimCharacterHelperSaveGame> lc_tList      = null;
  
  
  if (null != (lc_tSaveGames = getSaveGames()))
  {
   if (null != (lc_tList = new ArrayList<>()))
   {
    lc_tList.addAll(Arrays.asList(lc_tSaveGames));
   }
  }
  return lc_tList;
 }
 
 
 
 /**
  * Removes a savegame from the char
  * 
  * @param pa_tSaveGame   savegame
  * @return   true, if a savegame was removed, false otherwise
  */
 public final boolean removeSaveGame(SkyrimCharacterHelperSaveGame pa_tSaveGame)
 {
  if (null != pa_tSaveGame && null != m_tSaveGames && null != getName()) 
  {
   if (0 == m_tCollator.compare(getName(), pa_tSaveGame.getPlayerName())) 
   {
    return (m_tSaveGames.remove(pa_tSaveGame)); 
   }
  }
  return false;
 }
 
 
 
  
 /**
  * Removes a savegame from the char
  * 
  * @param pa_tSaveGame   savegame
  * @return   true, if a savegame was removed, false otherwise
  */
 public final SkyrimCharacterHelperSaveGame removeSaveGame(String pa_sFileName)
 {
  boolean lc_bFound = false;
  int     lc_iLen   = 0    ;
  int     lc_iI     = 0    ;
   
  SkyrimCharacterHelperSaveGame[] lc_tSaveGames = null;
  SkyrimCharacterHelperSaveGame   lc_tSaveGame = null;

  if (null != m_tSaveGames && null != pa_sFileName)
  {
   if (0 < (lc_iLen = getSaveGameAmount()))
   {
    if (null != (lc_tSaveGames = getSaveGames()))
    {
     while (false == lc_bFound && lc_iI < lc_iLen) 
     {
      if (null != (lc_tSaveGame = lc_tSaveGames[lc_iI])) 
      {
       lc_bFound = (0 == m_tCollator.compare(pa_sFileName, lc_tSaveGame.getFileName())); 
      }
      lc_iI++;
     } // while
    }
   }
  }
  
  if (true == lc_bFound)
  {
   if (true == removeSaveGame(lc_tSaveGame))
   {
    return  lc_tSaveGame;
   }
  }
  return null;
 }
 
 



 /**
  * Checks whether such s savegame (means: Same file name) exists. We can just use contains(), because
  * contains calls Object.equals(), and SkyrimCharacterHelperSaveGame overrides equals() such that
  * the filenames (and not filepathes!) are compared.
  * 
  * @param pa_tSaveGame  savegame
  * @return true, if it exists, false otherwise
  */
 public final boolean hasSaveGame(SkyrimCharacterHelperSaveGame pa_tSaveGame)
 {
  return (null != m_tSaveGames && null != pa_tSaveGame ? m_tSaveGames.contains(pa_tSaveGame) : false);
 }
 
 
 
 /**
  * Checks whether such s savegame (means: given file name) exists. We can just use contains(), because
  * contains calls Object.equals(), and SkyrimCharacterHelperSaveGame overrides equals() such that
  * the filenames (and not filepathes!) are compared.
  * 
  * @param pa_sFileName  filename
  * @return true, if it exists, false otherwise
  */
 public final SkyrimCharacterHelperSaveGame hasSaveGame(String pa_sFileName)
 {
  SkyrimCharacterHelperSaveGame[]  lc_tSaveGames = null;
  SkyrimCharacterHelperSaveGame    lc_tSaveGame  = null;
  
  int lc_iLen = 0;
  int lc_iI   = 0;
  
  boolean lc_bFound = false;
  
  if (null != (lc_tSaveGames = getSaveGames()))
  {
   if (0 < (lc_iLen = lc_tSaveGames.length))
   {
    while (false == lc_bFound && lc_iI < lc_iLen)
    {
     if (null != (lc_tSaveGame = lc_tSaveGames[lc_iI]))
     {
      lc_bFound = (0 == m_tCollator.compare(pa_sFileName, lc_tSaveGame.getFileName())); 
     }
     lc_iI++; 
    } // while
    
   }
  }   
  return (true == lc_bFound ? lc_tSaveGame : null);
 }
 
 
 
 /**
  * Comparable.compareTo
  * 
  * @param pa_tObject   object to compare to
  * @return a negative integer, zero, or a positive integer as this object is less than, equal to, or greater than the specified object.
  */
 public int compareTo(SkyrimCharacterHelperPlayer pa_tObject) 
 {
  String lc_sName = (null != pa_tObject ? pa_tObject.getName() : null);
  
  if (null == lc_sName)
  {
   return (null == getName() ? 0 : 1);
  }
  else
  {
   return (null == getName() ? -1 : m_tCollator.compare(getName(), lc_sName)); 
  }
 }

 
 
 /**
  * Returns the hashCode
  * 
  * @return  hashCode
  */ 
  @Override 
  public int hashCode() 
  {
   int lc_iHash = 7;
   lc_iHash = 17 * lc_iHash + Objects.hashCode(this.m_sName);
   return lc_iHash;
  }
 
 
 
 /**
  * equals
  * 
  * @param pa_tObject   object to compare to
  * @return a negative integer, zero, or a positive integer as this object is less than, equal to, or greater than the specified object.
  */
 @Override 
 public boolean equals(Object pa_tObject) 
 {
  SkyrimCharacterHelperPlayer lc_tPlayer = null;
  
  if (this == pa_tObject) return true;
  
  if (null != pa_tObject)
  {
   if (this.getClass() == pa_tObject.getClass())
   {
    if (null != (lc_tPlayer = (SkyrimCharacterHelperPlayer) pa_tObject))
    {   
     return (0 == m_tCollator.compare(getName(), lc_tPlayer.getName())); 
    }
   }
  }
  return false;
 }
 
 
 
 /**
  * toString()
  * 
  * @return   player name
  */
 @Override
 public String toString() {return m_sName;} 
 
 
} // eoc
