/**
 * SkyrimCharacterHelper
 * 
 * UI based tool to save and backup skyrim character save-files
 * 
 */
package skyrimcharacterhelper;



/**
 * Imports
 * 
 */
import java.util.HashMap     ;
import java.util.Collection  ;
import java.util.Iterator    ;
import java.util.Comparator  ;
import java.util.Arrays      ;
import java.util.List        ;
import java.util.ArrayList   ;
import java.text.Collator   ;    


/**
 * This class manages a collection of characters ( = a bunch of savegames)
 * 
 * @author Ulf Wagemann
 */
public class SkyrimCharacterHelperPlayerManager 
{
 private int  m_iAmountOfChars ; 
 private int  m_iAmountOfSaveFiles;
 
 private HashMap<String, SkyrimCharacterHelperPlayer> m_tCharMap ;
 private static Collator                              m_tCollator;

 
 /**
  * Constructor
  * 
  */
 public SkyrimCharacterHelperPlayerManager()
 {
  m_tCharMap           = new HashMap<>();
  m_iAmountOfChars     = 0;
  m_iAmountOfSaveFiles = 0;
  
  m_tCollator = java.text.Collator.getInstance(); 
  m_tCollator.setStrength(java.text.Collator.TERTIARY);      
 }
 
 
 
 /**
  * We do not want to eat up memory, since we store own datastructures we need to remove them from the
  * hashmap and mark then as gone for the GC.
  * 
  * dispose() should be called when this shapeset manager is not needed anymore.
  * 
  */
 public final void clear()
 {
  if (null != m_tCharMap) {m_tCharMap.clear();}
  m_iAmountOfChars = 0;
  m_iAmountOfSaveFiles = 0;
 }

 
 /**
  * Returns the amount of savegames for the given list of players
  * 
  * @param pa_tPlayers
  * @return 
  */
 public final int getSaveGameAmountForList(List<SkyrimCharacterHelperPlayer> pa_tPlayers)
 {
  int lc_iResult = 0;
  
  if (null != pa_tPlayers)
  {
   for (SkyrimCharacterHelperPlayer lc_tPlayer : pa_tPlayers)
   {
    if (null != lc_tPlayer) 
    {
     lc_iResult += lc_tPlayer.getSaveGameAmount(); 
    }
   }
  }
  return lc_iResult;
 }
 
 
 /**
  * Retrieves the sorted array of names of all valid characters 
  * 
  * @return   Names of all characters as a sorted SkyrimCharacterHelperPlayer array, null in case of error
  */
 public final SkyrimCharacterHelperPlayer[] getPlayers()
 {
  Collection<SkyrimCharacterHelperPlayer>  lc_tChars         = null;
  Iterator<SkyrimCharacterHelperPlayer>    lc_tCharsIterator = null;
  SkyrimCharacterHelperPlayer              lc_tCharacter     = null;
  SkyrimCharacterHelperPlayer[]            lc_tResult        = null;
  
  int lc_iAmount = 0;
  int lc_iCount  = 0;
  
  if (null != m_tCharMap)
  {
   if (0 < (lc_iAmount = m_iAmountOfChars))
   {
    if (null != (lc_tChars = m_tCharMap.values())) 
    {
     if (null != (lc_tCharsIterator = lc_tChars.iterator())) 
     {
      if (null != (lc_tResult = new SkyrimCharacterHelperPlayer[lc_iAmount]))  
      {
       while (true == lc_tCharsIterator.hasNext())
       {
        if (null != (lc_tCharacter = lc_tCharsIterator.next()))
        {
         lc_tResult[lc_iCount++] = lc_tCharacter;
        }
       } // while
       
       if (0 < lc_iCount)
       {
        java.util.Arrays.sort(lc_tResult);   
       }
       return lc_tResult;
      }
     }
    } 
   }
  }
  return null;
 }

 
 
 /**
  * Retrieves the character with the given name
  * 
  * @param pa_sName   Name of character
  * @return           player information, null in case of error
  */
 public final SkyrimCharacterHelperPlayer getPlayer(String pa_sName)
 {
  SkyrimCharacterHelperPlayer lc_tCharacter = null;
  
  if (null != m_tCharMap)
  {
   if (null != pa_sName)
   {
    if (false == pa_sName.isEmpty())
    {
     lc_tCharacter = m_tCharMap.get(pa_sName); 
    }
   }
  }
  return lc_tCharacter;
 }
 
 
 
 /**
  * Retrieves the savegames of the character with the given name in form of an array
  * 
  * @param pa_sName   Name of character
  * @return           savegames, null in case of error
  */
 public final SkyrimCharacterHelperSaveGame[] getSaveGames(String pa_sName)
 {
  SkyrimCharacterHelperPlayer    lc_tCharacter = null;
  SkyrimCharacterHelperSaveGame[]  lc_tSaveGames = null;
  
  if (null != (lc_tCharacter = getPlayer(pa_sName)))
  {
   if (null != (lc_tSaveGames = lc_tCharacter.getSaveGames()))
   {
    Arrays.sort(lc_tSaveGames, new Comparator<SkyrimCharacterHelperSaveGame>() 
                                 {
                                  public int compare(SkyrimCharacterHelperSaveGame o1, SkyrimCharacterHelperSaveGame o2)
                                  { 
                                   return m_tCollator.compare(o1.getFileName(), o2.getFileName()); // compare NAME and not PATH !!
                                  }
                                 });
    
    
    return lc_tSaveGames; 
   }
  }
  return null;
 }
 
 
 
 /**
  * Retrieves the savegames of the character with the given name in form of an array
  * 
  * @param pa_sName   Name of player
  * @return           list, null in case of error
  */
 public final List<SkyrimCharacterHelperSaveGame> getSaveGamesList(String pa_sName)
 {
  SkyrimCharacterHelperSaveGame[]          lc_tSaveGames = null;
  ArrayList<SkyrimCharacterHelperSaveGame> lc_tList      = null;
  
  
  if (null != (lc_tSaveGames = getSaveGames(pa_sName)))
  {
   if (null != (lc_tList = new ArrayList<>()))
   {
    lc_tList.addAll(Arrays.asList(lc_tSaveGames));
   }
  }
  return lc_tList;
 }
 
 
 
 /**
  * Retrieves the amount of characters 
  * 
  * @return   Amount of characters, 0 in case of error
  */
 public final int getAmountOfPlayers() {return m_iAmountOfChars;};

 
 
 /**
  * Retrieves the amount of savegames 
  * 
  * @return   Amount of savegames, 0 in case of error
  */
 public final int getSaveGameAmount() {return m_iAmountOfSaveFiles;}
 
  
 
 /**
  * Adds a savegame for a character to the manager
  * 
  * @param pa_tSaveGame    new character
  * @return   true, if the character was added successfully, false otherwise
  * 
  */
 public final boolean add(SkyrimCharacterHelperSaveGame pa_tSaveGame)
 {
  boolean lc_bResult   = false;
  SkyrimCharacterHelperPlayer lc_tEntry = null;
  
  if (null != m_tCharMap) 
  {
   if (null != pa_tSaveGame) 
   {
    //
    // create new entry, if none exists yet
    //
    if (null == (lc_tEntry = m_tCharMap.get(pa_tSaveGame.getPlayerName())))
    {
     if (null != (lc_tEntry = new SkyrimCharacterHelperPlayer(pa_tSaveGame)))
     {
      if (null == m_tCharMap.put(pa_tSaveGame.getPlayerName(), lc_tEntry))
      {
       m_iAmountOfChars++;
       m_iAmountOfSaveFiles++;
       lc_bResult = true;
      }
     }
    }
    //
    // otherwise add savegame to existing character entry
    //
    else
    {
     if (true == lc_tEntry.addSaveGame(pa_tSaveGame))
     {
      m_iAmountOfSaveFiles++; 
     }
     lc_bResult = true;
    }
   
   }
  }
  return lc_bResult;
 }
 


 /**
  * Creates a copy pf the given savegame with the new path and adds it to the manager. 
  * 
  * @param pa_tSaveGame    new character
  * @param pa_sDestinationFullPath    filepath to use
  * @return   true, if the character was added successfully, false otherwise
  * 
  */
 public final boolean add(SkyrimCharacterHelperSaveGame pa_tSaveGame, String pa_sDestinationFullPath)
 {
  return (add(new SkyrimCharacterHelperSaveGame(pa_tSaveGame, pa_sDestinationFullPath)));
 }
 
 
 
 /**
  * Removes the given save game from the corresponding character
  * 
  * @param pa_tSaveGame       savegame to remove
  */
 public final boolean removeSaveGame(SkyrimCharacterHelperSaveGame pa_tSaveGame)
 {
  SkyrimCharacterHelperPlayer lc_tEntry = null;
  
  if (null != m_tCharMap) 
  {
   if (null != pa_tSaveGame) 
   {
    //
    // get entry
    //
    if (null != (lc_tEntry = m_tCharMap.get(pa_tSaveGame.getPlayerName())))
    {
     m_iAmountOfSaveFiles--;
     return (lc_tEntry.removeSaveGame(pa_tSaveGame)); 
    }
   }
  }
  return false;
 }
 
 
 
  
 /**
  * Removes the given save game from the corresponding character
  * 
  * @param pa_tSaveGame       savegame to remove
  */
 public final SkyrimCharacterHelperSaveGame removeSaveGame(String pa_sFileName)
 {
  SkyrimCharacterHelperPlayer lc_tEntry = null;
  String  lc_sPlayer = null;
  
  if (null != m_tCharMap) 
  {
   if (null != (lc_sPlayer = hasSaveGame(pa_sFileName)))
   {
    //
    // get entry
    //
    if (null != (lc_tEntry = m_tCharMap.get(lc_sPlayer)))
    {
     m_iAmountOfSaveFiles--;
     return (lc_tEntry.removeSaveGame(pa_sFileName)); 
    }
   }
  }
  return null;
 }
 
 
 
 /**
  * Removes all entries for the player with the given name and the player itself
  * 
  * @param pa_sName   name of the player to remove
  */
 public final SkyrimCharacterHelperPlayer removePlayer(String pa_sName)
 {
  SkyrimCharacterHelperPlayer lc_tPlayer = null;
  
  if (null != m_tCharMap && null != pa_sName) 
  {
   if (null != (lc_tPlayer = m_tCharMap.get(pa_sName)))
   {
    m_iAmountOfChars--; 
    m_iAmountOfSaveFiles -= lc_tPlayer.getSaveGameAmount();
    return m_tCharMap.remove(pa_sName);
   }
  }
  return null;
 }
 


 /**
  * Checks whether a toon with the given name exists
  * 
  * @param pa_sName   name
  * @return  true, if such a toon exists, false otherwise
  */
 public final boolean hasPlayer(String pa_sName)
 {
  boolean lc_bResult = false;
  
  if (null != m_tCharMap && null != pa_sName)
  {
   return (null != m_tCharMap.get(pa_sName)); 
  }
  return lc_bResult;
 }
 
 
 
 
 /**
  * Checks, whether the corresponding toon has this savegame (means a savegame with the same filename)
  * 
  * @param pa_tSaveGame
  * @return  true, if such a savegame exists, false otherwise
  */
 public final boolean hasSaveGame(SkyrimCharacterHelperSaveGame pa_tSaveGame) 
 {
  boolean lc_bResult = false;
  String  lc_sName   = null ;
  
  SkyrimCharacterHelperPlayer lc_tPlayer   = null;
  
  if (null != m_tCharMap && null != pa_tSaveGame)
  {
   if (null != (lc_sName = pa_tSaveGame.getPlayerName()))  
   {
    if (null != (lc_tPlayer = m_tCharMap.get(lc_sName))) 
    {
     lc_bResult = lc_tPlayer.hasSaveGame(pa_tSaveGame);
    }
   }
  }
  return lc_bResult;
 }
 
 
 
  /**
  * Checks, whether the corresponding toon has this savegame (means a savegame with the same filename)
  * 
  * @param pa_tSaveGame
  * @return  true, if such a savegame exists, false otherwise
  */
 public final String hasSaveGame(String pa_sFileName) 
 {
  String  lc_sResult = null;
  int     lc_iAmount = 0   ;
  
  Collection<SkyrimCharacterHelperPlayer>  lc_tChars         = null;
  Iterator<SkyrimCharacterHelperPlayer>    lc_tCharsIterator = null;
  SkyrimCharacterHelperPlayer              lc_tCharacter     = null;
  
  if (null != m_tCharMap && null != pa_sFileName)
  {
   if (0 < (lc_iAmount = m_iAmountOfChars))
   {
    if (null != (lc_tChars = m_tCharMap.values())) 
    {
     if (null != (lc_tCharsIterator = lc_tChars.iterator())) 
     {
      //
      // iterate over characters until we find the one who may own this save
      //
      while (null == lc_sResult && true == lc_tCharsIterator.hasNext())
      {
       if (null != (lc_tCharacter = lc_tCharsIterator.next()))
       {
        lc_sResult = (null != lc_tCharacter.hasSaveGame(pa_sFileName) ? lc_tCharacter.getName() : null);
       }
      } // while
     } 
    }
   }
  }
  return lc_sResult;
 }
 
 
 
 /**
  * Returns the amount of savegames this player owns
  * 
  * param pa_sPlayer player name
  * @return    amount, -1 = player is not contained.
  */
 public final int getSaveGameAmount(String pa_sPlayer)
 {
  SkyrimCharacterHelperPlayer lc_tPlayer   = null;
  
  if (null != m_tCharMap && null != pa_sPlayer)
  {
   if (null != (lc_tPlayer = m_tCharMap.get(pa_sPlayer))) 
   {
    return lc_tPlayer.getSaveGameAmount();
   }
   return 0;
  }
  return -1;
 }
 
 
 
} // eoc

