/**
 * SkyrimCharacterHelper
 * 
 * UI based tool to save and backup skyrim character save-files
 * 
 */
package skyrimcharacterhelper;



/**
 * Imports
 * 
 */
import java.io.File           ;
import java.io.FileInputStream;

import java.awt.image.ColorModel    ;
import java.awt.image.WritableRaster;
import java.awt.image.BufferedImage ;
import java.util.Objects ;
import java.text.Collator;    


/**
 * Class representing the data for a single Skyrim savegame. Implements Comparable for the sorted lists :)
 * 
 * @author Ulf Wagemann
 */
public class SkyrimCharacterHelperSaveGame  implements Comparable<SkyrimCharacterHelperSaveGame>
{
 private SkyrimCharacterHelperScreenshot  m_tScreenshot      ;
 private String                           m_sPlayerName      ;
 private String                           m_sPlayerLocation  ;
 private String                           m_sPlayerRace      ;
 private int                              m_iPlayerLevel     ;
 private String                           m_sFileName        ;
 private String                           m_sFilePath        ;
 private String                           m_sGameDate        ;
 private long                             m_lFileDate        ;
 private int                              m_iScreenShotWidth ; 
 private int                              m_iScreenShotHeight; 
 private int                              m_iSaveNumber      ; 
 private int                              m_iVersion         ; 
 private static Collator                  m_tCollator        ;

 
 
 /**
  * Constructor
  * 
  */
 public SkyrimCharacterHelperSaveGame()
 {
  init();
 }
 
 
 
 /**
  * Constructor, clones the given object
  * 
  * Note: String are immutable, so they are not allocated with new here
  * 
  * @param pa_tSource 
  */
 public SkyrimCharacterHelperSaveGame(SkyrimCharacterHelperSaveGame pa_tSource, String pa_sDestinationFullPath)
 {
  init();
  
  WritableRaster lc_tRaster               = null;
  ColorModel     lc_tColorModel           = null ;
  boolean        lc_bIsAlphaPremultiplied = false;
  
  SkyrimCharacterHelperScreenshot lc_tScreenshot = null;
  
  if (this != pa_tSource && pa_tSource != null) 
  {
   m_sFileName         = pa_tSource.getFileName()      ;
   m_sPlayerName       = pa_tSource.getPlayerName()    ;
   m_sPlayerLocation   = pa_tSource.getPlayerLocation();
   m_sPlayerRace       = pa_tSource.getPlayerRace()    ;
   m_iPlayerLevel      = pa_tSource.getPlayerLevel()   ;
   m_sGameDate         = pa_tSource.getGameDate()      ;
   m_lFileDate         = pa_tSource.getFileDate()      ;
   m_iSaveNumber       = pa_tSource.getSaveNumber()    ;
   m_iVersion          = pa_tSource.getVersion()       ;
   m_sFilePath         = pa_sDestinationFullPath       ;
   
   m_tScreenshot       = null;
   m_iScreenShotWidth  = 0;
   m_iScreenShotHeight = 0;
   //
   // clone the screenshot
   //
   if (null != (lc_tScreenshot = pa_tSource.getScreenshot()))
   {
    if (null != lc_tScreenshot.getImage())
    {
     if (null != (m_tScreenshot = new SkyrimCharacterHelperScreenshot()))
     {
      m_iScreenShotWidth  = pa_tSource.getScreenShotWidth() ;
      m_iScreenShotHeight = pa_tSource.getScreenShotHeight();
     
      lc_tColorModel           = lc_tScreenshot.getImage().getColorModel();
      lc_bIsAlphaPremultiplied = lc_tColorModel.isAlphaPremultiplied();
      lc_tRaster               = lc_tScreenshot.getImage().copyData(null);
    
      m_tScreenshot.init(m_iScreenShotWidth, m_iScreenShotHeight, new BufferedImage(lc_tColorModel, lc_tRaster, lc_bIsAlphaPremultiplied, null));
     }
    }
   }
  }
 }
 
 
 
 /**
  * Private init method
  * 
  */
 private void init()
 {
  m_tScreenshot     = null;
  m_sFileName       = null;
  m_sPlayerName     = null;
  m_sPlayerLocation = null;
  m_sGameDate       = null;
  m_sPlayerRace     = null;
  m_sFilePath       = null;
  
  m_lFileDate         = 0;
  m_iPlayerLevel      = 0;
  m_iScreenShotWidth  = 0;
  m_iScreenShotHeight = 0;
  m_iSaveNumber       = 0;
  m_iVersion          = 0;
  
  m_tCollator = java.text.Collator.getInstance(); 
  m_tCollator.setStrength(java.text.Collator.TERTIARY);  
 }
 
 
 
 /**
  * Returns the screenshot width
  * 
  * @return   screenshot width, may be 0
  */
 public final int getScreenShotWidth() {return m_iScreenShotWidth;}

 
 
 /**
  * Returns the screenshot height
  * 
  * @return   screenshot height, may be 0
  */
 public final int getScreenShotHeight() {return m_iScreenShotHeight;}


 
 /**
  * Returns the save number
  * 
  * @return   save number, may be 0
  */
 public final int getSaveNumber() {return m_iSaveNumber;}

 
 
 /**
  * Returns the version
  * 
  * @return   version, may be 0
  */
 public final int getVersion() {return m_iVersion;}

 
 
 /**
  * Returns the player name to whom this save game belongs to
  * 
  * @return  player name, may be null
  */
 public final String getPlayerName() {return m_sPlayerName;}

 

 /**
  * Returns the player location
  * 
  * @return  player location, may be null
  */
 public final String getPlayerLocation() {return m_sPlayerLocation;}

 
 
 /**
  * Returns the player race
  * 
  * @return  player race, may be null
  */
 public final String getPlayerRace() {return m_sPlayerRace;}

 
 
 /**
  * Returns the player level
  * 
  * @return  player level, may be 0
  */
 public final int getPlayerLevel() {return m_iPlayerLevel;}

 
 
 /**
  * Returns the file name
  * 
  * @return  file name, may be null
  */
 public final String getFileName() {return m_sFileName;}

 
 

 /**
  * Returns the file path
  * 
  * @return  file path, may be null
  */
 public final String getFilePath() {return m_sFilePath;}

 
 

 /**
  * Returns the game date
  * 
  * @return  game date
  */
 public final long getFileDate() {return m_lFileDate;}
 
 
 
 /**
  * Returns the game date
  * 
  * @return  game date, may be null
  */
 public final String getGameDate() {return m_sGameDate;}

 
 
  /**
  * Returns the screenshot
  * 
  * @return screenshot, may be null
  */
 public final SkyrimCharacterHelperScreenshot getScreenshot() {return m_tScreenshot;}



 /**
  * Converts a byte array to an int. So the byte array b1b2b3b4 must be
  * interpredted as b4b3b2b1 due to UInt32 being stored as little endian... Waaaaaaaah!
  * 
  * @param pa_tBytes    bytes to convert
  * @return             int value, -1 in case of error
  */
 private int getInt(byte[] pa_tBytes)
 {
  int lc_iValue = 0;
  
  if (4 == pa_tBytes.length)
  {
   lc_iValue += (0xFF & pa_tBytes[3]) << 24;
   lc_iValue += (0xFF & pa_tBytes[2]) << 16;
   lc_iValue += (0xFF & pa_tBytes[1]) << 8;
   lc_iValue += (0xFF & pa_tBytes[0]);
  }
  if (2 == pa_tBytes.length)
  {
   lc_iValue += (0xFF & pa_tBytes[1]) << 8;
   lc_iValue += (0xFF & pa_tBytes[0]);
  }
  return lc_iValue;
 }



 
 
 /**
  * Reads-in the savegame
  * 
  * @param pa_sFileName File name
  * @param SkyrimCharacterHelperProgressNotifier  notifier for updateing the UI
  * @param pa_bReadScreenShotData true = read screenshot data, false = don't
  */
 public final boolean read(String pa_sFileName, SkyrimCharacterHelperProgressNotifier pa_tNotifier, boolean pa_bReadScreenShotData)
 {
  boolean          lc_bResult      = false;
  File             lc_tFile        = null;
  FileInputStream  lc_tInputStream = null;    

  int lc_iPos = -1;
  
  if (null != (lc_tFile = new File(pa_sFileName)))
  {
   if (true == lc_tFile.exists() && true == lc_tFile.isFile()) 
   {
    m_lFileDate = lc_tFile.lastModified();
    
    try 
    {
     if (null != (lc_tInputStream = new FileInputStream(lc_tFile)))
     {
      m_sFilePath  = pa_sFileName;
      m_sFileName  = pa_sFileName;
      
      if (-1 != (lc_iPos = pa_sFileName.lastIndexOf(System.getProperty("file.separator"))))
      {
       m_sFileName  = pa_sFileName.substring(1 + lc_iPos);
      }
      if (true == readMagic(lc_tInputStream))
      {
       if (true == readHeader(lc_tInputStream))
       {
        lc_bResult = (true == pa_bReadScreenShotData ? readScreenShotData(lc_tInputStream, pa_tNotifier) : true);
       }
      }
     }
    }
    catch (Exception lc_tException) 
    {
     lc_bResult = false;
    }
   }
  }
  
  //
  // for safety, we close at the end of th method
  //
  if (null != lc_tInputStream) {try {lc_tInputStream.close();} catch (Exception lc_tException) {}}
  
  return lc_bResult;
 }
 
 
 
 /**
  * Reads the save game magic
  * 
  * @param pa_tInputStream   input stream
  * @return  true, if the mafic is fine, false otherwise
  */
 private boolean readMagic(FileInputStream  pa_tInputStream)
 {
  byte[] lc_tMagic     = new byte[SkyrimCharacterHelperConstants.SKH_FILE_TESV_SAVEGAME_MAGIC_LENGTH];
  String lc_sResult    = null        ;
 
  if (null != pa_tInputStream)
  {
   try
   {
    if (SkyrimCharacterHelperConstants.SKH_FILE_TESV_SAVEGAME_MAGIC_LENGTH == pa_tInputStream.read(lc_tMagic, 0, SkyrimCharacterHelperConstants.SKH_FILE_TESV_SAVEGAME_MAGIC_LENGTH))
    {
     if (null != (lc_sResult = new String(lc_tMagic)))     
     {
      return lc_sResult.equalsIgnoreCase(SkyrimCharacterHelperConstants.SKH_FILE_TESV_SAVEGAME_MAGIC); 
     }
    }
   }
   catch (Exception lc_tException) 
   {}
  }
  return false;
 }
 
 
 
 /**
  * Reads the save game header
  * 
  * @param pa_tInputStream   input stream
  * @return  true, if the header is fine, false otherwise
  */
 private boolean readHeader(FileInputStream  pa_tInputStream)
 {
  int     lc_iHeaderSize = 0   ; 
  boolean lc_bResult = false;
  
  if (null != pa_tInputStream)
  {
   try
   {
    //
    // header size
    //
    if (0 < (lc_iHeaderSize = readUInt32(pa_tInputStream)))
    {
     //
     // good stuff
     //
     m_iSaveNumber     = readUInt32 (pa_tInputStream);
     m_iVersion        = readUInt32 (pa_tInputStream);
     m_sPlayerName     = readWString(pa_tInputStream);
     m_iPlayerLevel    = readUInt32 (pa_tInputStream);
     m_sPlayerLocation = readWString(pa_tInputStream);
     m_sGameDate       = readWString(pa_tInputStream);
     m_sPlayerRace     = readWString(pa_tInputStream);

     //
     // skip unknown UInt16, Float32, Float32, filetime structure = 2 + 4 + 4 + 8 bytes
     //
     readSkip(pa_tInputStream, 18); // advance 18 bytes
     
     //
     // remainder
     //
     m_iScreenShotWidth  = readUInt32  (pa_tInputStream);
     m_iScreenShotHeight = readUInt32  (pa_tInputStream);
     lc_bResult = true;
    }
   }
   catch (Exception lc_tException) {}
  }
  return lc_bResult;
 }

 
 
 /**
  * Reads in the screenshot data
  * 
  * Remember, that Java uses 4 bytes to encode a color, since an alpha byte is added. So
  * we consider an alpha value of 255 for each pixel when computig the color.
  * 
  * @param pa_tInputStream   input stream
  * @return  true, if the screenshot was fine, false otherwise
  */
 private boolean readScreenShotData(FileInputStream  pa_tInputStream, SkyrimCharacterHelperProgressNotifier pa_tNotifier)
 {
  int lc_iColumns = 0;
  int lc_iRows    = 0;  
  
  int lc_iRed      = 0;
  int lc_iGreen    = 0;
  int lc_iBlue     = 0;
  int lc_iValue    = 0;
  
  int lc_iCount    = 0;
  int lc_iAmount   = 0;
  
  int lc_iProgressStep    = 0;
  int lc_iProgressCount   = 0;
  int lc_iProgressDisplay = 0;
  
  boolean lc_bResult = false;
  
  if (null != (m_tScreenshot = new SkyrimCharacterHelperScreenshot()))
  {
   m_tScreenshot.init(m_iScreenShotWidth, m_iScreenShotHeight); 

   if (null != pa_tNotifier) pa_tNotifier.notifyProgress(lc_iProgressDisplay);

   try
   {
    //
    // compute how much input causes an increase of the current progress display
    //
    lc_iAmount       = m_tScreenshot.getHeight() * m_tScreenshot.getWidth();
    lc_iProgressStep = (lc_iAmount/100);
   
    //
    // get the picture
    //
    for (lc_iRows = 0; lc_iRows < m_tScreenshot.getHeight(); lc_iRows++)
    {
     for (lc_iColumns = 0; lc_iColumns < m_tScreenshot.getWidth(); lc_iColumns++)
     {
      lc_iRed   = readUInt8(pa_tInputStream);
      lc_iGreen = readUInt8(pa_tInputStream);
      lc_iBlue  = readUInt8(pa_tInputStream);
      
      if ((0 <= lc_iRed && 255 >= lc_iRed) && (0 <= lc_iGreen && 255 >= lc_iGreen) && (0 <= lc_iBlue && 255 >= lc_iBlue))
      {
       lc_iValue = ((255 & 0xFF) << 24) | ((lc_iRed & 0xFF) << 16) | ((lc_iGreen & 0xFF) << 8)  | ((lc_iBlue & 0xFF) << 0); // the last term is pretty useless, but details what is computed
      }
      else
      {
       lc_iValue = 0;
      }
      m_tScreenshot.setRGB(lc_iColumns, lc_iRows, lc_iValue);
     
      //
      // handle progress and progress display
      //
      lc_iCount++;
      lc_iProgressCount++;
      
      if (lc_iProgressCount >= lc_iProgressStep)
      {
       lc_iProgressCount = 0;
       if (null != pa_tNotifier)
       {
        pa_tNotifier.notifyProgress(++lc_iProgressDisplay); 
       }
      }
     } // for
    } // for
    lc_bResult = (lc_iCount == lc_iAmount);
   }
   catch (Exception lc_tException) {}
  }
  return lc_bResult;
 }

 
 
 /**
  * Reads an UInt32 in little endian enconding
  * 
  * @param pa_tInputStream   input stream
  * @return   int value
  */
 private int readUInt32(FileInputStream  pa_tInputStream)
 {
  byte[] lc_tBytes = new byte[4];  
   
  try
  {
   if (4 == pa_tInputStream.read(lc_tBytes)) 
   {
    return getInt(lc_tBytes); 
   }
  }
  catch (Exception lc_tException) 
  {}
  return -1;
 }
 
 
 
 /**
  * Reads an UInt16 in little endian enconding
  * 
  * @param pa_tInputStream   input stream
  * @return   int value
  */
 private int readUInt16(FileInputStream  pa_tInputStream)
 {
  byte[] lc_tBytes = new byte[2];  
   
  try
  {
   if (2 == pa_tInputStream.read(lc_tBytes)) 
   {
    return getInt(lc_tBytes); 
   }
  }
  catch (Exception lc_tException) 
  {}
  return -1;
 }
  
  
  
  /**
  * Reads an UInt8. Of course, damn little endian again...
  * 
  * @param pa_tInputStream   input stream
  * @return   int value
  */
 private int readUInt8(FileInputStream  pa_tInputStream)
 {
  try
  {
   return (0xFF & pa_tInputStream.read()); // bleh
  }
  catch (Exception lc_tException) 
  {}
  return -1;
 }
 
 
 
 /**
  * Reads a WString
  * 
  * @param pa_tInputStream   input stream
  * @return   int value
  */
 private String readWString(FileInputStream  pa_tInputStream)
 {
  byte[]       lc_tBytesSize   = new byte[2];  
  byte[]       lc_tBytesString = null       ;
  int          lc_iSize        =   0        ;
  StringBuffer lc_tResult      = new StringBuffer();
  
  try
  {
   if ((null != lc_tResult) && (2 == pa_tInputStream.read(lc_tBytesSize)))
   {
    if (0 < (lc_iSize = getInt(lc_tBytesSize)))
    {
     if (null != (lc_tBytesString = new byte[lc_iSize])) 
     {
      if (lc_iSize == pa_tInputStream.read(lc_tBytesString)) 
      {
       //
       // convert string char by char to take unicode into account
       //
       for (byte lc_tCharByte : lc_tBytesString) 
       {
        lc_tResult.append((Character.toChars(0xFF & lc_tCharByte))); // hey little endian, finally we meet again...
       }
       return lc_tResult.toString(); 
      }
     }
    }
   }
  }
  catch (Exception lc_tException)
  {}
  
  return null;
 }
  

 
 
 /**
  * Reads and so skips the given amount of bytes
  * 
  * @param pa_tInputStream    input stream
  * @param pa_iAmount         bytes to skip
  */
 private void readSkip(FileInputStream  pa_tInputStream, int pa_iAmount)
 {
  byte[] lc_tSkipBytes  = new byte[pa_iAmount];  
   
  if (null != lc_tSkipBytes)
  {
   try
   {   
    pa_tInputStream.read(lc_tSkipBytes); 
   }
   catch (Exception lc_tException)
   {}
  }
 }
 

 
 
 /**
  * Comparable.compareTo
  * 
  * @param pa_tObject   object to compare to
  * @return a negative integer, zero, or a positive integer as this object is less than, equal to, or greater than the specified object.
  */
 public int compareTo(SkyrimCharacterHelperSaveGame pa_tObject) 
 {
  String lc_sFileName = null;
  
  if (null == pa_tObject) return 1;
  
  lc_sFileName = pa_tObject.getFileName();
  
  if (null == lc_sFileName)
  {
   return (null == getFileName() ? 0 : 1);
  }
  else
  {
   return (null == getFileName() ? -1 : m_tCollator.compare(getFileName(), lc_sFileName)); 
  }
 }

 
 
 /**
  * Returns the hashCode
  * 
  * @return   hashCode
  */
 @Override
  public int hashCode() 
  {
   int lc_iHash = 3;
   lc_iHash = 11 * lc_iHash + Objects.hashCode(this.m_sFileName);
   return lc_iHash;
  }
 
  
 
 /**
  * equals: two savegames are equal, if their filenames (without path!!!) are equal.
  * 
  * @param pa_tObject   object to compare to
  * @return a negative integer, zero, or a positive integer as this object is less than, equal to, or greater than the specified object.
  */
 @Override
 public boolean equals(Object pa_tObject) 
 {
  SkyrimCharacterHelperSaveGame lc_tSaveGame = null;
  
  if (this == pa_tObject) return true;
  
  if (null != pa_tObject)
  {
   if (this.getClass() == pa_tObject.getClass())
   {
    if (null != (lc_tSaveGame = (SkyrimCharacterHelperSaveGame) pa_tObject))
    {   
     return (0 == m_tCollator.compare(getFileName(), lc_tSaveGame.getFileName()));
    }
   }
  }
  return false;
 }
 
 
 
 /**
  * toString()
  * 
  * @return   file path
  */
 @Override
 public String toString() {return m_sFileName;}
} // eoc