/**
 * SkyrimCharacterHelper
 * 
 * UI based tool to save and backup skyrim character save-files
 * 
 */
package skyrimcharacterhelper;

import java.awt.Color;



/**
 * UI
 * 
 * @author Ulf Wagemann
 */
public class SkyrimCharacterHelperView extends javax.swing.JFrame 
                                       implements java.awt.event.MouseListener,
                                                  javax.swing.event.DocumentListener

{
 private SkyrimCharacterHelperGuiConnector    m_tGuiConnector;
 private SkyrimCharacterHelperScreenshotPanel m_tActivePaintPanel;
 private SkyrimCharacterHelperScreenshotPanel m_tBackupPaintPanel;
 
 private java.awt.Color                       m_tLabelOnSkyrimActiveBg;
 private java.awt.Color                       m_tLabelOnSkyrimNormalBg;
 private java.awt.Color                       m_tTextfieldActivePathNormalFg;
 private java.awt.Color                       m_tTextfieldBackupPathNormalFg;
 
 private javax.swing.table.TableRowSorter<javax.swing.table.TableModel> m_tRowSorterActiveSaveGames;
 private javax.swing.table.TableRowSorter<javax.swing.table.TableModel> m_tRowSorterBackupSaveGames;

 private boolean  m_bIsWindowsOs  ; 
 private boolean  m_bLocked       ;
 private boolean  m_bLaunched     ;
 private boolean  m_bActiveScanned;
 private boolean  m_bBackupScanned;
 
 private String[] m_tTableColumnHeaders;
 private String[] m_tMonitoringDisplays;
 
 private java.util.ResourceBundle  m_tResources;

 private java.util.List<SkyrimCharacterHelperSaveGame> m_tStoredActiveSaveGameSelection;
 private java.util.List<SkyrimCharacterHelperSaveGame> m_tStoredBackupSaveGameSelection;
 
 private java.util.List<SkyrimCharacterHelperPlayer> m_tStoredActivePlayerSelection;
 private java.util.List<SkyrimCharacterHelperPlayer> m_tStoredBackupPlayerSelection;

 
 
/**
 * Inner class for comparing savegames doing comparisions bound to its initial model index!
 * 
 * @author Ulf Wagemann
 * 
 * ---------------------------------------------------------------------------------------------------------------------
 */
 private class SaveGameComparator implements java.util.Comparator<SkyrimCharacterHelperSaveGame>
 {
  private java.text.Collator m_tCollator;
  private int                m_iColumn  ;
  
  
  
  /**
   * Constructor
   * 
   * @param pa_iColumn   column model index
   */
  public SaveGameComparator(int pa_iColumn)
  {
   m_tCollator = java.text.Collator.getInstance(); 
   m_tCollator.setStrength(java.text.Collator.TERTIARY);
   
   m_iColumn = pa_iColumn;
  }

  
  /**
   * Comparator.compare
   * 
   * Hardened, because I caught skyrim once writing quicksaves and autosaves without a proper location name....
   * 
   * @param o1  object to compare
   * @param o2  object to compare with
   * @reutn -1, 0 or 1
   */
  public int compare( SkyrimCharacterHelperSaveGame o1, SkyrimCharacterHelperSaveGame o2 ) 
  {
   if (null == o1 && null == o2) return  0;
   if (null == o1              ) return -1;
   if (null == o2              ) return  1;
 
   try
   {     
    switch(m_iColumn)
    {
     case 0:
             if (o1.getPlayerLevel()  < o2.getPlayerLevel()) return -1;
             if (o1.getPlayerLevel() == o2.getPlayerLevel()) return  0;
             return 1;
       
     case 1: if (null == o1.getPlayerLocation()) return -1;
             if (null == o2.getPlayerLocation()) return  1;
             return (null != m_tCollator ? m_tCollator.compare(o1.getPlayerLocation(), o2.getPlayerLocation()) : 0);
       
     case 2:
             if (o1.getFileDate()  < o2.getFileDate()) return -1;
             if (o1.getFileDate() == o2.getFileDate()) return  0;
             return 1; 
      
     case 3:  
            return (null != m_tCollator ? m_tCollator.compare(o1.getFileName(), o2.getFileName()) : 0);
      
    }
    return (null != m_tCollator ? m_tCollator.compare(o1.toString(), o2.toString()) : 0);
   }
   catch (Exception lc_tException)
   {
    return 0; 
   }
  }
 }// eoc ---------------------------------------------------------------------------------------------------------------
  
 
 
 /**
  * Constructor. Does a lot of stuff, is very ugly and lenghty.
  * 
  * 
  * @param pa_tGuiConnector   gui connector object
  * @param pa_tListener       window listener
  */
 public SkyrimCharacterHelperView(SkyrimCharacterHelperGuiConnector pa_tGuiConnector  , 
                                  java.awt.event.WindowListener     pa_tWindowListener, 
                                  String[] pa_sComboBoxFileModeValues  , 
                                  String[] pa_sComboBoxYesNo           ,
                                  String[] pa_sComboBoxLaunchMode      ,
                                  String[] pa_sComboBoxImageMode       ,
                                  String[] pa_sComboBoxInitMode        ,
                                  String[] pa_sMonitoringDisplays      ,
                                  String[] pa_tTableHeaders            ,
                                  String   pa_sHeaderArrangementActive ,
                                  String   pa_sHeaderArrangementBackup ,
                                  String   pa_sSortOrderActive         ,
                                  String   pa_sSortOrderBackup         ) 
 {
  m_tResources = java.util.ResourceBundle.getBundle(SkyrimCharacterHelperConstants.SKH_FILE_VIEW_RESOURCES);
  m_tTableColumnHeaders = pa_tTableHeaders      ;
  m_tMonitoringDisplays = pa_sMonitoringDisplays;
  
  m_bIsWindowsOs   = false;
  m_bLocked        = false;  
  m_bLaunched      = false;
  m_bBackupScanned = false;
  m_bActiveScanned = false;
  
  m_tStoredActiveSaveGameSelection = null;
  m_tStoredBackupSaveGameSelection = null;
  
  m_tStoredActivePlayerSelection = null;
  m_tStoredBackupPlayerSelection = null;
  
  m_tRowSorterActiveSaveGames = null;
  m_tRowSorterBackupSaveGames = null;
  
  //
  // workarounds for UI annoyances
  //
  pimpMyUi();
  
  //
  // init components
  //
  initComponents();
  
  //
  // setup killer savegame tables :-)
  //
  pimpMyTables(pa_sHeaderArrangementActive, pa_sHeaderArrangementBackup, pa_sSortOrderActive, pa_sSortOrderBackup);
  
  //
  // prepare lists
  //
  pimpMyLists();
  
  // 
  // add paint panels
  //
  pimpMyPaintPanels();
  
  //
  // add localized strings to combobox
  //
  pimpMyComboBoxes(pa_sComboBoxFileModeValues, pa_sComboBoxYesNo, pa_sComboBoxLaunchMode, pa_sComboBoxImageMode, pa_sComboBoxInitMode);
  
  //
  // add mouselisteners to jtables and their column headers, because we only are interested in getting events when a click occured and not when the list model changed
  //
  pimpMyListeners();
  
  //
  // remove arrow keys, we just want mouse-based list-selection
  //
  unpimpMyKeys();
    
  //
  // add event handlers to text fields
  //
  jTextFieldPrefix.setDocument(new SkyrimCharacterHelperTextDocument());
  jTextFieldPrefix.getDocument().addDocumentListener((javax.swing.event.DocumentListener) this);
  
  //
  // determine some colors
  //
  m_tLabelOnSkyrimNormalBg = jLabelOnSkyrim.getBackground();
  if (null != m_tResources) m_tLabelOnSkyrimActiveBg = new java.awt.Color(Integer.valueOf(m_tResources.getString(SkyrimCharacterHelperConstants.SKH_LABEL_ON_SKYRIM_ACTIVE_BGCOLOR), 16));
  else                      m_tLabelOnSkyrimActiveBg = java.awt.Color.red;
  
  m_tTextfieldActivePathNormalFg = jTextFieldActivePath.getForeground();
  m_tTextfieldBackupPathNormalFg = jTextFieldBackupPath.getForeground();
          
  //
  // add listener to window
  //
  m_tGuiConnector = pa_tGuiConnector;
  addWindowListener(pa_tWindowListener);
  
  //
  // finally, determine if we're runing on windows, since the launch buttons are only available then..
  //
  m_bIsWindowsOs = checkForWindows();
  
 }


 
/**
 * work-arounds for following bugs:
 * 
 * comboboxes with listcellrenderers paint the main button in wrong colors (namely in list colors...)
 * cancel button text in jfilechooser cannot be set
 * progress bar does not offer support for setting colors
 * 
 *  FUGLY SHIT! especially some of the stuff is years old....
 */ 
 private void pimpMyUi()
 {
  if (null != m_tResources)
  {
   try
   {
    javax.swing.UIManager.put("ProgressBar.background"         , new java.awt.Color(Integer.valueOf((m_tResources.getString(SkyrimCharacterHelperConstants.SKH_FUGLY_BACKGROUND         )), 16))); //colour of the background
    javax.swing.UIManager.put("ProgressBar.foreground"         , new java.awt.Color(Integer.valueOf((m_tResources.getString(SkyrimCharacterHelperConstants.SKH_FUGLY_FOREGROUND         )), 16))); //colour of progress bar
    javax.swing.UIManager.put("ProgressBar.selectionBackground", new java.awt.Color(Integer.valueOf((m_tResources.getString(SkyrimCharacterHelperConstants.SKH_FUGLY_SELECTED_BACKGROUND)), 16))); //colour of percentage counter on black background
    javax.swing.UIManager.put("ProgressBar.selectionForeground", new java.awt.Color(Integer.valueOf((m_tResources.getString(SkyrimCharacterHelperConstants.SKH_FUGLY_SELECTED_FOREGROUND)), 16))); //colour of precentage counter on red background  

    javax.swing.UIManager.put("ComboBox.selectionBackground", new java.awt.Color(Integer.valueOf((m_tResources.getString(SkyrimCharacterHelperConstants.SKH_COMBO_SELECTED_BACKGROUND)), 16))); //colour of percentage counter on black background
    javax.swing.UIManager.put("ComboBox.selectionForeground", new java.awt.Color(Integer.valueOf((m_tResources.getString(SkyrimCharacterHelperConstants.SKH_COMBO_SELECTED_FOREGROUND)), 16))); //colour of precentage counter on red background

    javax.swing.UIManager.put("FileChooser.cancelButtonText", m_tResources.getString(SkyrimCharacterHelperConstants.SKH_CHOOSER_CANCEL_TEXT));   
   }
   catch (Exception lc_tException) {}
  }   
 }
 
 
 
 /**
  * Removes cursor keys from lists and tables
  * 
  */
 private void unpimpMyKeys()
 {
  removeArrowKeysFromTable(jTableActiveSaveGames);
  removeArrowKeysFromTable(jTableBackupSaveGames);
  removeArrowKeysFromList(jListActiveCharacters);
  removeArrowKeysFromList(jListBackupCharacters);   
 }
 
 
 
 /**
  * add mouselisteners to jtables and their column headers, because we only are interested in getting events when a click occured and not when the list model changed
  * 
  */
 private void pimpMyListeners()
 {
  jTableActiveSaveGames.addMouseListener((java.awt.event.MouseListener) (this));
  jTableBackupSaveGames.addMouseListener((java.awt.event.MouseListener) (this));
  
  jTableActiveSaveGames.getTableHeader().addMouseListener((java.awt.event.MouseListener) (this));
  jTableBackupSaveGames.getTableHeader().addMouseListener((java.awt.event.MouseListener) (this));

  //
  // add mouselisteners to jlists, because we only are interested in getting events when a click occured and not when the list model changed
  //
  jListActiveCharacters.addMouseListener((java.awt.event.MouseListener) (this));
  jListBackupCharacters.addMouseListener((java.awt.event.MouseListener) (this));
 }
 
 
 
 /**
  * Creates and adds paint panelsfor screenshot thumbnails
  * 
  */
 private void pimpMyPaintPanels()
 {
  if (null != (m_tActivePaintPanel = new SkyrimCharacterHelperScreenshotPanel(jPanelActiveCharacterData, m_tResources)))
  {
   jPanelActiveCharacterData.setLayout(new java.awt.GridLayout(1,1)); 
   jPanelActiveCharacterData.add(m_tActivePaintPanel); 
  }
  if (null != (m_tBackupPaintPanel = new SkyrimCharacterHelperScreenshotPanel(jPanelBackupCharacterData, m_tResources)))
  {
   jPanelBackupCharacterData.setLayout(new java.awt.GridLayout(1,1)); 
   jPanelBackupCharacterData.add(m_tBackupPaintPanel); 
  }   
 }
 
 
 
 /**
  * Sets custom stuff to lists
  * 
  */
 private void pimpMyLists()
 {
  jListActiveCharacters.setCellRenderer(new SkyrimCharacterHelperListCellRenderer()); 
  jListActiveCharacters.setModel(new SkyrimCharacterHelperListModel());

  jListBackupCharacters.setCellRenderer(new SkyrimCharacterHelperListCellRenderer()); 
  jListBackupCharacters.setModel(new SkyrimCharacterHelperListModel());  
 }
 
 
 
/**
 * add localized strings to comboboxes
 * 
 * @param pa_sComboBoxModeValues
 * @param pa_sComboBoxYesNo 
 * @param pa_sComboBoxLaunchMode 
 */
 private void pimpMyComboBoxes(String[] pa_sComboBoxFileModeValues, 
                               String[] pa_sComboBoxYesNo     , 
                               String[] pa_sComboBoxLaunchMode, 
                               String[] pa_sComboBoxImageMode ,
                               String[] pa_sComboBoxInitMode  )
 {
  if (null != pa_sComboBoxLaunchMode)
  {
   for (String lc_sValue : pa_sComboBoxLaunchMode)
   {
    jComboBoxMinimize.addItem(lc_sValue);
   }
  }
  if (null != pa_sComboBoxFileModeValues)
  {
   for (String lc_sValue : pa_sComboBoxFileModeValues)
   {
    jComboBoxMode.addItem(lc_sValue);
   }
  }
  if (null != pa_sComboBoxImageMode)
  {
   for (String lc_sValue : pa_sComboBoxImageMode)
   {
    jComboBoxImageMode.addItem(lc_sValue);
   }
  }
  if (null != pa_sComboBoxInitMode)
  {
   for (String lc_sValue : pa_sComboBoxInitMode)
   {
    jComboBoxInitMode.addItem(lc_sValue);
   }
  }
  if (null != pa_sComboBoxYesNo)
  {
   for (String lc_sValue : pa_sComboBoxYesNo)
   {
    jComboBoxConfirmation.addItem(lc_sValue);
    jComboBoxOverwrite.addItem   (lc_sValue);
    jComboBoxAutoSaves.addItem   (lc_sValue);
    jComboBoxQuickSaves.addItem  (lc_sValue);
   }
  }   
  jComboBoxMode.setSelectedIndex        (0); // pre-select copy
  jComboBoxAutoSaves.setSelectedIndex   (0); // pre-select copy autosves
  jComboBoxQuickSaves.setSelectedIndex  (0); // pre-select copy quicksaves
  jComboBoxConfirmation.setSelectedIndex(0); // pre-select confirmation
  jComboBoxOverwrite.setSelectedIndex   (0); // pre-select confirmation
  jComboBoxImageMode.setSelectedIndex   (0); // pre-select show images
  jComboBoxMinimize.setSelectedIndex    (1); // pre-select not minimizing
  jComboBoxInitMode.setSelectedIndex    (0); // pre-select scan both
 }
 
 
 
 /**
  * Sets up tables. Pretty intrinsic stuff.
  * 
  * @param pa_sHeaderArrangementActive
  * @param pa_sHeaderArrangementBackup
  * @param pa_sSortOrderActive
  * @param pa_sSortOrderBackup 
  */
 private void pimpMyTables(String pa_sHeaderArrangementActive, String pa_sHeaderArrangementBackup, String pa_sSortOrderActive, String pa_sSortOrderBackup)
 {
  SkyrimCharacterHelperTableModel  lc_tTableModel1 = null; 
  SkyrimCharacterHelperTableModel  lc_tTableModel2 = null; 
  
  javax.swing.table.TableColumnModel lc_tColumnModel = null;
  javax.swing.table.TableColumn      lc_tColumn      = null;
  
  String lc_sHeaderArrangementActiveSaveGames = checkHeaderArrangement(pa_sHeaderArrangementActive);
  String lc_sHeaderArrangementBackupSaveGames = checkHeaderArrangement(pa_sHeaderArrangementBackup);
          
  String lc_sSortOrderActive = checkSortOrder(pa_sSortOrderActive);
  String lc_sSortOrderBackup = checkSortOrder(pa_sSortOrderBackup);

  int   lc_iCount     = 0;
  int   lc_iNewColumn = 0;
  int[] lc_iHeaderActive = new int[4];
  int[] lc_iHeaderBackup = new int[4];
     
  lc_iHeaderActive[0] = Character.digit(lc_sHeaderArrangementActiveSaveGames.charAt(0),10);
  lc_iHeaderActive[1] = Character.digit(lc_sHeaderArrangementActiveSaveGames.charAt(1),10);
  lc_iHeaderActive[2] = Character.digit(lc_sHeaderArrangementActiveSaveGames.charAt(2),10);
  lc_iHeaderActive[3] = Character.digit(lc_sHeaderArrangementActiveSaveGames.charAt(3),10);
  
  if (null != (lc_tTableModel1 = new SkyrimCharacterHelperTableModel(m_tTableColumnHeaders)))
  {
   jTableActiveSaveGames.setModel(lc_tTableModel1);
   jTableActiveSaveGames.createDefaultColumnsFromModel();
   jTableActiveSaveGames.setAutoCreateRowSorter(false);
   jTableActiveSaveGames.setDefaultRenderer(SkyrimCharacterHelperSaveGame.class, new SkyrimCharacterHelperTableCellRenderer()); 

   if (null != (m_tRowSorterActiveSaveGames = new javax.swing.table.TableRowSorter<>(jTableActiveSaveGames.getModel())))
   {
    if (null != (lc_tColumnModel = jTableActiveSaveGames.getColumnModel()))
    {
     for (lc_iCount = 0; lc_iCount < SkyrimCharacterHelperConstants.SKH_TABLE_COLUMNS; lc_iCount++)
     {
      if (null != (lc_tColumn = lc_tColumnModel.getColumn(lc_iCount)))
      {
       lc_iNewColumn = lc_iHeaderActive[lc_iCount];
       
       lc_tColumn.setModelIndex (lc_iNewColumn); 
       lc_tColumn.setHeaderValue(m_tTableColumnHeaders[lc_iNewColumn]); 
       m_tRowSorterActiveSaveGames.setComparator(lc_iNewColumn, new SaveGameComparator(lc_iNewColumn)); // the comparator only cares about this column afterwards!
      }
     }
    }    
    m_tRowSorterActiveSaveGames.setSortsOnUpdates(true);
    jTableActiveSaveGames.getTableHeader().resizeAndRepaint(); 
    jTableActiveSaveGames.setRowSorter(m_tRowSorterActiveSaveGames);
    setSortOrder(m_tRowSorterActiveSaveGames, lc_sSortOrderActive, jTableActiveSaveGames);
   }
  }
  

  //
  // prepare backup table
  //
  lc_iHeaderBackup[0] = Character.digit(lc_sHeaderArrangementBackupSaveGames.charAt(0),10);
  lc_iHeaderBackup[1] = Character.digit(lc_sHeaderArrangementBackupSaveGames.charAt(1),10);
  lc_iHeaderBackup[2] = Character.digit(lc_sHeaderArrangementBackupSaveGames.charAt(2),10);
  lc_iHeaderBackup[3] = Character.digit(lc_sHeaderArrangementBackupSaveGames.charAt(3),10);

  if (null != (lc_tTableModel2 = new SkyrimCharacterHelperTableModel(m_tTableColumnHeaders)))
  {
   jTableBackupSaveGames.setModel(lc_tTableModel2);
   jTableBackupSaveGames.createDefaultColumnsFromModel();
   jTableBackupSaveGames.setAutoCreateRowSorter(false);
   jTableBackupSaveGames.setDefaultRenderer(SkyrimCharacterHelperSaveGame.class, new SkyrimCharacterHelperTableCellRenderer()); 
 
   if (null != (m_tRowSorterBackupSaveGames = new javax.swing.table.TableRowSorter<>(jTableBackupSaveGames.getModel())))
   {
    if (null != (lc_tColumnModel = jTableBackupSaveGames.getColumnModel()))
    {
     for (lc_iCount = 0; lc_iCount < SkyrimCharacterHelperConstants.SKH_TABLE_COLUMNS; lc_iCount++)
     {
      if (null != (lc_tColumn = lc_tColumnModel.getColumn(lc_iCount)))
      {
       lc_iNewColumn = lc_iHeaderBackup[lc_iCount];
       
       lc_tColumn.setModelIndex(lc_iNewColumn); 
       lc_tColumn.setHeaderValue(m_tTableColumnHeaders[lc_iNewColumn]); 
       m_tRowSorterBackupSaveGames.setComparator(lc_iNewColumn, new SaveGameComparator(lc_iNewColumn)); // the comparator only cares about this column afterwards!
      }
     }
    }
    m_tRowSorterBackupSaveGames.setSortsOnUpdates(true);    
    jTableBackupSaveGames.getTableHeader().resizeAndRepaint(); 
    jTableBackupSaveGames.setRowSorter(m_tRowSorterBackupSaveGames);
    setSortOrder(m_tRowSorterBackupSaveGames, lc_sSortOrderBackup, jTableBackupSaveGames);
   }
  }
 }
 
 
 
 /**
  * Sets the launched flag
  * 
  * @param pa_bValue   true == Skyrim was launched
  */
 public final void setLaunched(boolean pa_bValue) 
 {
  m_bLaunched = pa_bValue;
 }
 
 
 
 /**
  * Sets the backup scanned flag
  * 
  * @param pa_bValue   true == backup folder was scanned completely
  */
 public final void setActiveScanned(boolean pa_bValue) 
 {
  m_bActiveScanned = pa_bValue;
  jTextFieldActivePath.setForeground(true == pa_bValue ? m_tTextfieldActivePathNormalFg : Color.red);
 }
 
 
 
 
 
 /**
  * Sets the backup scanned flag
  * 
  * @param pa_bValue   true == backup folder was scanned completely
  */
 public final void setBackupScanned(boolean pa_bValue) 
 {
  m_bBackupScanned = pa_bValue;
  jTextFieldBackupPath.setForeground(true == pa_bValue ? m_tTextfieldBackupPathNormalFg : Color.red);
 }
   
 
 
 /**
  * Locks the UI according to the given state
  * 
  * @param pa_bLock   true for locking, false for unlocking
  * @param pa_bAtomic if pa_bLock is true and pa_bAtomic is true, the cancel buttons gets disabled, too
  */
 public final void setUILockMode(boolean pa_bLock, boolean pa_bAtomic)
 {
  m_bLocked = pa_bLock;
  
  jButtonProgressCancel.setEnabled(pa_bLock && !pa_bAtomic          ); // in case of launch -> enabled. in case of launch -> enabled.
  jButtonProgressExit.setEnabled  (!pa_bLock && false == m_bLaunched); // in case of a launch, let the user the ability to end the tool   
  
  updateUIElements(); 
 }
 
 
 
 /**
  * Checks the small buttons according to data state
  * 
  */
 public final void updateUIElements()
 {
  boolean lc_bHasOneActiveCharacter = (1 == getListSize(jListActiveCharacters));
  boolean lc_bHasOneActiveSavegame  = (1 == getTableSize(jTableActiveSaveGames));
  boolean lc_bHasActiveCharacters   = (0 < getListSize (jListActiveCharacters));
  boolean lc_bHasActiveSaveGames    = (0 < getTableSize(jTableActiveSaveGames));
  boolean lc_bHasBackupCharacters   = (0 < getListSize (jListBackupCharacters));
  boolean lc_bHasBackupSaveGames    = (0 < getTableSize(jTableBackupSaveGames));
  
  boolean lc_bHasActiveImage      = (null != m_tActivePaintPanel.getImage());
  boolean lc_bHasBackupImage      = (null != m_tBackupPaintPanel.getImage());
  
  boolean lc_bHasActiveCharacterSelection = (null != getSelectedActivePlayers()   );
  boolean lc_bHasActiveSaveGameSelection  = (null != getSelectedActiveSaveGames() );
  boolean lc_bHasBackupCharacterSelection = (null != getSelectedBackupPlayers()   );
  boolean lc_bHasBackupSaveGameSelection  = (null != getSelectedBackupSaveGames() );
  
  boolean lc_b0 = (false == m_bLocked && false == m_bLaunched);
  boolean lc_b1 = (true == m_bIsWindowsOs);
  boolean lc_b2 = (true == lc_b0 && true == lc_bHasActiveCharacters && true == lc_bHasActiveCharacterSelection);
  boolean lc_b3 = (true == lc_b0 && true == lc_bHasActiveSaveGames  && true == lc_bHasActiveSaveGameSelection );
  boolean lc_b4 = (true == lc_b0 && true == lc_bHasBackupCharacters && true == lc_bHasBackupCharacterSelection);
  boolean lc_b5 = (true == lc_b0 && true == lc_bHasBackupSaveGames  && true == lc_bHasBackupSaveGameSelection );
  boolean lc_b6 = (true == checkString(getActivePath()  ));
  boolean lc_b7 = (true == checkString(getBackupPath()  ));
  boolean lc_b8 = (true == checkString(getLaunchBinary()));
  boolean lc_b9 = (true == checkString(getPrefix()      ));

  boolean lc_bLaunch = false;
  
  //
  // comboboxes
  //
  jComboBoxMode.setEnabled        (lc_b0);   
  jComboBoxConfirmation.setEnabled(lc_b0);
  jComboBoxOverwrite.setEnabled   (lc_b0);
  jComboBoxMinimize.setEnabled    (lc_b0);
  jComboBoxImageMode.setEnabled   (lc_b0);
  jComboBoxAutoSaves.setEnabled   (lc_b0 && lc_b8 && lc_b9);
  jComboBoxQuickSaves.setEnabled  (lc_b0 && lc_b8 && lc_b9);
  jComboBoxInitMode.setEnabled    (lc_b0);  
  
  //
  // lists
  //
  jListActiveCharacters.setEnabled(lc_b0 && m_bActiveScanned);   
  jListBackupCharacters.setEnabled(lc_b0 && m_bBackupScanned);   

  //
  // tables and table headers
  //
  jTableActiveSaveGames.setEnabled (lc_b0 && m_bActiveScanned);   
  jTableBackupSaveGames.setEnabled (lc_b0 && m_bBackupScanned);   
  
  jTableActiveSaveGames.getTableHeader().setEnabled(lc_b0 && m_bActiveScanned);   
  jTableBackupSaveGames.getTableHeader().setEnabled(lc_b0 && m_bBackupScanned);    
  
  //
  // non-savegame/character buttons and stuff
  //
  jTextFieldPrefix.setEnabled          (lc_b0);
  jTextFieldLaunchOptions.setEnabled   (lc_b0);
  jButtonAbout.setEnabled              (lc_b0);   
  jButtonChooseBinary.setEnabled       (lc_b0);
  jButtonRestoreRegistryPath.setEnabled(lc_b0 && lc_b1);
  jButtonLaunchSolo.setEnabled         (lc_b0 && lc_b1 && lc_b8 && lc_b9);
  
  //
  // buttons for the active character
  //
  jButtonActiveCharacterBackup.setEnabled(lc_b0 && lc_b2 && lc_b6 && lc_b7 && m_bActiveScanned && m_bBackupScanned);
  jButtonActiveCharacterDelete.setEnabled(lc_b0 && lc_b2 && m_bActiveScanned);
  jButtonActiveCharacterClear.setEnabled (lc_b0 && lc_bHasActiveCharacters && m_bActiveScanned);  
  
  // for launching, we need a backup directory only if there is more than 1 active player.
  lc_bLaunch = lc_b0 && lc_b1 && lc_b2 && lc_b6 && lc_b8 && lc_b9 && !hasListMultiSelectionActivePlayers() && m_bActiveScanned;
  if (false == lc_bHasOneActiveCharacter) lc_bLaunch &= (lc_b7 && m_bBackupScanned);  
  
  jButtonActiveCharacterLaunch.setEnabled(lc_bLaunch);
  
  //
  // buttons for the active savegame/path
  //
  jButtonActiveSaveGameBackup.setEnabled(lc_b0 && lc_b3 && lc_b6 && lc_b7 && m_bBackupScanned && m_bActiveScanned);
  jButtonActiveSaveGameDelete.setEnabled(lc_b0 && lc_b3 && m_bActiveScanned);
  jButtonActiveClipboard.setEnabled     (lc_b0 && lc_bHasActiveImage && m_bActiveScanned);
  jButtonActiveLaunchSavegame.setEnabled(lc_b0 && lc_b3 && !hasListMultiSelectionActiveSaveGames() && m_bActiveScanned && (true == lc_bHasOneActiveSavegame ? true : lc_b7 && m_bBackupScanned) );

  //
  // buttons for the active path
  //
  jButtonActivePath.setEnabled  (lc_b0);
  jButtonActiveRescan.setEnabled(lc_b0 && lc_b6);

  //
  // buttons for the backup character
  //
  jButtonBackupCharacterRestore.setEnabled(lc_b0 && lc_b4 && lc_b6 && lc_b7 && m_bBackupScanned && m_bActiveScanned);
  jButtonBackupCharacterDelete.setEnabled (lc_b0 && lc_b4 && m_bBackupScanned);
  jButtonBackupCharacterClear.setEnabled  (lc_b0 && lc_bHasBackupCharacters && m_bBackupScanned);
  jButtonBackupCharacterLaunch.setEnabled (lc_b0 && lc_b1 && lc_b4 && lc_b6 && lc_b7 && lc_b8 && lc_b9 && !hasListMultiSelectionBackupPlayers() && m_bBackupScanned && m_bActiveScanned);

  
  //
  // buttons for the active savegame/path
  //
  jButtonBackupSaveGameRestore.setEnabled (lc_b0 && lc_b5 && lc_b6 && lc_b7 && m_bBackupScanned && m_bActiveScanned);
  jButtonBackupSaveGameDelete.setEnabled  (lc_b0 && lc_b5);
  jButtonBackupClipboard.setEnabled       (lc_b0 && lc_bHasBackupImage);      
  jButtonBackupLaunchSavegame.setEnabled  (lc_b0 && lc_b5 && !hasListMultiSelectionBackupSaveGames() && m_bBackupScanned && m_bActiveScanned);
  
  //
  // buttons for the active path
  //
  jButtonBackupPath.setEnabled  (lc_b0);  
  jButtonBackupRescan.setEnabled(lc_b0 && lc_b7);  
 }
 
 
 
 /**
  * Checks whether source and destination directory are non-null, non-empty and equal
  * 
  * @return  true, if they're equal, false otherwise
  */
 public final boolean hasEqualDirectories()
 {
  String lc_sSourceDir      = getActivePath();
  String lc_sDestinationDir = getBackupPath();
  boolean lc_bEquals = false;
  
  if (true == checkString(lc_sSourceDir) && true == checkString(lc_sDestinationDir))
  {
   if (true == lc_sSourceDir.equals(lc_sDestinationDir)) 
   {
    lc_bEquals = true;
   }
  }
  return lc_bEquals;
 }
 
 
 
 /**
  * Initializes the player list with data and eventually
  * 
  * Warning: setListData(Object[] array) a new instance of the ListModel incompatible with DefaultListModel. Therefore,
  *          new entries must be added one by one... which really sucks!
  * 
  * @param pa_tData   List data
  * @param pa_bMode   true = active player list, false = backup player list
  */
 public final synchronized void initPlayerList(SkyrimCharacterHelperPlayerManager pa_tManager, boolean pa_bMode)
 {
  SkyrimCharacterHelperPlayer   lc_tFirstSelectedPlayer = null;
  SkyrimCharacterHelperPlayer[] lc_tData                = null;
         
  java.util.List<SkyrimCharacterHelperPlayer> lc_tOldSelection = (true == pa_bMode ? m_tStoredActivePlayerSelection : m_tStoredBackupPlayerSelection);
  javax.swing.JList                           lc_tList         = (true == pa_bMode ? jListActiveCharacters : jListBackupCharacters);
  SkyrimCharacterHelperListModel              lc_tModel        = (null != lc_tList ? (SkyrimCharacterHelperListModel) lc_tList.getModel() : null);
  
  boolean lc_bMultiSelection = false;
  boolean lc_bUpdated        = false;
  
  if (null != lc_tModel && null != lc_tList)
  {
   lc_tList.getSelectionModel().clearSelection();
   lc_tModel.clear();

   if (null != pa_tManager)
   {
    if (null != (lc_tData = pa_tManager.getPlayers()))
    {
     if (0 < lc_tData.length)
     {
      lc_tModel.addAll(lc_tData);
      
      //
      // all the stuff below needs to be done here to stay in sync with the model! 
      if (null != lc_tOldSelection)
      {
       //
       // relesect player
       //
       reselectPlayers(lc_tOldSelection, pa_bMode);
       lc_tFirstSelectedPlayer =  (true == pa_bMode ? getFirstSelectedActivePlayer() : getFirstSelectedBackupPlayer()); 

       //
       // build new savegame list
       //
       lc_bMultiSelection = (1 < lc_tOldSelection.size());
       
       if (null != lc_tFirstSelectedPlayer && false == lc_bMultiSelection)
       {
        initSaveGameList(lc_tFirstSelectedPlayer.getSaveGames(), pa_bMode, true);
        lc_bUpdated = true; // don't update UI twice, because initSaveGameList will call updateUI itself...
       }       
      }
     }
    }
   }
  }
  if (null != pa_tManager)
  {
   if (false == lc_bMultiSelection)
   {
    if (null != lc_tFirstSelectedPlayer)
    {
     if (true == pa_bMode) {setActivePlayerRace(lc_tFirstSelectedPlayer.getRace()); setAmountActive(pa_tManager, lc_tFirstSelectedPlayer.getSaveGameAmount());}
     else                  {setBackupPlayerRace(lc_tFirstSelectedPlayer.getRace()); setAmountBackup(pa_tManager, lc_tFirstSelectedPlayer.getSaveGameAmount());}
    }
   }
   else
   {
    if (true == pa_bMode) {setActivePlayerRace(null); setAmountActive(pa_tManager, pa_tManager.getSaveGameAmountForList(getSelectedActivePlayers()));}
    else                  {setBackupPlayerRace(null); setAmountBackup(pa_tManager, pa_tManager.getSaveGameAmountForList(getSelectedBackupPlayers()));}
   }
  }
  if (false == lc_bUpdated) updateUIElements();
 }
  
 
 
 /**
  * Initializes the savegame list with data
  * 
  * @param pa_tData         List data
  * @param pa_bMode         true = active player list, false = backup player list
  */
 public final synchronized void initSaveGameList(SkyrimCharacterHelperSaveGame[] pa_tData, boolean pa_bMode, boolean pa_bUseStoredSelection)
 {
  final SkyrimCharacterHelperSaveGame[] lc_tFData = pa_tData;
  final boolean lc_bMode = pa_bMode;
  final boolean lc_bUseStoredSelection = pa_bUseStoredSelection;
  
  javax.swing.SwingUtilities.invokeLater(new Runnable()
  {
   public void run() 
   { 
    SkyrimCharacterHelperSaveGame   lc_tSaveGame = null;;
    SkyrimCharacterHelperSaveGame[] lc_tData     = lc_tFData;
  
    java.awt.image.BufferedImage                  lc_tImage        = null;
    javax.swing.JTable                            lc_tTable        = (true == lc_bMode ? jTableActiveSaveGames : jTableBackupSaveGames);
    SkyrimCharacterHelperTableModel               lc_tModel        = (SkyrimCharacterHelperTableModel) lc_tTable.getModel();
    java.util.List<SkyrimCharacterHelperSaveGame> lc_tOldSelection = (true == lc_bMode ? (true == lc_bUseStoredSelection ? m_tStoredActiveSaveGameSelection : getSelectedActiveSaveGames())
                                                                                       : (true == lc_bUseStoredSelection ? m_tStoredBackupSaveGameSelection : getSelectedBackupSaveGames()));
    
    if (true == lc_bMode) setActiveSaveGameScreenshot(null); 
    else                  setBackupSaveGameScreenshot(null); 

    if (null != lc_tModel && null != lc_tTable && null != lc_tData)
    {
     java.util.List lc_tSortKeys = disableTableRowSorter(true == lc_bMode ? jTableActiveSaveGames : jTableBackupSaveGames); // disable rowsorter  
     
     lc_tModel.clear();
     lc_tTable.getSelectionModel().clearSelection();

     if (0 < lc_tData.length)
     {
      lc_tModel.addAll(lc_tData);
      if (null != lc_tOldSelection)
      {
       reselectSaveGames(lc_tOldSelection, lc_bMode, false);
       lc_tSaveGame = (true == lc_bMode ? getFirstSelectedActiveSaveGame() : getFirstSelectedBackupSaveGame()); 
       
       //
       // display any stuff?
       //
       if (null != lc_tSaveGame && 1 == lc_tTable.getSelectedRowCount())
       {
        if (null != lc_tSaveGame.getScreenshot())
        {
         lc_tImage = lc_tSaveGame.getScreenshot().getImage();
        }
       }
      }
     }
     if (true == lc_bMode) setActiveSaveGameScreenshot(lc_tImage); 
     else                  setBackupSaveGameScreenshot(lc_tImage); 
     
     enableTableRowSorter(true == lc_bMode ? jTableActiveSaveGames : jTableBackupSaveGames, lc_tSortKeys); // enable rowsorter  
     updateUIElements(); // IMPORTANT: Need to stay in sync, so this must happen here!
    }
   }
  });      
 }

 

 /**
  * Reselects savegames. Note, that some elements of the old selection may not be contained anymore
  * 
  * @param pa_tOldSelection  previous selection
  * @param pa_bMode         true = active player list, false = backup player list
  * @param pa_bNotify       currently unused
  */
 private synchronized void reselectSaveGames(java.util.List<SkyrimCharacterHelperSaveGame> pa_tOldSelection, boolean pa_bMode, boolean pa_bNotify)
 {
  int lc_iLastIndex     = -1;
  int lc_iModelIndex    = -1;
  int lc_iViewIndex     = -1;
  int lc_iNewSelectionSize = 0;
  
  SkyrimCharacterHelperSaveGame   lc_tSaveGame         = null;
  SkyrimCharacterHelperSaveGame   lc_tSelectedSaveGame = null;
  
  java.awt.image.BufferedImage    lc_tImage    = null;
  javax.swing.JTable              lc_tTable    = (true == pa_bMode ? jTableActiveSaveGames : jTableBackupSaveGames);
  SkyrimCharacterHelperTableModel lc_tModel    = (SkyrimCharacterHelperTableModel) lc_tTable.getModel();
  
  if (null != pa_tOldSelection && null != lc_tModel && null != lc_tTable)
  {
   if (0 < pa_tOldSelection.size())
   { 
    lc_tTable.getSelectionModel().clearSelection();
    
    //
    // process every list entry
    //
    for (Object lc_tObject : pa_tOldSelection) 
    {
     if (null != lc_tObject)
     {
      if (lc_tObject instanceof SkyrimCharacterHelperSaveGame) 
      {
       if (null != (lc_tSaveGame = (SkyrimCharacterHelperSaveGame) lc_tObject))
       {
        if (-1 != (lc_iModelIndex = lc_tModel.indexOf(lc_tSaveGame)))
        {
         if (-1 != (lc_iViewIndex = lc_tTable.convertRowIndexToView(lc_iModelIndex)))
         {
          lc_iLastIndex = lc_iViewIndex;
          lc_tTable.addRowSelectionInterval(lc_iViewIndex, lc_iViewIndex);
          lc_iNewSelectionSize++;
         }
        }
       }
      }
     }
    } // for
    
    //
    // scroll to last visible selection
    //
    if (-1 != lc_iLastIndex)
    {
     lc_tTable.scrollRectToVisible(new java.awt.Rectangle(lc_tTable.getCellRect(lc_iLastIndex, 0, true)));
    }
    
    //
    // display any stuff?
    //
    if (1 == lc_iNewSelectionSize)
    {
     if (null != (lc_tSelectedSaveGame = (true == pa_bMode ? getFirstSelectedActiveSaveGame() : getFirstSelectedBackupSaveGame())))
     {
      if (null != lc_tSelectedSaveGame.getScreenshot())
      {
       lc_tImage = lc_tSelectedSaveGame.getScreenshot().getImage();
      }
     }
    }
   }
  }
  if (true == pa_bMode) setActiveSaveGameScreenshot(lc_tImage); 
  else                  setBackupSaveGameScreenshot(lc_tImage);    
 }
 
 
 
/**
  * Reselects players. Note, that some elements of the old selection may not be contained anymore
  * 
  * @param pa_tOldSelection  previous selection
  * @param pa_bMode         true = active player list, false = backup player list
  * @return amount of
  */
 public final synchronized void reselectPlayers(java.util.List<SkyrimCharacterHelperPlayer> pa_tOldSelection, boolean pa_bMode)
 {
  Object[]                      lc_tEntries   = null;
  int                           lc_iIndex     = 0   ;
  int                           lc_iLastIndex = 0   ;
  SkyrimCharacterHelperPlayer   lc_tPlayer    = null;
  
  javax.swing.JList              lc_tList         = (false == pa_bMode ? jListBackupCharacters : jListActiveCharacters);
  java.util.List<Integer>        lc_tPositionList = new java.util.ArrayList<>();
  SkyrimCharacterHelperListModel lc_tModel        = (null != lc_tList ? (SkyrimCharacterHelperListModel) lc_tList.getModel() : null);
  
  if (null != pa_tOldSelection && null != lc_tModel && null != lc_tList && null != lc_tPositionList)
  {
   if (0 < pa_tOldSelection.size())
   {
    if (null != (lc_tEntries = lc_tModel.elements()))
    {
     if (0 < lc_tEntries.length)
     {
      lc_tList.getSelectionModel().clearSelection();

      //
      // process every list entry
      //
      for (Object lc_tObject : lc_tEntries) 
      {
       if (lc_tObject instanceof SkyrimCharacterHelperPlayer) 
       {
        if (null != (lc_tPlayer = (SkyrimCharacterHelperPlayer) lc_tObject))
        {
         //
         // check, if this savegame was contained in the old selection. if true, memorize the index
         //       
         for (SkyrimCharacterHelperPlayer lc_tSelectionPlayer : pa_tOldSelection)
         {
          if (null != lc_tSelectionPlayer)
          {
           if (lc_tSelectionPlayer.getName().equals(lc_tPlayer.getName()))
           {
            lc_tPositionList.add(lc_iIndex); 
            lc_iLastIndex = lc_iIndex;
           }
          }
         } // for       
        }
       }
       lc_iIndex++;
      } // for
      if (0 < lc_tPositionList.size())
      {
       setSelectedIndices(lc_tList, lc_tPositionList);
       lc_tList.ensureIndexIsVisible(lc_iLastIndex);
      }
     }
    }
   }
  }
 } 

 
 
 
/**
  * Reselects indices for a list
  * 
  * @param pa_tList          list 
  * @param pa_tPositionList  list of indices to select
  */
 private void setSelectedIndices(javax.swing.JList pa_tList, java.util.List<Integer> pa_tPositionList)
 {
  int[] lc_tIncidices = null;
  int   lc_iIndex     = 0   ; 
   
  if (null != pa_tList && null != pa_tPositionList)
  {
   if (0 < pa_tPositionList.size())
   {
    if (null != (lc_tIncidices = new int[pa_tPositionList.size()])) 
    {
     for (Integer lc_tInteger : pa_tPositionList) 
     {
      lc_tIncidices[lc_iIndex] = lc_tInteger.intValue();
      lc_iIndex++;        
     }
     pa_tList.setSelectedIndices(lc_tIncidices);
    }
   }
  }
 }
 
 
 
 /**
  * Adds the given player to the active player list
  * 
  * @param pa_tPlayer new player
  */
 public final void addActivePlayer(SkyrimCharacterHelperPlayer pa_tPlayer)
 {
  SkyrimCharacterHelperListModel lc_tModel = (SkyrimCharacterHelperListModel) jListActiveCharacters.getModel();
  
  if (null != lc_tModel && null != pa_tPlayer)
  {
   lc_tModel.add(pa_tPlayer);    
  }
 }
 
 
 
 /**
  * Adds the given player to the backup player list
  * 
  * @param pa_tPlayer new player
  */
 public final void addBackupPlayer(SkyrimCharacterHelperPlayer pa_tPlayer)
 {
  SkyrimCharacterHelperListModel lc_tModel = (SkyrimCharacterHelperListModel) jListBackupCharacters.getModel();
  
  if (null != lc_tModel && null != pa_tPlayer)
  {
    lc_tModel.add(pa_tPlayer);    
  }
 }
 
 
 
 /**
  * Adds the given savegame to the active savegame list, selects the new entry, if needed
  * 
  * @param pa_tSaveGame new savegame
  * @param püa_bSelect     true = select entry, false = don't select
  */
 public final synchronized void addActiveSaveGame(SkyrimCharacterHelperSaveGame pa_tSaveGame, boolean pa_bSelect)
 {
  final SkyrimCharacterHelperSaveGame lc_tSaveGame = pa_tSaveGame;
  final boolean lc_bSelect = pa_bSelect;
  
  javax.swing.SwingUtilities.invokeLater(new Runnable()
  {
   public void run() 
   { 
    SkyrimCharacterHelperTableModel lc_tModel = (SkyrimCharacterHelperTableModel) jTableActiveSaveGames.getModel();
    int lc_iModelIndex = -1;
    int lc_iViewIndex  = -1;
    
    if (null != lc_tModel && null != lc_tSaveGame)
    {
     if (-1 != (lc_iModelIndex = lc_tModel.add(lc_tSaveGame)))
     {
     //
      // select new entry, if needed
      //
      if (true == lc_bSelect)
      {
       if (-1 != (lc_iViewIndex  = jTableActiveSaveGames.convertRowIndexToView(lc_iModelIndex)))
       {
        jTableActiveSaveGames.getSelectionModel().clearSelection();
        jTableActiveSaveGames.setRowSelectionInterval(lc_iViewIndex, lc_iViewIndex);
        jTableActiveSaveGames.scrollRectToVisible(new java.awt.Rectangle(jTableActiveSaveGames.getCellRect(lc_iViewIndex, 0, true)));
       
        if (null != m_tGuiConnector) {m_tGuiConnector.handleListSelectionActiveSaveGames();} // force processing by pretending a mouseclick :)
       }
      }
     } 
    }
   }
  });     
 }
 
 
 
 /**
  * Adds the given savegame to the backup savegame list
  * 
  * @param pa_tSaveGame new savegame
  */
 public final synchronized void addBackupSaveGame(SkyrimCharacterHelperSaveGame pa_tSaveGame, boolean pa_bSelect)
 {
  final SkyrimCharacterHelperSaveGame lc_tSaveGame = pa_tSaveGame;
  final boolean lc_bSelect = pa_bSelect;

  javax.swing.SwingUtilities.invokeLater(new Runnable()
  {
   public void run() 
   { 
    SkyrimCharacterHelperTableModel lc_tModel = (SkyrimCharacterHelperTableModel) jTableBackupSaveGames.getModel();
    int lc_iModelIndex = -1;
    int lc_iViewIndex  = -1;

    if (null != lc_tModel && null != lc_tSaveGame)
    {
     if (-1 != (lc_iModelIndex = lc_tModel.add(lc_tSaveGame)))
     {
     //
      // select new entry, if needed
      //
      if (true == lc_bSelect)
      {
       if (-1 != (lc_iViewIndex  = jTableBackupSaveGames.convertRowIndexToView(lc_iModelIndex)))
       {
        jTableBackupSaveGames.getSelectionModel().clearSelection();
        jTableBackupSaveGames.setRowSelectionInterval(lc_iViewIndex, lc_iViewIndex);
        jTableBackupSaveGames.scrollRectToVisible(new java.awt.Rectangle(jTableBackupSaveGames.getCellRect(lc_iViewIndex, 0, true)));
       
        if (null != m_tGuiConnector) {m_tGuiConnector.handleListSelectionBackupSaveGames();} // force processing by pretending a mouseclick :)
       }
      }
     }
    }
   }
  });     
 }
 
         
         
 /**
  * Returns the quicksave mode
  * 
  * @return   0 = copy,1 = leave
  */
 public final int getQuickSaveMode()
 {
  return jComboBoxQuickSaves.getSelectedIndex();
 }
 
 
 
 /**
  * Sets the quicksave mode
  * 
  * @return   0 = copy,1 = leave
  */
 public final void setQuickSaveMode(int pa_iMode)
 {
  if (pa_iMode == 0 || pa_iMode == 1)  
  {
   jComboBoxQuickSaves.setSelectedIndex(pa_iMode); 
  }
 }
 
 

 /**
  * Returns the autosave mode
  * 
  * @return   0 = copy,1 = leave
  */
 public final int getAutoSaveMode()
 {
  return jComboBoxAutoSaves.getSelectedIndex();
 }
 
 
 
 /**
  * Sets the autosave mode
  * 
  * @return   0 = copy,1 = leave
  */
 public final void setAutoSaveMode(int pa_iMode)
 {
  if (pa_iMode == 0 || pa_iMode == 1)  
  {
   jComboBoxAutoSaves.setSelectedIndex(pa_iMode); 
  }
 }
 
 
 
 /**
  * Returns the minimize mode
  * 
  * @return   0 = Backup,1 = Copy
  */
 public final int getMinimizeMode()
 {
  return jComboBoxMinimize.getSelectedIndex();
 }
 
 
 
 /**
  * Sets the minimize mode
  * 
  * @param pa_iMode   0 = Backup,1 = Copy
  */
 public final void setMinimizeMode(int pa_iMode)
 {
  if (pa_iMode == 0 || pa_iMode == 1)  
  {
   jComboBoxMinimize.setSelectedIndex(pa_iMode); 
  }
 }

 
  /**
  * Returns the init mode
  * 
  * @return   0 = scan both, 1 = scan skyrim only
  */
 public final int getInitMode()
 {
  return jComboBoxInitMode.getSelectedIndex();
 }
 
 
 
 /**
  * Sets the init mode mode
  * 
  * @return   0 = scan both, 1 = scan skyrim only
  */
 public final void setInitMode(int pa_iMode)
 {
  if (pa_iMode == 0 || pa_iMode == 1)  
  {
   jComboBoxInitMode.setSelectedIndex(pa_iMode); 
  }
 }
 
 
 /**
  * Returns the file mode
  * 
  * @return   0 = Backup,1 = Copy
  */
 public final int getFileMode()
 {
  return jComboBoxMode.getSelectedIndex();
 }
 
 
 
 /**
  * Sets the file mode
  * 
  * @param pa_iMode   0 = Backup,1 = Copy
  */
 public final void setFileMode(int pa_iMode)
 {
  if (pa_iMode == 0 || pa_iMode == 1)  
  {
   jComboBoxMode.setSelectedIndex(pa_iMode); 
  }
 }

 
 
 /**
  * Returns the work mode
  * 
  * @return   0 = yes,1 = no
  */
 public final int getOverwriteMode()
 {
  return jComboBoxOverwrite.getSelectedIndex();
 }
 
 
 
 /**
  * Sets the work mode
  * 
  * @param pa_iMode  0 = yes,1 = no
  */
 public final void setOverwriteMode(int pa_iMode)
 {
  if (pa_iMode == 0 || pa_iMode == 1)  
  {
   jComboBoxOverwrite.setSelectedIndex(pa_iMode); 
  }
 }

 
 
 /**
  * Returns the confirmation mode
  * 
  * @return   0 = yes,1 = no
  */
 public final int getConfirmationMode()
 {
  return jComboBoxConfirmation.getSelectedIndex();
 }
 
 
 
 /**
  * Sets the confirmation mode
  * 
  * @param pa_iMode   0 = Backup,1 = Copy
  */
 public final void setConfirmationMode(int pa_iMode)
 {
  if (pa_iMode == 0 || pa_iMode == 1)  
  {
   jComboBoxConfirmation.setSelectedIndex(pa_iMode); 
  }
 }
 
 
 /**
  * Returns the confirmation mode
  * 
  * @return   0 = yes,1 = no
  */
 public final int getImageMode()
 {
  return jComboBoxImageMode.getSelectedIndex();
 }
 
 
 
 /**
  * Sets the confirmation mode
  * 
  * @param pa_iMode   0 = Backup,1 = Copy
  */
 public final void setImageMode(int pa_iMode)
 {
  if (pa_iMode == 0 || pa_iMode == 1)  
  {
   jComboBoxImageMode.setSelectedIndex(pa_iMode); 
   setImageModeDisplay(pa_iMode == 0);
  }
 }
 
 
 
 /**
  * Returns the amount of characters in the active list
  * 
  * @return  amount, may be 0
  */
 public final int getAmountActivePlayers() {return getListSize(jListActiveCharacters);}
 
 
 
 /**
  * Returns the amount of savegames in the active savegame list
  * 
  * @return  amount, may be 0
  */
 public final int getAmountActiveSaveGames()  {return getTableSize(jTableActiveSaveGames);}
 
 
 
 /**
  * Returns the amount of characters in the backup list
  * 
  * @return  amount, may be 0
  */
 public final int getAmountBackupPlayers() {return getListSize(jListBackupCharacters);}
 
 
 
 /**
  * Returns the amount of savegames in the backup savegame list
  * 
  * @return  amount, may be 0
  */
 public final int getAmountBackupSaveGames()  {return getTableSize(jTableBackupSaveGames);}

 
 
 /**
  * Displays the amount of active savegames
  * 
  * @param pa_iValue   amount
  */
 public final void setAmountActive(SkyrimCharacterHelperPlayerManager pa_tManager, int pa_iValue) 
 {
  String lc_sValue = "";
  
  if (0 <  pa_iValue) lc_sValue += Integer.toString(pa_iValue) + "/";    
  if (null != pa_tManager)
  {
   if (0 < pa_tManager.getSaveGameAmount()) 
     lc_sValue += Integer.toString(pa_tManager.getSaveGameAmount());
  }
  jTextFieldAmountActive.setText(lc_sValue); jTextFieldAmountActive.setCaretPosition(0);
 }
 
 
 
 /**
  * Clears the amount of backup savegames
  * 
  */
 public final void resetAmountBackup()
 {
  jTextFieldAmountBackup.setText(""); 
 }

 
 
 
 /**
  * Clears the amount of active savegames
  * 
  */
  public final void resetAmountActive()
 {
  jTextFieldAmountActive.setText(""); 
 }
 
 
 
 /**
  * Displays the amount of active savegames
  * 
  * @param SkyrimCharacterHelperPlayerManager  savegame manager to use
  * @param pa_iValue   amount
  */
 public final void setAmountBackup(SkyrimCharacterHelperPlayerManager pa_tManager, int pa_iValue) 
 {
  String lc_sValue = "";
  
  if (0 <  pa_iValue) lc_sValue += Integer.toString(pa_iValue) + "/";    
  if (null != pa_tManager)
  {
   if (0 < pa_tManager.getSaveGameAmount()) 
     lc_sValue += Integer.toString(pa_tManager.getSaveGameAmount());
  }
  jTextFieldAmountBackup.setText(lc_sValue); jTextFieldAmountBackup.setCaretPosition(0);
 }

 

 /**
  * Deletes the given selected savegame from the list and selects the next entry
  * 
  * @param pa_tSaveGame    savegame to delete
  * @parma pa_bReselect     true for reselecting after deleting, false otherwise
  */
 public final synchronized void removeActiveSaveGame(SkyrimCharacterHelperSaveGame pa_tSaveGame, boolean pa_bReselect)
 {
  final SkyrimCharacterHelperSaveGame lc_tSaveGame = pa_tSaveGame;
  final boolean lc_bReselect = pa_bReselect;

  javax.swing.SwingUtilities.invokeLater(new Runnable()
  {
   public void run() 
   { 
    int lc_iNewSelectionPos = -1;
    
    if (-1 != (lc_iNewSelectionPos = removeSaveGameListEntry(jTableActiveSaveGames, lc_tSaveGame)))
    {
     if (true == lc_bReselect)
     {
      jTableActiveSaveGames.getSelectionModel().clearSelection();
      jTableActiveSaveGames.addRowSelectionInterval(lc_iNewSelectionPos, lc_iNewSelectionPos); 
     }
    }
   }
  });     
 }
 
 
 
 /**
  * Deletes the given backup savegame from the list and selects the next entry
  * 
  * @param pa_tSaveGame    savegame to delete
  * @parma pa_bReselect     true for reselecting after deleting, false otherwise
  */
 public final synchronized void removeBackupSaveGame(SkyrimCharacterHelperSaveGame pa_tSaveGame, boolean pa_bReselect)
 {
  final SkyrimCharacterHelperSaveGame lc_tSaveGame = pa_tSaveGame;
  final boolean lc_bReselect = pa_bReselect;
  
  javax.swing.SwingUtilities.invokeLater(new Runnable()
  {
   public void run() 
   { 
    int lc_iNewSelectionPos = -1;

    if (-1 != (lc_iNewSelectionPos = removeSaveGameListEntry(jTableBackupSaveGames, lc_tSaveGame)))
    {
     if (true == lc_bReselect)
     {
      jTableBackupSaveGames.getSelectionModel().clearSelection();
      jTableBackupSaveGames.addRowSelectionInterval(lc_iNewSelectionPos, lc_iNewSelectionPos); 
     }
    }
   }
  });       
 }
 
 
 
 /**
  * Deletes the given backup character from the list
  * 
  * @param pa_tCharacter    character to delete
  * @parma pa_bReselect     true for reselecting after deleting, false otherwise
  */
 public final synchronized void removeActivePlayer(SkyrimCharacterHelperPlayer pa_tCharacter, boolean pa_bReselect, boolean pa_bUseStoredSelection)
 {
  int lc_iNewSelectionPos = -1;
  
  if (true == pa_bUseStoredSelection) storeSaveGameSelection(true );
  
  clearActiveSaveGameTable();
  setActiveSaveGameScreenshot(null);
  setActivePlayerRace(null);  

  if (-1 != (lc_iNewSelectionPos = removePlayerListEntry(jListActiveCharacters, pa_tCharacter)))
  { 
   if (true == pa_bReselect)
   {
    jListActiveCharacters.getSelectionModel().clearSelection();
    jListActiveCharacters.setSelectedIndex(lc_iNewSelectionPos); 
  
    if (null != getFirstSelectedActivePlayer())
    { 
     initSaveGameList(getFirstSelectedActivePlayer().getSaveGames(), true, pa_bUseStoredSelection);
     
     if (false == pa_bUseStoredSelection)
     {
      if (null != m_tGuiConnector) {m_tGuiConnector.handleListSelectionActivePlayers();} // force processing by pretending a mouseclick :)
     }
    }
   }
  }
 }
 
 
 
 /**
  * Deletes the given backup character from the list
  * 
  * @param pa_tCharacter    character to delete
  * @parma pa_bReselect     true for reselecting after deleting, false otherwise
  */
 public final synchronized void removeBackupPlayer(SkyrimCharacterHelperPlayer pa_tCharacter, boolean pa_bReselect, boolean pa_bUseStoredSelection)
 {
  int lc_iNewSelectionPos = -1;
  
  if (true == pa_bUseStoredSelection) storeSaveGameSelection(false );

  clearBackupSaveGameTable();
  setBackupSaveGameScreenshot(null);
  setBackupPlayerRace(null);   

  if (-1 != (lc_iNewSelectionPos = removePlayerListEntry(jListBackupCharacters, pa_tCharacter)))
  {
   if (true == pa_bReselect)
   {
    jListBackupCharacters.getSelectionModel().clearSelection();
    jListBackupCharacters.setSelectedIndex(lc_iNewSelectionPos); 

    if (null != getFirstSelectedBackupPlayer())
    {
     initSaveGameList(getFirstSelectedBackupPlayer().getSaveGames(), false, pa_bUseStoredSelection);
     
     if (false == pa_bUseStoredSelection)
     {
      if (null != m_tGuiConnector) {m_tGuiConnector.handleListSelectionBackupPlayers();} // force processing by pretending a mouseclick :)
     }
    }
   }
  }
 }


 
 /**
  * Clears the active player list
  * 
  */
 public final void clearActivePlayerList()
 {
  clearList(jListActiveCharacters);
 }
  
  
  
 /**
  * Clears the backup player list
  * 
  */
 public final void clearBackupPlayerList()
 {
  clearList(jListBackupCharacters);
 }
 
 
 
 /**
  * Clears the active savegame list
  * 
  */
 public final void clearActiveSaveGameTable()
 {
  clearTable(jTableActiveSaveGames);
 }
  
  
  
 /**
  * Clears the backup savegame list
  * 
  */
 public final void clearBackupSaveGameTable()
 {
  clearTable(jTableBackupSaveGames);
 }

 
  
/**
  * Returns the selected savegams
  * 
  * @param pa_tTable  table
  * @return  savegames
  */
 private synchronized java.util.List<SkyrimCharacterHelperSaveGame> getSelectedSaveGames(javax.swing.JTable pa_tTable)
 {
  java.util.ArrayList<SkyrimCharacterHelperSaveGame> lc_tResult = null;
  
  int[] lc_tIndices    =  pa_tTable.getSelectedRows();
  int   lc_iModelIndex = -1;
  
  Object  lc_tObject = null;
  boolean lc_bOk     = false;
  
  if (null != lc_tIndices)
  {
   if (0 < lc_tIndices.length)
   {
    if (null != (lc_tResult = new java.util.ArrayList<>()))
    {
     try
     {
      synchronized(pa_tTable)
      {
      for (int lc_iIndex : lc_tIndices) 
      {
       //
       // if this beast throws an exception, this just means that a display runnable which came late wants to do something and needs a list. so we just leave the break and do nothing, which
       // returns null, which renders this late-comer useless and prevents it from causing any damage. tables are extremely sensitive to sync issues...
       // 
       lc_iModelIndex = pa_tTable.convertRowIndexToModel(lc_iIndex);
      
       if (0 <= lc_iModelIndex && lc_iModelIndex < pa_tTable.getRowCount())
       {
        if (null != (lc_tObject = pa_tTable.getValueAt(lc_iIndex, 0))) 
        {                                                              
         if (lc_tObject instanceof SkyrimCharacterHelperSaveGame)
         {
          lc_tResult.add((SkyrimCharacterHelperSaveGame) lc_tObject); 
         }
        }
       }
      } // for
      lc_bOk = true; // at this point, we can be sure all went fine and no exception was raised, which in turn
                    // means that the associated display runnable is in sync. so we can pass back the results.
     }
     }
     catch (Exception lc_tException) {}
    }
   }
  }
  return (true == lc_bOk ? lc_tResult : null);
 }
 
 
 
 /**
  * Returns the currently selected active savegames
  * 
  * @return   list, may be null
  */
 public final synchronized java.util.List<SkyrimCharacterHelperSaveGame> getSelectedActiveSaveGames()
 {
  return getSelectedSaveGames(jTableActiveSaveGames);
 }
  
  
  
 /**
  * Returns the currently selected backup savegames
  * 
  * @return   list, may be null
  */
 public final synchronized java.util.List<SkyrimCharacterHelperSaveGame> getSelectedBackupSaveGames()
 {
  return getSelectedSaveGames(jTableBackupSaveGames);
 }
  
  
  
  /**
   * Returns the currently selected active players
   * 
   * @return   list, may be null
   */
  public final synchronized java.util.List<SkyrimCharacterHelperPlayer> getSelectedActivePlayers()
  {
   java.util.List<SkyrimCharacterHelperPlayer> lc_tList = jListActiveCharacters.getSelectedValuesList();
   
   if (null != lc_tList)
   {
    if (false == lc_tList.isEmpty()) 
    {
     return lc_tList;
    }
   }
   return null;
  }
  
  
  
  /**
   * Returns all active players
   * 
   * @return   list, may be null
   */
  public final synchronized java.util.List<SkyrimCharacterHelperPlayer> getActivePlayers()
  {
   SkyrimCharacterHelperListModel lc_tModel = (SkyrimCharacterHelperListModel) jListActiveCharacters.getModel();
   return (null != lc_tModel ? lc_tModel.getPlayerList() : null);
  }
  
  
  
  /**
   * Returns all backup players
   * 
   * @return   list, may be null
   */
  public final synchronized java.util.List<SkyrimCharacterHelperPlayer> getBackupPlayers()
  {
   SkyrimCharacterHelperListModel lc_tModel = (SkyrimCharacterHelperListModel) jListBackupCharacters.getModel();
   return (null != lc_tModel ? lc_tModel.getPlayerList() : null);
  }

  
  
  /**
   * Returns the currently selected backup players
   * 
   * @return   list, may be null
   */
  public final synchronized java.util.List<SkyrimCharacterHelperPlayer> getSelectedBackupPlayers()
  {
   java.util.List<SkyrimCharacterHelperPlayer> lc_tList = jListBackupCharacters.getSelectedValuesList();
   
   if (null != lc_tList)
   {
    if (false == lc_tList.isEmpty()) 
    {
     return lc_tList;
    }
   }
   return null;
  }
  
  
  
 /**
  * Returns the active directory path
  * 
  * @return active active path, may be null
  */
 public final String getActivePath() {return jTextFieldActivePath.getText();}
 

 
 /**
  * Sets the active directory path
  * 
  * @param pa_sValue  active path, may be null
  */
 public final void setActivePath(String pa_sValue) 
 {
  jTextFieldActivePath.setForeground(true == m_bActiveScanned ? m_tTextfieldActivePathNormalFg : Color.red);
  jTextFieldActivePath.setText(pa_sValue); 
  jTextFieldActivePath.setCaretPosition(0);
 }
 

 
 
 /**
  * Returns the backup directory path
  * 
  * @return backup path, may be null
  */
 public final String getBackupPath() {return jTextFieldBackupPath.getText();}
 
 
 
 /**
  * Sets the  backup directory path
  * 
  * @param pa_sValue   backup path, may be null
  */
 public final void setBackupPath(String pa_sValue) 
 {
  jTextFieldBackupPath.setForeground(true == m_bBackupScanned ? m_tTextfieldBackupPathNormalFg : Color.red);
  jTextFieldBackupPath.setText(pa_sValue); 
  jTextFieldBackupPath.setCaretPosition(0);
 }

 
 
 /**
  * Sets the message
  * 
  * @param pa_sValue message
  */
 public final synchronized void setMessage(String pa_sValue) {jTextFieldProgressAction.setText(pa_sValue);  jTextFieldProgressAction.setCaretPosition(0);}

 
  
 /**
  * Sets the launch paramaters
  * 
  * @param pa_sValue launch paramaters
  */
 public final String getLaunchOptions() {return jTextFieldLaunchOptions.getText();}

 
 
 
 /**
  * Sets the launch paramaters
  * 
  * @param pa_sValue launch paramaters
  */
 public final void setLaunchOptions(String pa_sValue) {jTextFieldLaunchOptions.setText(pa_sValue);  jTextFieldLaunchOptions.setCaretPosition(0);}

 
 
 /**
  * Sets the launch binary information
  * 
  * @param pa_sValue   text
  */
 public final void setLaunchBinary(String pa_sValue) {jTextFieldBinary.setText(pa_sValue); jTextFieldBinary.setCaretPosition(0);}
 
 
  
 /**
  * Sets the launch binary information
  * 
  * @param pa_sValue   text
  */
 public final String getLaunchBinary() {return jTextFieldBinary.getText();}
 
 
 
 /**
  * Returns the active savegame screenshot displayed, may be null
  * 
  * @return  screenshot
  */
 public final java.awt.image.BufferedImage getActiveSaveGameScreenshot()
 {
  return (null != m_tActivePaintPanel ? m_tActivePaintPanel.getBufferedImage() : null);
 }
 
 
 /**
  * Sets the active save game screenshot
  * 
  * @param pa_tImage 
  */
 public final synchronized void setActiveSaveGameScreenshot(java.awt.image.BufferedImage pa_tImage)
 {
  if (null != m_tActivePaintPanel) m_tActivePaintPanel.setImage(pa_tImage); 
 }
 
 
 
 
 /**
  * Returns the active savegame screenshot displayed, may be null
  * 
  * @return  screenshot
  */
 public final java.awt.image.BufferedImage getBackupSaveGameScreenshot()
 {
  return (null != m_tBackupPaintPanel ? m_tBackupPaintPanel.getBufferedImage() : null);
 }
 
 
 
 
 /**
  * Sets the backup save game screenshot
  * 
  * @param pa_tImage 
  */
 public final synchronized void setBackupSaveGameScreenshot(java.awt.image.BufferedImage pa_tImage)
 {
  if (null != m_tBackupPaintPanel) m_tBackupPaintPanel.setImage(pa_tImage); 
 }
 

 
 
 /**
  * Initializes the current progress bar with the given maximum value and resets its display
  * 
  * @param pa_iMaxValue  maximum value
  */
 public final synchronized void initCurrentProgress(int pa_iMaxValue)
 {
  final int lc_iFugly = pa_iMaxValue;
  
  javax.swing.SwingUtilities.invokeLater(new Runnable()
  {
   public void run() 
   { 
    jProgressBarCurrent.setMinimum(0); 
    jProgressBarCurrent.setMaximum(lc_iFugly);
    jProgressBarCurrent.setValue(0);
   }
  });   
 }
  
  

 /**
  * Initializes the total progress bar with the given maximum value and resets its display
  * 
  * @param pa_iMaxValue  maximum value
  */
 public final synchronized void initTotalProgress(int pa_iMaxValue)
 {
  final int lc_iFugly = pa_iMaxValue;
  
  javax.swing.SwingUtilities.invokeLater(new Runnable()
  {
   public void run() 
   { 
    jProgressBarTotal.setMinimum(0); 
    jProgressBarTotal.setMaximum(lc_iFugly);
    jProgressBarTotal.setValue(0);
   }
  });   
 }

  
  
 /**
  * Sets the current progress string 
  * 
  * @param pa_sValue   string
  */
 public final synchronized void setStringCurrentProgress(String pa_sValue)
 {
  final String lc_sFugly = pa_sValue;
  
  javax.swing.SwingUtilities.invokeLater(new Runnable()
  {
   public void run() 
   { 
    jProgressBarCurrent.setString(lc_sFugly);
   }  
  }); 
 }
 
 
 
 
 /**
  * Sets the current progress to the given value
  * 
  * @param pa_iValue    progress value
  */
 public final synchronized void setCurrentProgress(int pa_iValue) 
 {
  final int lc_iFugly = pa_iValue;
  
  javax.swing.SwingUtilities.invokeLater(new Runnable()
  {
   public void run() 
   { 
    jProgressBarCurrent.setValue(lc_iFugly);
   }
  });   
 }
    
 

 /**
  * Sets the total progress to the given value
  * 
  * @param pa_iValue    progress value
  */
 public final synchronized void setTotalProgress(int pa_iValue) 
 {
  final int lc_iFugly = pa_iValue;
  
  javax.swing.SwingUtilities.invokeLater(new Runnable()
  {
   public void run() 
   { 
    jProgressBarTotal.setValue(lc_iFugly);
   }
  });   
 } 
  

    
 /**
  * Resets the progress display
  * 
  */
 public final synchronized void resetProgress()
 {
  javax.swing.SwingUtilities.invokeLater(new Runnable()
  {
   public void run() 
   { 
    setCurrentProgress(0);
    setTotalProgress(0);
   }
  });   
 }
  
  
 
 
 /**
  * Checks whether the active savegame list as an active multi-selection
  * 
  * @return  true, if a multi-seleciton is active, false otherwise
  */
 public synchronized final boolean hasListMultiSelectionActiveSaveGames()
 {
  return (1 < jTableActiveSaveGames.getSelectedRowCount());
 }

 
 
 /**
  * Checks whether the backup savegame list as an active multi-selection
  * 
  * @return  true, if a multi-seleciton is active, false otherwise
  */
 public synchronized final boolean hasListMultiSelectionBackupSaveGames()
 {
  return (1 < jTableBackupSaveGames.getSelectedRowCount());
 }
 
 
 
 /**
  * Checks whether the active player list as an active multi-selection
  * 
  * @return  true, if a multi-seleciton is active, false otherwise
  */
 public synchronized final boolean hasListMultiSelectionActivePlayers()
 {
  return  hasMultiSelection(jListActiveCharacters);
 }
 
 
 
 /**
  * Checks whether the backup player list as an active multi-selection
  * 
  * @return  true, if a multi-seleciton is active, false otherwise
  */
 public synchronized final boolean hasListMultiSelectionBackupPlayers()
 {
  return  hasMultiSelection(jListBackupCharacters);
 }
 
 
 
 /**
  * Returns the first selected active savegame
  * 
  * @return   savegame, may be null
  */
 public synchronized final SkyrimCharacterHelperSaveGame getFirstSelectedActiveSaveGame()
 {
  java.util.List<SkyrimCharacterHelperSaveGame> lc_tList = getSelectedActiveSaveGames();
  
  if (null != lc_tList)
  {
   if (false == lc_tList.isEmpty())
   {
    return lc_tList.get(0); 
   }
  }
  return null;
 }
 
 
 
 /**
  * Returns the first selected backup savegame
  * 
  * @return   savegame, may be null
  */
 public synchronized final SkyrimCharacterHelperSaveGame getFirstSelectedBackupSaveGame()
 {
  java.util.List<SkyrimCharacterHelperSaveGame> lc_tList = getSelectedBackupSaveGames();
  
  if (null != lc_tList)
  {
   if (false == lc_tList.isEmpty())
   {
    return lc_tList.get(0); 
   }
  }
  return null;
 }
 
 
 
 /**
  * Returns the first selected active player
  * 
  * @return   player, may be null
  */
 public final synchronized SkyrimCharacterHelperPlayer getFirstSelectedActivePlayer()
 {
  java.util.List<SkyrimCharacterHelperPlayer> lc_tList = getSelectedActivePlayers();
  
  if (null != lc_tList)
  {
   if (false == lc_tList.isEmpty())
   {
    return lc_tList.get(0); 
   }
  }
  return null;
 }
 
 
 
 /**
  * Returns the first selected backup player
  * 
  * @return   player, may be null
  */
 public final synchronized SkyrimCharacterHelperPlayer getFirstSelectedBackupPlayer()
 {
  java.util.List<SkyrimCharacterHelperPlayer> lc_tList = getSelectedBackupPlayers();
  
  if (null != lc_tList)
  {
   if (false == lc_tList.isEmpty())
   {
    return lc_tList.get(0); 
   }
  }
  return null;
 }
 
 
 
 /**
  * Clears the given list
  * 
  * @param pa_tList    list to clear
  * 
  */
 private synchronized void clearList(javax.swing.JList pa_tList)
 {
  SkyrimCharacterHelperListModel lc_tModel = (null != pa_tList ? (SkyrimCharacterHelperListModel) pa_tList.getModel() : null);
   
  pa_tList.getSelectionModel().clearSelection();
  
  if (null != lc_tModel)
  {
   if (0 < lc_tModel.size())
   {
    lc_tModel.clear();
   }
  }     
 }
 
 
 
 /**
  * Clears the given list
  * 
  * @param pa_tList    list to clear
  * 
  */
 private synchronized void clearTable(javax.swing.JTable pa_tTable)
 {
  final javax.swing.JTable lc_tTable = pa_tTable;
  
  javax.swing.SwingUtilities.invokeLater(new Runnable()
  {
   public void run() 
   { 
    SkyrimCharacterHelperTableModel lc_tModel = (null != lc_tTable ? (SkyrimCharacterHelperTableModel) lc_tTable.getModel() : null);
   
    lc_tTable.getSelectionModel().clearSelection();
    if (null != lc_tModel)
    {
     
     if (0 < lc_tModel.size())
     {
      lc_tModel.clear();
     }
    }     
   }
  });        
 }
 
 
 
 /**
   * Checks whether the given string is non-null and non-empty
   * 
   * @param pa_sValue   string
   * @return  true, if he given string is non-null and non-empty, false otherwise
   */
 private boolean checkString(String pa_sValue)
 {
  if (null != pa_sValue)
  {
   if (0 < pa_sValue.length()) 
   {
    return true; 
   }
  }
  return false; 
 }
 

 
 /**
  * Checks whether the given list has an active multi-selection
  * 
  * @param pa_tList   list
  * @return   true, if a multi-seleciton is active, false otherwise
  */
 private boolean hasMultiSelection(javax.swing.JList pa_tList)
 {
  int[] lc_iIndices = null;
  
  if (null != pa_tList)
  {
   if (null != (lc_iIndices = pa_tList.getSelectedIndices()))
   {
    return (1 < lc_iIndices.length); 
   }
  }
  return false;
 }
 

 
 
 /**
  * Returns the size of the given list
  * 
  * @param pa_tList  list
  * @return  size, may be 0
  */
 private int getListSize(javax.swing.JList pa_tList)
 {
  SkyrimCharacterHelperListModel lc_tModel = (null != pa_tList ? (SkyrimCharacterHelperListModel) pa_tList.getModel() : null);
   
  if (null != lc_tModel)
  {
   return (lc_tModel.getSize());
  }
  return 0;
 }
 
 
 

 
 
 /**
  * Returns the size of the given list
  * 
  * @param pa_tList  list
  * @return  size, may be 0
  */
 private int getTableSize(javax.swing.JTable pa_tTable)
 {
  javax.swing.table.AbstractTableModel lc_tModel = (null != pa_tTable ? (javax.swing.table.AbstractTableModel) pa_tTable.getModel() : null);
   
  if (null != lc_tModel)
  {
   return (lc_tModel.getRowCount());
  }
  return 0;
 }
 
 
 
 
 /**
  * Removes the given player from the given list 
  * 
  * @param pa_tList       list
  * @param pa_tCharacter   player   
  */
 private synchronized int removePlayerListEntry(javax.swing.JList pa_tList, SkyrimCharacterHelperPlayer pa_tCharacter)
 {
  int lc_iPos  = (null != pa_tList ? pa_tList.getSelectedIndex() : -1);
  int lc_iSize = 0;
  
  SkyrimCharacterHelperListModel lc_tModel = (null != pa_tList ? (SkyrimCharacterHelperListModel) pa_tList.getModel() : null);

  if (null != pa_tCharacter && null != pa_tList)
  {
   if (null != lc_tModel)
   {
    lc_tModel.removeElement(pa_tCharacter); 
    lc_iSize = lc_tModel.size();
   }
  }   
  if (0 == lc_iSize) lc_iPos = -1;
  else if (lc_iPos >= lc_iSize) lc_iPos = lc_iSize-1; 

  return lc_iPos;
 }
 
 
 
 /**
  * Removes the given savegame from the given list
  * 
  * @param pa_tList       list
  * @param pa_tSaveGame   savegame   
  */
 private synchronized int removeSaveGameListEntry(javax.swing.JTable pa_tTable, SkyrimCharacterHelperSaveGame pa_tSaveGame)
 {
  int lc_iModelIndex = -1;
  int lc_iViewIndex  = -1;
  int lc_iSize       = 0;
  
  SkyrimCharacterHelperTableModel lc_tModel  = (null != pa_tTable ? (SkyrimCharacterHelperTableModel) pa_tTable.getModel() : null);

  if (null != pa_tTable && null != pa_tSaveGame)
  {
   if (null != lc_tModel)
   {
    if (-1 != (lc_iModelIndex = lc_tModel.indexOf(pa_tSaveGame)))
    {
     if (-1 != (lc_iViewIndex = pa_tTable.convertRowIndexToView(lc_iModelIndex)))
     {
      if (true == pa_tTable.isRowSelected(lc_iViewIndex))
      {
       pa_tTable.getSelectionModel().removeSelectionInterval(lc_iViewIndex, lc_iViewIndex); // we need to remove the selection!
      }
     }
     lc_tModel.removeElement(pa_tSaveGame);
     lc_iSize = lc_tModel.size();
    }
   }
  }
  if (0 == lc_iSize) lc_iViewIndex = -1;
  else if (lc_iViewIndex >= lc_iSize) lc_iViewIndex = lc_iSize-1; 

  return lc_iViewIndex;
 }
 
 
 
 
 /** This method is called from within the constructor to
  * initialize the form.
  * WARNING: Do NOT modify this code. The content of this method is
  * always regenerated by the Form Editor.
  */
 @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        jPanelProgress = new javax.swing.JPanel();
        jLabelProgressTotal = new javax.swing.JLabel();
        jLabelProgressCurrent = new javax.swing.JLabel();
        jProgressBarCurrent = new javax.swing.JProgressBar();
        jProgressBarTotal = new javax.swing.JProgressBar();
        jTextFieldProgressAction = new javax.swing.JTextField();
        jLabelProgressAction = new javax.swing.JLabel();
        jPanelControl = new javax.swing.JPanel();
        jComboBoxMode = new javax.swing.JComboBox();
        jButtonProgressExit = new javax.swing.JButton();
        jButtonProgressCancel = new javax.swing.JButton();
        jLabelConfirm = new javax.swing.JLabel();
        jComboBoxConfirmation = new javax.swing.JComboBox();
        jLabelMode = new javax.swing.JLabel();
        jLabelOverwrite = new javax.swing.JLabel();
        jComboBoxOverwrite = new javax.swing.JComboBox();
        jButtonAbout = new javax.swing.JButton();
        jLabelBinaryPath = new javax.swing.JLabel();
        jTextFieldBinary = new javax.swing.JTextField();
        jButtonChooseBinary = new javax.swing.JButton();
        jButtonRestoreRegistryPath = new javax.swing.JButton();
        jComboBoxMinimize = new javax.swing.JComboBox();
        jButtonLaunchSolo = new javax.swing.JButton();
        jLabelImageMode = new javax.swing.JLabel();
        jComboBoxImageMode = new javax.swing.JComboBox();
        jLabelQuickSave = new javax.swing.JLabel();
        jLabelAutoSaves = new javax.swing.JLabel();
        jLabelPrefix = new javax.swing.JLabel();
        jComboBoxAutoSaves = new javax.swing.JComboBox();
        jComboBoxQuickSaves = new javax.swing.JComboBox();
        jTextFieldPrefix = new javax.swing.JTextField();
        filler31 = new javax.swing.Box.Filler(new java.awt.Dimension(0, 0), new java.awt.Dimension(0, 0), new java.awt.Dimension(32767, 0));
        jLabelLaunchTarget = new javax.swing.JLabel();
        jLabelOnSkyrim = new javax.swing.JLabel();
        jLabelLaunchOptions = new javax.swing.JLabel();
        jTextFieldLaunchOptions = new javax.swing.JTextField();
        jLabelInitMode = new javax.swing.JLabel();
        jComboBoxInitMode = new javax.swing.JComboBox();
        filler21 = new javax.swing.Box.Filler(new java.awt.Dimension(0, 0), new java.awt.Dimension(0, 0), new java.awt.Dimension(32767, 0));
        jPanelBackup = new javax.swing.JPanel();
        jScrollPaneBackupCharacters = new javax.swing.JScrollPane();
        jListBackupCharacters = new javax.swing.JList();
        jButtonBackupCharacterRestore = new javax.swing.JButton();
        jButtonBackupCharacterDelete = new javax.swing.JButton();
        jButtonBackupCharacterClear = new javax.swing.JButton();
        jButtonBackupRescan = new javax.swing.JButton();
        jButtonBackupPath = new javax.swing.JButton();
        jScrollPaneBackupSaveFiles = new javax.swing.JScrollPane();
        jTableBackupSaveGames = new javax.swing.JTable();
        jTextFieldBackupPath = new javax.swing.JTextField();
        jButtonBackupSaveGameRestore = new javax.swing.JButton();
        jButtonBackupSaveGameDelete = new javax.swing.JButton();
        jPanelBackupCharacterData = new javax.swing.JPanel();
        jButtonBackupClipboard = new javax.swing.JButton();
        jTextFieldAmountBackup = new javax.swing.JTextField();
        jButtonBackupCharacterLaunch = new javax.swing.JButton();
        jTextFieldRaceBackupPlayer = new javax.swing.JTextField();
        jButtonBackupLaunchSavegame = new javax.swing.JButton();
        jPanelActive = new javax.swing.JPanel();
        jScrollPaneActiveCharacters = new javax.swing.JScrollPane();
        jListActiveCharacters = new javax.swing.JList();
        jButtonActiveCharacterBackup = new javax.swing.JButton();
        jButtonActiveCharacterDelete = new javax.swing.JButton();
        jButtonActiveCharacterClear = new javax.swing.JButton();
        jButtonActiveRescan = new javax.swing.JButton();
        jButtonActivePath = new javax.swing.JButton();
        jScrollPaneActiveSaveFiles = new javax.swing.JScrollPane();
        jTableActiveSaveGames = new javax.swing.JTable();
        jTextFieldActivePath = new javax.swing.JTextField();
        jButtonActiveSaveGameBackup = new javax.swing.JButton();
        jButtonActiveSaveGameDelete = new javax.swing.JButton();
        jPanelActiveCharacterData = new javax.swing.JPanel();
        jButtonActiveClipboard = new javax.swing.JButton();
        jTextFieldAmountActive = new javax.swing.JTextField();
        jButtonActiveCharacterLaunch = new javax.swing.JButton();
        jTextFieldRaceActivePlayer = new javax.swing.JTextField();
        jButtonActiveLaunchSavegame = new javax.swing.JButton();

        setDefaultCloseOperation(javax.swing.WindowConstants.DO_NOTHING_ON_CLOSE);
        java.util.ResourceBundle bundle = java.util.ResourceBundle.getBundle("skyrimcharacterhelper/resources/SkyrimCharacterHelperView"); // NOI18N
        setTitle(bundle.getString("Form.title")); // NOI18N
        setName("Form"); // NOI18N
        getContentPane().setLayout(new java.awt.GridBagLayout());

        jPanelProgress.setBorder(javax.swing.BorderFactory.createEtchedBorder());
        jPanelProgress.setAlignmentX(1.0F);
        jPanelProgress.setAlignmentY(0.0F);
        jPanelProgress.setName("jPanelProgress"); // NOI18N

        jLabelProgressTotal.setLabelFor(jProgressBarTotal);
        jLabelProgressTotal.setText(bundle.getString("jLabelProgressTotal.text")); // NOI18N
        jLabelProgressTotal.setName("jLabelProgressTotal"); // NOI18N

        jLabelProgressCurrent.setLabelFor(jProgressBarCurrent);
        jLabelProgressCurrent.setText(bundle.getString("jLabelProgressCurrent.text")); // NOI18N
        jLabelProgressCurrent.setName("jLabelProgressCurrent"); // NOI18N

        jProgressBarCurrent.setDoubleBuffered(true);
        jProgressBarCurrent.setName("jProgressBarCurrent"); // NOI18N
        jProgressBarCurrent.setStringPainted(true);

        jProgressBarTotal.setDoubleBuffered(true);
        jProgressBarTotal.setName("jProgressBarTotal"); // NOI18N
        jProgressBarTotal.setStringPainted(true);

        jTextFieldProgressAction.setBackground(new java.awt.Color(Integer.valueOf(bundle.getString("displayTextfield.backgroundColor"), 16)));
        jTextFieldProgressAction.setEditable(false);
        jTextFieldProgressAction.setText(bundle.getString("jTextFieldProgressAction.text")); // NOI18N
        jTextFieldProgressAction.setName("jTextFieldProgressAction"); // NOI18N

        jLabelProgressAction.setLabelFor(jTextFieldProgressAction);
        jLabelProgressAction.setText(bundle.getString("jLabelProgressAction.text")); // NOI18N
        jLabelProgressAction.setName("jLabelProgressAction"); // NOI18N

        javax.swing.GroupLayout jPanelProgressLayout = new javax.swing.GroupLayout(jPanelProgress);
        jPanelProgress.setLayout(jPanelProgressLayout);
        jPanelProgressLayout.setHorizontalGroup(
            jPanelProgressLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(jPanelProgressLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(jPanelProgressLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(jPanelProgressLayout.createSequentialGroup()
                        .addComponent(jLabelProgressAction)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(jTextFieldProgressAction, javax.swing.GroupLayout.DEFAULT_SIZE, 1216, Short.MAX_VALUE))
                    .addGroup(jPanelProgressLayout.createSequentialGroup()
                        .addGroup(jPanelProgressLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addComponent(jLabelProgressCurrent)
                            .addComponent(jLabelProgressTotal))
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addGroup(jPanelProgressLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addComponent(jProgressBarTotal, javax.swing.GroupLayout.DEFAULT_SIZE, 1216, Short.MAX_VALUE)
                            .addComponent(jProgressBarCurrent, javax.swing.GroupLayout.DEFAULT_SIZE, 1216, Short.MAX_VALUE))))
                .addContainerGap())
        );

        jPanelProgressLayout.linkSize(javax.swing.SwingConstants.HORIZONTAL, new java.awt.Component[] {jLabelProgressAction, jLabelProgressCurrent, jLabelProgressTotal});

        jPanelProgressLayout.setVerticalGroup(
            jPanelProgressLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(jPanelProgressLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(jPanelProgressLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(jLabelProgressCurrent)
                    .addComponent(jProgressBarCurrent, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(jPanelProgressLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(jLabelProgressTotal)
                    .addComponent(jProgressBarTotal, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(jPanelProgressLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(jTextFieldProgressAction, javax.swing.GroupLayout.PREFERRED_SIZE, 29, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(jLabelProgressAction))
                .addContainerGap())
        );

        jPanelProgressLayout.linkSize(javax.swing.SwingConstants.VERTICAL, new java.awt.Component[] {jLabelProgressAction, jLabelProgressCurrent, jLabelProgressTotal, jProgressBarCurrent, jProgressBarTotal, jTextFieldProgressAction});

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(18, 10, 0, 10);
        getContentPane().add(jPanelProgress, gridBagConstraints);

        jPanelControl.setBorder(javax.swing.BorderFactory.createEtchedBorder());
        jPanelControl.setAlignmentX(1.0F);
        jPanelControl.setAlignmentY(0.0F);
        jPanelControl.setName("jPanelControl"); // NOI18N
        jPanelControl.setLayout(new java.awt.GridBagLayout());

        jComboBoxMode.setMaximumSize(new java.awt.Dimension(32767, 29));
        jComboBoxMode.setMinimumSize(new java.awt.Dimension(97, 29));
        jComboBoxMode.setName("jComboBoxMode"); // NOI18N
        jComboBoxMode.setPreferredSize(new java.awt.Dimension(97, 29));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridwidth = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(10, 5, 10, 0);
        jPanelControl.add(jComboBoxMode, gridBagConstraints);

        org.jdesktop.application.ResourceMap resourceMap = org.jdesktop.application.Application.getInstance(skyrimcharacterhelper.SkyrimCharacterHelperApp.class).getContext().getResourceMap(SkyrimCharacterHelperView.class);
        jButtonProgressExit.setIcon(resourceMap.getIcon("jButtonProgressExit.icon")); // NOI18N
        jButtonProgressExit.setText(bundle.getString("jButtonProgressExit.text")); // NOI18N
        jButtonProgressExit.setToolTipText(bundle.getString("jButtonProgressExit.toolTipText")); // NOI18N
        jButtonProgressExit.setMaximumSize(new java.awt.Dimension(29, 29));
        jButtonProgressExit.setMinimumSize(new java.awt.Dimension(29, 29));
        jButtonProgressExit.setName("jButtonProgressExit"); // NOI18N
        jButtonProgressExit.setPreferredSize(new java.awt.Dimension(29, 29));
        jButtonProgressExit.setRolloverIcon(resourceMap.getIcon("jButtonProgressExit.rolloverIcon")); // NOI18N
        jButtonProgressExit.setRolloverSelectedIcon(resourceMap.getIcon("jButtonProgressExit.rolloverSelectedIcon")); // NOI18N
        jButtonProgressExit.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                actionPerformedButtonProcessExit(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 16;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHEAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 5, 10, 5);
        jPanelControl.add(jButtonProgressExit, gridBagConstraints);

        jButtonProgressCancel.setIcon(resourceMap.getIcon("jButtonProgressCancel.icon")); // NOI18N
        jButtonProgressCancel.setText(bundle.getString("jButtonProgressCancel.text")); // NOI18N
        jButtonProgressCancel.setToolTipText(bundle.getString("jButtonProgressCancel.toolTipText")); // NOI18N
        jButtonProgressCancel.setEnabled(false);
        jButtonProgressCancel.setMaximumSize(new java.awt.Dimension(29, 29));
        jButtonProgressCancel.setMinimumSize(new java.awt.Dimension(29, 29));
        jButtonProgressCancel.setName("jButtonProgressCancel"); // NOI18N
        jButtonProgressCancel.setPreferredSize(new java.awt.Dimension(29, 29));
        jButtonProgressCancel.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                actionPerformedButtonProcessCancel(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 18;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHEAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 10, 10);
        jPanelControl.add(jButtonProgressCancel, gridBagConstraints);

        jLabelConfirm.setLabelFor(jComboBoxConfirmation);
        jLabelConfirm.setText(bundle.getString("jLabelConfirm.text")); // NOI18N
        jLabelConfirm.setName("jLabelConfirm"); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 6;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.ipady = 15;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(10, 20, 10, 0);
        jPanelControl.add(jLabelConfirm, gridBagConstraints);

        jComboBoxConfirmation.setMaximumSize(new java.awt.Dimension(32767, 29));
        jComboBoxConfirmation.setMinimumSize(new java.awt.Dimension(57, 29));
        jComboBoxConfirmation.setName("jComboBoxConfirmation"); // NOI18N
        jComboBoxConfirmation.setPreferredSize(new java.awt.Dimension(57, 29));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 7;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(10, 5, 10, 0);
        jPanelControl.add(jComboBoxConfirmation, gridBagConstraints);

        jLabelMode.setLabelFor(jComboBoxMode);
        jLabelMode.setText(bundle.getString("jLabelMode.text")); // NOI18N
        jLabelMode.setName("jLabelMode"); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.ipady = 15;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(10, 10, 10, 0);
        jPanelControl.add(jLabelMode, gridBagConstraints);

        jLabelOverwrite.setLabelFor(jComboBoxOverwrite);
        jLabelOverwrite.setText(bundle.getString("jLabelOverwrite.text")); // NOI18N
        jLabelOverwrite.setName("jLabelOverwrite"); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 6;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.ipady = 15;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 20, 10, 0);
        jPanelControl.add(jLabelOverwrite, gridBagConstraints);

        jComboBoxOverwrite.setMaximumSize(new java.awt.Dimension(32767, 29));
        jComboBoxOverwrite.setMinimumSize(new java.awt.Dimension(57, 29));
        jComboBoxOverwrite.setName("jComboBoxOverwrite"); // NOI18N
        jComboBoxOverwrite.setPreferredSize(new java.awt.Dimension(57, 29));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 7;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 5, 10, 0);
        jPanelControl.add(jComboBoxOverwrite, gridBagConstraints);

        jButtonAbout.setIcon(resourceMap.getIcon("jButtonAbout.icon")); // NOI18N
        jButtonAbout.setText(bundle.getString("jButtonAbout.text")); // NOI18N
        jButtonAbout.setToolTipText(bundle.getString("jButtonAbout.toolTipText")); // NOI18N
        jButtonAbout.setMaximumSize(new java.awt.Dimension(29, 29));
        jButtonAbout.setMinimumSize(new java.awt.Dimension(29, 29));
        jButtonAbout.setName("jButtonAbout"); // NOI18N
        jButtonAbout.setPreferredSize(new java.awt.Dimension(29, 29));
        jButtonAbout.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                actionPerformedButtonAbout(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 17;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHEAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 10, 5);
        jPanelControl.add(jButtonAbout, gridBagConstraints);

        jLabelBinaryPath.setText(bundle.getString("jLabelBinaryPath.text")); // NOI18N
        jLabelBinaryPath.setName("jLabelBinaryPath"); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 4;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.ipady = 15;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 20, 10, 0);
        jPanelControl.add(jLabelBinaryPath, gridBagConstraints);

        jTextFieldBinary.setBackground(new java.awt.Color(Integer.valueOf(bundle.getString("displayTextfield.backgroundColor"), 16)));
        jTextFieldBinary.setEditable(false);
        jTextFieldBinary.setText(bundle.getString("jTextFieldBinary.text")); // NOI18N
        jTextFieldBinary.setMaximumSize(new java.awt.Dimension(2147483647, 29));
        jTextFieldBinary.setMinimumSize(new java.awt.Dimension(6, 29));
        jTextFieldBinary.setName("jTextFieldBinary"); // NOI18N
        jTextFieldBinary.setPreferredSize(new java.awt.Dimension(6, 29));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 4;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.gridwidth = 6;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 0.25;
        gridBagConstraints.insets = new java.awt.Insets(0, 5, 10, 0);
        jPanelControl.add(jTextFieldBinary, gridBagConstraints);

        jButtonChooseBinary.setIcon(resourceMap.getIcon("jButtonChooseBinary.icon")); // NOI18N
        jButtonChooseBinary.setText(bundle.getString("jButtonChooseBinary.text")); // NOI18N
        jButtonChooseBinary.setToolTipText(bundle.getString("jButtonChooseBinary.toolTipText")); // NOI18N
        jButtonChooseBinary.setMaximumSize(new java.awt.Dimension(29, 29));
        jButtonChooseBinary.setMinimumSize(new java.awt.Dimension(29, 29));
        jButtonChooseBinary.setName("jButtonChooseBinary"); // NOI18N
        jButtonChooseBinary.setPreferredSize(new java.awt.Dimension(29, 29));
        jButtonChooseBinary.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                actionPerformedButtonChooseBinary(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 10, 0);
        jPanelControl.add(jButtonChooseBinary, gridBagConstraints);

        jButtonRestoreRegistryPath.setIcon(resourceMap.getIcon("jButtonRestoreRegistryPath.icon")); // NOI18N
        jButtonRestoreRegistryPath.setText(bundle.getString("jButtonRestoreRegistryPath.text")); // NOI18N
        jButtonRestoreRegistryPath.setToolTipText(bundle.getString("jButtonRestoreRegistryPath.toolTipText")); // NOI18N
        jButtonRestoreRegistryPath.setMaximumSize(new java.awt.Dimension(29, 29));
        jButtonRestoreRegistryPath.setMinimumSize(new java.awt.Dimension(29, 29));
        jButtonRestoreRegistryPath.setName("jButtonRestoreRegistryPath"); // NOI18N
        jButtonRestoreRegistryPath.setPreferredSize(new java.awt.Dimension(29, 29));
        jButtonRestoreRegistryPath.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                actionPerformedButtonRestoreRegistry(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 10, 5);
        jPanelControl.add(jButtonRestoreRegistryPath, gridBagConstraints);

        jComboBoxMinimize.setMaximumSize(new java.awt.Dimension(32767, 29));
        jComboBoxMinimize.setMinimumSize(new java.awt.Dimension(100, 29));
        jComboBoxMinimize.setName("jComboBoxMinimize"); // NOI18N
        jComboBoxMinimize.setPreferredSize(new java.awt.Dimension(100, 29));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 5;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 0.25;
        gridBagConstraints.insets = new java.awt.Insets(0, 5, 10, 0);
        jPanelControl.add(jComboBoxMinimize, gridBagConstraints);

        jButtonLaunchSolo.setIcon(resourceMap.getIcon("jButtonLaunchSolo.icon")); // NOI18N
        jButtonLaunchSolo.setText(bundle.getString("jButtonLaunchSolo.text")); // NOI18N
        jButtonLaunchSolo.setToolTipText(bundle.getString("jButtonLaunchSolo.toolTipText")); // NOI18N
        jButtonLaunchSolo.setEnabled(false);
        jButtonLaunchSolo.setMaximumSize(new java.awt.Dimension(29, 29));
        jButtonLaunchSolo.setMinimumSize(new java.awt.Dimension(29, 29));
        jButtonLaunchSolo.setName("jButtonLaunchSolo"); // NOI18N
        jButtonLaunchSolo.setPreferredSize(new java.awt.Dimension(29, 29));
        jButtonLaunchSolo.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                actionPerformedButtonLaunchSolo(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 5, 10, 5);
        jPanelControl.add(jButtonLaunchSolo, gridBagConstraints);

        jLabelImageMode.setText(bundle.getString("jLabelImageMode.text")); // NOI18N
        jLabelImageMode.setName("jLabelImageMode"); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.ipady = 15;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 10, 10, 0);
        jPanelControl.add(jLabelImageMode, gridBagConstraints);

        jComboBoxImageMode.setMaximumSize(new java.awt.Dimension(32767, 29));
        jComboBoxImageMode.setMinimumSize(new java.awt.Dimension(97, 29));
        jComboBoxImageMode.setName("jComboBoxImageMode"); // NOI18N
        jComboBoxImageMode.setPreferredSize(new java.awt.Dimension(97, 29));
        jComboBoxImageMode.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                actionPerformedComboBoxImageMode(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.gridwidth = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 5, 10, 0);
        jPanelControl.add(jComboBoxImageMode, gridBagConstraints);

        jLabelQuickSave.setText(bundle.getString("jLabelQuickSave.text")); // NOI18N
        jLabelQuickSave.setMaximumSize(new java.awt.Dimension(125, 144));
        jLabelQuickSave.setMinimumSize(new java.awt.Dimension(120, 14));
        jLabelQuickSave.setName("jLabelQuickSave"); // NOI18N
        jLabelQuickSave.setPreferredSize(new java.awt.Dimension(120, 14));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 8;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.ipady = 15;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 20, 10, 0);
        jPanelControl.add(jLabelQuickSave, gridBagConstraints);

        jLabelAutoSaves.setText(bundle.getString("jLabelAutoSaves.text")); // NOI18N
        jLabelAutoSaves.setMaximumSize(new java.awt.Dimension(125, 14));
        jLabelAutoSaves.setName("jLabelAutoSaves"); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 8;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.ipady = 15;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(10, 20, 10, 0);
        jPanelControl.add(jLabelAutoSaves, gridBagConstraints);

        jLabelPrefix.setText(bundle.getString("jLabelPrefix.text")); // NOI18N
        jLabelPrefix.setName("jLabelPrefix"); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 11;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.ipady = 15;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(10, 5, 10, 0);
        jPanelControl.add(jLabelPrefix, gridBagConstraints);

        jComboBoxAutoSaves.setMaximumSize(new java.awt.Dimension(32767, 29));
        jComboBoxAutoSaves.setMinimumSize(new java.awt.Dimension(57, 29));
        jComboBoxAutoSaves.setName("jComboBoxAutoSaves"); // NOI18N
        jComboBoxAutoSaves.setPreferredSize(new java.awt.Dimension(57, 29));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 9;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(10, 5, 10, 0);
        jPanelControl.add(jComboBoxAutoSaves, gridBagConstraints);

        jComboBoxQuickSaves.setMaximumSize(new java.awt.Dimension(32767, 29));
        jComboBoxQuickSaves.setMinimumSize(new java.awt.Dimension(57, 29));
        jComboBoxQuickSaves.setName("jComboBoxQuickSaves"); // NOI18N
        jComboBoxQuickSaves.setPreferredSize(new java.awt.Dimension(57, 29));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 9;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 5, 10, 0);
        jPanelControl.add(jComboBoxQuickSaves, gridBagConstraints);

        jTextFieldPrefix.setText(bundle.getString("jTextFieldPrefix.text")); // NOI18N
        jTextFieldPrefix.setMaximumSize(new java.awt.Dimension(2147483647, 29));
        jTextFieldPrefix.setMinimumSize(new java.awt.Dimension(60, 29));
        jTextFieldPrefix.setName("jTextFieldPrefix"); // NOI18N
        jTextFieldPrefix.setPreferredSize(new java.awt.Dimension(80, 29));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 12;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridwidth = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 0.25;
        gridBagConstraints.insets = new java.awt.Insets(10, 5, 10, 0);
        jPanelControl.add(jTextFieldPrefix, gridBagConstraints);

        filler31.setName("filler31"); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 15;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.ipadx = 50;
        jPanelControl.add(filler31, gridBagConstraints);

        jLabelLaunchTarget.setText(bundle.getString("jLabelLaunchTarget.text")); // NOI18N
        jLabelLaunchTarget.setName("jLabelLaunchTarget"); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.ipady = 15;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 10, 10, 0);
        jPanelControl.add(jLabelLaunchTarget, gridBagConstraints);

        jLabelOnSkyrim.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
        jLabelOnSkyrim.setIcon(resourceMap.getIcon("jLabelOnSkyrim.icon")); // NOI18N
        jLabelOnSkyrim.setText(bundle.getString("jLabelOnSkyrim.text")); // NOI18N
        jLabelOnSkyrim.setBorder(javax.swing.BorderFactory.createEtchedBorder());
        jLabelOnSkyrim.setDisabledIcon(resourceMap.getIcon("jLabelOnSkyrim.disabledIcon")); // NOI18N
        jLabelOnSkyrim.setEnabled(false);
        jLabelOnSkyrim.setHorizontalTextPosition(javax.swing.SwingConstants.RIGHT);
        jLabelOnSkyrim.setIconTextGap(15);
        jLabelOnSkyrim.setMaximumSize(new java.awt.Dimension(5000, 29));
        jLabelOnSkyrim.setMinimumSize(new java.awt.Dimension(82, 29));
        jLabelOnSkyrim.setName("jLabelOnSkyrim"); // NOI18N
        jLabelOnSkyrim.setOpaque(true);
        jLabelOnSkyrim.setPreferredSize(new java.awt.Dimension(79, 29));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 11;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.gridwidth = 4;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 0.25;
        gridBagConstraints.insets = new java.awt.Insets(0, 5, 10, 0);
        jPanelControl.add(jLabelOnSkyrim, gridBagConstraints);

        jLabelLaunchOptions.setText(bundle.getString("jLabelLaunchOptions.text")); // NOI18N
        jLabelLaunchOptions.setName("jLabelLaunchOptions"); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 11;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.ipady = 15;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 5, 10, 0);
        jPanelControl.add(jLabelLaunchOptions, gridBagConstraints);

        jTextFieldLaunchOptions.setText(bundle.getString("jTextFieldLaunchOptions.text")); // NOI18N
        jTextFieldLaunchOptions.setMaximumSize(new java.awt.Dimension(2147483647, 29));
        jTextFieldLaunchOptions.setMinimumSize(new java.awt.Dimension(60, 29));
        jTextFieldLaunchOptions.setName("jTextFieldLaunchOptions"); // NOI18N
        jTextFieldLaunchOptions.setPreferredSize(new java.awt.Dimension(60, 29));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 12;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.gridwidth = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 0.25;
        gridBagConstraints.insets = new java.awt.Insets(0, 5, 10, 0);
        jPanelControl.add(jTextFieldLaunchOptions, gridBagConstraints);

        jLabelInitMode.setText(bundle.getString("jLabelInitMode.text")); // NOI18N
        jLabelInitMode.setMaximumSize(new java.awt.Dimension(63, 14));
        jLabelInitMode.setMinimumSize(new java.awt.Dimension(63, 14));
        jLabelInitMode.setName("jLabelInitMode"); // NOI18N
        jLabelInitMode.setPreferredSize(new java.awt.Dimension(63, 14));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 4;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.ipady = 15;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(10, 20, 10, 0);
        jPanelControl.add(jLabelInitMode, gridBagConstraints);

        jComboBoxInitMode.setFocusCycleRoot(true);
        jComboBoxInitMode.setMaximumSize(new java.awt.Dimension(32767, 29));
        jComboBoxInitMode.setMinimumSize(new java.awt.Dimension(100, 29));
        jComboBoxInitMode.setName("jComboBoxInitMode"); // NOI18N
        jComboBoxInitMode.setPreferredSize(new java.awt.Dimension(100, 29));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 5;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 0.25;
        gridBagConstraints.insets = new java.awt.Insets(10, 5, 10, 0);
        jPanelControl.add(jComboBoxInitMode, gridBagConstraints);

        filler21.setName("filler21"); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 10;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.ipadx = 50;
        jPanelControl.add(filler21, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(18, 10, 11, 10);
        getContentPane().add(jPanelControl, gridBagConstraints);

        jPanelBackup.setBorder(javax.swing.BorderFactory.createEtchedBorder());
        jPanelBackup.setAlignmentY(1.0F);
        jPanelBackup.setMinimumSize(new java.awt.Dimension(496, 300));
        jPanelBackup.setName("jPanelBackup"); // NOI18N
        jPanelBackup.setPreferredSize(new java.awt.Dimension(819, 300));
        jPanelBackup.setLayout(new java.awt.GridBagLayout());

        jScrollPaneBackupCharacters.setMaximumSize(new java.awt.Dimension(165, 32767));
        jScrollPaneBackupCharacters.setMinimumSize(new java.awt.Dimension(165, 130));
        jScrollPaneBackupCharacters.setName("jScrollPaneBackupCharacters"); // NOI18N
        jScrollPaneBackupCharacters.setPreferredSize(new java.awt.Dimension(165, 130));

        jListBackupCharacters.setBackground(new java.awt.Color(Integer.valueOf(bundle.getString("jList.background"), 16)));
        jListBackupCharacters.setForeground(new java.awt.Color(Integer.valueOf(bundle.getString("jList.foreground"), 16)));
        jListBackupCharacters.setAlignmentX(0.2F);
        jListBackupCharacters.setFocusable(false);
        jListBackupCharacters.setName("jListBackupCharacters"); // NOI18N
        jListBackupCharacters.setSelectionBackground(new java.awt.Color(Integer.valueOf(bundle.getString("jList.selectionBackground"), 16)));
        jListBackupCharacters.setSelectionForeground(new java.awt.Color(Integer.valueOf(bundle.getString("jList.selectionForeground"), 16)));
        jScrollPaneBackupCharacters.setViewportView(jListBackupCharacters);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridwidth = 6;
        gridBagConstraints.gridheight = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weighty = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(10, 10, 10, 10);
        jPanelBackup.add(jScrollPaneBackupCharacters, gridBagConstraints);

        jButtonBackupCharacterRestore.setIcon(resourceMap.getIcon("jButtonBackupCharacterRestore.icon")); // NOI18N
        jButtonBackupCharacterRestore.setText(bundle.getString("jButtonBackupCharacterRestore.text")); // NOI18N
        jButtonBackupCharacterRestore.setToolTipText(bundle.getString("jButtonBackupCharacterRestore.toolTipText")); // NOI18N
        jButtonBackupCharacterRestore.setEnabled(false);
        jButtonBackupCharacterRestore.setMaximumSize(new java.awt.Dimension(29, 29));
        jButtonBackupCharacterRestore.setMinimumSize(new java.awt.Dimension(29, 29));
        jButtonBackupCharacterRestore.setName("jButtonBackupCharacterRestore"); // NOI18N
        jButtonBackupCharacterRestore.setPreferredSize(new java.awt.Dimension(29, 29));
        jButtonBackupCharacterRestore.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                actionPerformedButtonBackupPlayerRestore(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 10, 5);
        jPanelBackup.add(jButtonBackupCharacterRestore, gridBagConstraints);

        jButtonBackupCharacterDelete.setIcon(resourceMap.getIcon("jButtonBackupCharacterDelete.icon")); // NOI18N
        jButtonBackupCharacterDelete.setText(bundle.getString("jButtonBackupCharacterDelete.text")); // NOI18N
        jButtonBackupCharacterDelete.setToolTipText(bundle.getString("jButtonBackupCharacterDelete.toolTipText")); // NOI18N
        jButtonBackupCharacterDelete.setEnabled(false);
        jButtonBackupCharacterDelete.setMaximumSize(new java.awt.Dimension(29, 29));
        jButtonBackupCharacterDelete.setMinimumSize(new java.awt.Dimension(29, 29));
        jButtonBackupCharacterDelete.setName("jButtonBackupCharacterDelete"); // NOI18N
        jButtonBackupCharacterDelete.setPreferredSize(new java.awt.Dimension(29, 29));
        jButtonBackupCharacterDelete.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                actionPerformedButtonBackupDelete(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 10, 5);
        jPanelBackup.add(jButtonBackupCharacterDelete, gridBagConstraints);

        jButtonBackupCharacterClear.setIcon(resourceMap.getIcon("jButtonBackupCharacterClear.icon")); // NOI18N
        jButtonBackupCharacterClear.setText(bundle.getString("jButtonBackupCharacterClear.text")); // NOI18N
        jButtonBackupCharacterClear.setToolTipText(bundle.getString("jButtonBackupCharacterClear.toolTipText")); // NOI18N
        jButtonBackupCharacterClear.setEnabled(false);
        jButtonBackupCharacterClear.setMaximumSize(new java.awt.Dimension(29, 29));
        jButtonBackupCharacterClear.setMinimumSize(new java.awt.Dimension(29, 29));
        jButtonBackupCharacterClear.setName("jButtonBackupCharacterClear"); // NOI18N
        jButtonBackupCharacterClear.setPreferredSize(new java.awt.Dimension(29, 29));
        jButtonBackupCharacterClear.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                actionPerformedButtonBackupClear(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 10, 5);
        jPanelBackup.add(jButtonBackupCharacterClear, gridBagConstraints);

        jButtonBackupRescan.setIcon(resourceMap.getIcon("jButtonBackupRescan.icon")); // NOI18N
        jButtonBackupRescan.setText(bundle.getString("jButtonBackupRescan.text")); // NOI18N
        jButtonBackupRescan.setToolTipText(bundle.getString("jButtonBackupRescan.toolTipText")); // NOI18N
        jButtonBackupRescan.setEnabled(false);
        jButtonBackupRescan.setMaximumSize(new java.awt.Dimension(29, 29));
        jButtonBackupRescan.setMinimumSize(new java.awt.Dimension(29, 29));
        jButtonBackupRescan.setName("jButtonBackupRescan"); // NOI18N
        jButtonBackupRescan.setPreferredSize(new java.awt.Dimension(29, 29));
        jButtonBackupRescan.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                actionPerformedButtonBackupRescan(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 6;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 10, 5);
        jPanelBackup.add(jButtonBackupRescan, gridBagConstraints);

        jButtonBackupPath.setIcon(resourceMap.getIcon("jButtonBackupPath.icon")); // NOI18N
        jButtonBackupPath.setText(bundle.getString("jButtonBackupPath.text")); // NOI18N
        jButtonBackupPath.setToolTipText(bundle.getString("jButtonBackupPath.toolTipText")); // NOI18N
        jButtonBackupPath.setAlignmentX(0.2F);
        jButtonBackupPath.setMaximumSize(new java.awt.Dimension(29, 29));
        jButtonBackupPath.setMinimumSize(new java.awt.Dimension(29, 29));
        jButtonBackupPath.setName("jButtonBackupPath"); // NOI18N
        jButtonBackupPath.setPreferredSize(new java.awt.Dimension(29, 29));
        jButtonBackupPath.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                actionPerformedButtonBackupSelectDirectory(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 7;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 10, 5);
        jPanelBackup.add(jButtonBackupPath, gridBagConstraints);

        jScrollPaneBackupSaveFiles.setMinimumSize(new java.awt.Dimension(260, 130));
        jScrollPaneBackupSaveFiles.setName("jScrollPaneBackupSaveFiles"); // NOI18N
        jScrollPaneBackupSaveFiles.setPreferredSize(new java.awt.Dimension(260, 130));

        jTableBackupSaveGames.setAutoCreateColumnsFromModel(false);
        jTableBackupSaveGames.setAutoCreateRowSorter(false);
        jTableBackupSaveGames.setBackground(new java.awt.Color(Integer.valueOf(bundle.getString("jList.background"), 16)));
        jTableBackupSaveGames.setForeground(new java.awt.Color(Integer.valueOf(bundle.getString("jList.foreground"), 16)));
        jTableBackupSaveGames.setEnabled(false);
        jTableBackupSaveGames.setFillsViewportHeight(true);
        jTableBackupSaveGames.setFocusable(false);
        jTableBackupSaveGames.setIntercellSpacing(new java.awt.Dimension(0, 0));
        jTableBackupSaveGames.setName("jTableBackupSaveGames"); // NOI18N
        jTableBackupSaveGames.setRowHeight(25);
        jTableBackupSaveGames.setSelectionBackground(new java.awt.Color(Integer.valueOf(bundle.getString("jList.selectionBackground"), 16)));
        jTableBackupSaveGames.setSelectionForeground(new java.awt.Color(Integer.valueOf(bundle.getString("jList.selectionForeground"), 16)));
        jTableBackupSaveGames.setShowHorizontalLines(false);
        jTableBackupSaveGames.setShowVerticalLines(false);
        jTableBackupSaveGames.setUpdateSelectionOnSort(false);
        jScrollPaneBackupSaveFiles.setViewportView(jTableBackupSaveGames);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 6;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridwidth = 10;
        gridBagConstraints.gridheight = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.ipadx = 400;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(10, 0, 10, 10);
        jPanelBackup.add(jScrollPaneBackupSaveFiles, gridBagConstraints);

        jTextFieldBackupPath.setBackground(new java.awt.Color(Integer.valueOf(bundle.getString("displayTextfield.backgroundColor"), 16)));
        jTextFieldBackupPath.setEditable(false);
        jTextFieldBackupPath.setText(bundle.getString("jTextFieldBackupPath.text")); // NOI18N
        jTextFieldBackupPath.setMaximumSize(new java.awt.Dimension(2147483647, 29));
        jTextFieldBackupPath.setMinimumSize(new java.awt.Dimension(150, 29));
        jTextFieldBackupPath.setName("jTextFieldBackupPath"); // NOI18N
        jTextFieldBackupPath.setPreferredSize(new java.awt.Dimension(100, 29));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 8;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.gridwidth = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 10, 5);
        jPanelBackup.add(jTextFieldBackupPath, gridBagConstraints);

        jButtonBackupSaveGameRestore.setIcon(resourceMap.getIcon("jButtonBackupSaveGameRestore.icon")); // NOI18N
        jButtonBackupSaveGameRestore.setText(bundle.getString("jButtonBackupSaveGameRestore.text")); // NOI18N
        jButtonBackupSaveGameRestore.setToolTipText(bundle.getString("jButtonBackupSaveGameRestore.toolTipText")); // NOI18N
        jButtonBackupSaveGameRestore.setEnabled(false);
        jButtonBackupSaveGameRestore.setMaximumSize(new java.awt.Dimension(29, 29));
        jButtonBackupSaveGameRestore.setMinimumSize(new java.awt.Dimension(29, 29));
        jButtonBackupSaveGameRestore.setName("jButtonBackupSaveGameRestore"); // NOI18N
        jButtonBackupSaveGameRestore.setPreferredSize(new java.awt.Dimension(29, 29));
        jButtonBackupSaveGameRestore.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                actionPerformedButtonBackupSaveGameRestore(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 14;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 10, 5);
        jPanelBackup.add(jButtonBackupSaveGameRestore, gridBagConstraints);

        jButtonBackupSaveGameDelete.setIcon(resourceMap.getIcon("jButtonBackupSaveGameDelete.icon")); // NOI18N
        jButtonBackupSaveGameDelete.setText(bundle.getString("jButtonBackupSaveGameDelete.text")); // NOI18N
        jButtonBackupSaveGameDelete.setToolTipText(bundle.getString("jButtonBackupSaveGameDelete.toolTipText")); // NOI18N
        jButtonBackupSaveGameDelete.setEnabled(false);
        jButtonBackupSaveGameDelete.setMaximumSize(new java.awt.Dimension(29, 29));
        jButtonBackupSaveGameDelete.setMinimumSize(new java.awt.Dimension(29, 29));
        jButtonBackupSaveGameDelete.setName("jButtonBackupSaveGameDelete"); // NOI18N
        jButtonBackupSaveGameDelete.setPreferredSize(new java.awt.Dimension(29, 29));
        jButtonBackupSaveGameDelete.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                actionPerformedButtonBackupSaveGameDelete(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 13;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 10, 5);
        jPanelBackup.add(jButtonBackupSaveGameDelete, gridBagConstraints);

        jPanelBackupCharacterData.setBorder(javax.swing.BorderFactory.createEtchedBorder());
        jPanelBackupCharacterData.setAlignmentX(0.3F);
        jPanelBackupCharacterData.setMaximumSize(new java.awt.Dimension(520, 32767));
        jPanelBackupCharacterData.setName("jPanelBackupCharacterData"); // NOI18N
        jPanelBackupCharacterData.setPreferredSize(new java.awt.Dimension(349, 263));

        javax.swing.GroupLayout jPanelBackupCharacterDataLayout = new javax.swing.GroupLayout(jPanelBackupCharacterData);
        jPanelBackupCharacterData.setLayout(jPanelBackupCharacterDataLayout);
        jPanelBackupCharacterDataLayout.setHorizontalGroup(
            jPanelBackupCharacterDataLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGap(0, 410, Short.MAX_VALUE)
        );
        jPanelBackupCharacterDataLayout.setVerticalGroup(
            jPanelBackupCharacterDataLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGap(0, 296, Short.MAX_VALUE)
        );

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 16;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridwidth = 5;
        gridBagConstraints.gridheight = 4;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.ipadx = 410;
        gridBagConstraints.ipady = 200;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weighty = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(10, 0, 10, 10);
        jPanelBackup.add(jPanelBackupCharacterData, gridBagConstraints);

        jButtonBackupClipboard.setIcon(resourceMap.getIcon("jButtonBackupClipboard.icon")); // NOI18N
        jButtonBackupClipboard.setText(bundle.getString("jButtonBackupClipboard.text")); // NOI18N
        jButtonBackupClipboard.setToolTipText(bundle.getString("jButtonBackupClipboard.toolTipText")); // NOI18N
        jButtonBackupClipboard.setEnabled(false);
        jButtonBackupClipboard.setMaximumSize(new java.awt.Dimension(29, 29));
        jButtonBackupClipboard.setMinimumSize(new java.awt.Dimension(29, 29));
        jButtonBackupClipboard.setName("jButtonBackupClipboard"); // NOI18N
        jButtonBackupClipboard.setPreferredSize(new java.awt.Dimension(29, 29));
        jButtonBackupClipboard.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                actionPerformedButtonBackupClipboard(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 15;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 10, 10);
        jPanelBackup.add(jButtonBackupClipboard, gridBagConstraints);

        jTextFieldAmountBackup.setBackground(new java.awt.Color(Integer.valueOf(bundle.getString("displayTextfield.backgroundColor"), 16)));
        jTextFieldAmountBackup.setEditable(false);
        jTextFieldAmountBackup.setText(bundle.getString("jTextFieldAmountBackup.text")); // NOI18N
        jTextFieldAmountBackup.setMaximumSize(new java.awt.Dimension(60, 29));
        jTextFieldAmountBackup.setMinimumSize(new java.awt.Dimension(60, 29));
        jTextFieldAmountBackup.setName("jTextFieldAmountBackup"); // NOI18N
        jTextFieldAmountBackup.setPreferredSize(new java.awt.Dimension(60, 29));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 11;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 10, 5);
        jPanelBackup.add(jTextFieldAmountBackup, gridBagConstraints);

        jButtonBackupCharacterLaunch.setIcon(resourceMap.getIcon("jButtonBackupCharacterLaunch.icon")); // NOI18N
        jButtonBackupCharacterLaunch.setText(bundle.getString("jButtonBackupCharacterLaunch.text")); // NOI18N
        jButtonBackupCharacterLaunch.setToolTipText(bundle.getString("jButtonBackupCharacterLaunch.toolTipText")); // NOI18N
        jButtonBackupCharacterLaunch.setEnabled(false);
        jButtonBackupCharacterLaunch.setMaximumSize(new java.awt.Dimension(29, 29));
        jButtonBackupCharacterLaunch.setMinimumSize(new java.awt.Dimension(29, 29));
        jButtonBackupCharacterLaunch.setName("jButtonBackupCharacterLaunch"); // NOI18N
        jButtonBackupCharacterLaunch.setPreferredSize(new java.awt.Dimension(29, 29));
        jButtonBackupCharacterLaunch.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                actionPerformedButtonBackupLaunchPlayer(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 10, 10, 5);
        jPanelBackup.add(jButtonBackupCharacterLaunch, gridBagConstraints);

        jTextFieldRaceBackupPlayer.setBackground(new java.awt.Color(Integer.valueOf(bundle.getString("displayTextfield.backgroundColor"), 16)));
        jTextFieldRaceBackupPlayer.setEditable(false);
        jTextFieldRaceBackupPlayer.setText(bundle.getString("jTextFieldRaceBackupPlayer")); // NOI18N
        jTextFieldRaceBackupPlayer.setMaximumSize(new java.awt.Dimension(100, 29));
        jTextFieldRaceBackupPlayer.setMinimumSize(new java.awt.Dimension(100, 29));
        jTextFieldRaceBackupPlayer.setName("jTextFieldRaceBackupPlayer"); // NOI18N
        jTextFieldRaceBackupPlayer.setPreferredSize(new java.awt.Dimension(100, 29));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 4;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 10, 10);
        jPanelBackup.add(jTextFieldRaceBackupPlayer, gridBagConstraints);

        jButtonBackupLaunchSavegame.setIcon(resourceMap.getIcon("jButtonBackupLaunchSavegame.icon")); // NOI18N
        jButtonBackupLaunchSavegame.setText(bundle.getString("jButtonBackupLaunchSavegame.text")); // NOI18N
        jButtonBackupLaunchSavegame.setToolTipText(bundle.getString("jButtonBackupLaunchSavegame.toolTipText")); // NOI18N
        jButtonBackupLaunchSavegame.setEnabled(false);
        jButtonBackupLaunchSavegame.setMaximumSize(new java.awt.Dimension(29, 29));
        jButtonBackupLaunchSavegame.setMinimumSize(new java.awt.Dimension(29, 29));
        jButtonBackupLaunchSavegame.setName("jButtonBackupLaunchSavegame"); // NOI18N
        jButtonBackupLaunchSavegame.setPreferredSize(new java.awt.Dimension(29, 29));
        jButtonBackupLaunchSavegame.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                actionPerformedButtonLaunchBackupSavegame(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 12;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 10, 5);
        jPanelBackup.add(jButtonBackupLaunchSavegame, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 0.5;
        gridBagConstraints.insets = new java.awt.Insets(18, 10, 0, 10);
        getContentPane().add(jPanelBackup, gridBagConstraints);

        jPanelActive.setBorder(javax.swing.BorderFactory.createEtchedBorder());
        jPanelActive.setAlignmentY(1.0F);
        jPanelActive.setMinimumSize(new java.awt.Dimension(496, 300));
        jPanelActive.setName("jPanelActive"); // NOI18N
        jPanelActive.setPreferredSize(new java.awt.Dimension(819, 300));
        jPanelActive.setLayout(new java.awt.GridBagLayout());

        jScrollPaneActiveCharacters.setMaximumSize(new java.awt.Dimension(165, 32767));
        jScrollPaneActiveCharacters.setMinimumSize(new java.awt.Dimension(165, 130));
        jScrollPaneActiveCharacters.setName("jScrollPaneActiveCharacters"); // NOI18N
        jScrollPaneActiveCharacters.setPreferredSize(new java.awt.Dimension(165, 130));

        jListActiveCharacters.setBackground(new java.awt.Color(Integer.valueOf(bundle.getString("jList.background"), 16)));
        jListActiveCharacters.setForeground(new java.awt.Color(Integer.valueOf(bundle.getString("jList.foreground"), 16)));
        jListActiveCharacters.setAlignmentX(0.2F);
        jListActiveCharacters.setFocusable(false);
        jListActiveCharacters.setName("jListActiveCharacters"); // NOI18N
        jListActiveCharacters.setSelectionBackground(new java.awt.Color(Integer.valueOf(bundle.getString("jList.selectionBackground"), 16)));
        jListActiveCharacters.setSelectionForeground(new java.awt.Color(Integer.valueOf(bundle.getString("jList.selectionForeground"), 16)));
        jScrollPaneActiveCharacters.setViewportView(jListActiveCharacters);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridwidth = 7;
        gridBagConstraints.gridheight = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weighty = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(10, 10, 10, 10);
        jPanelActive.add(jScrollPaneActiveCharacters, gridBagConstraints);

        jButtonActiveCharacterBackup.setIcon(resourceMap.getIcon("jButtonActiveCharacterBackup.icon")); // NOI18N
        jButtonActiveCharacterBackup.setText(bundle.getString("jButtonActiveCharacterRestore.text")); // NOI18N
        jButtonActiveCharacterBackup.setToolTipText(bundle.getString("jButtonActiveCharacterBackup.toolTipText")); // NOI18N
        jButtonActiveCharacterBackup.setEnabled(false);
        jButtonActiveCharacterBackup.setMaximumSize(new java.awt.Dimension(29, 29));
        jButtonActiveCharacterBackup.setMinimumSize(new java.awt.Dimension(29, 29));
        jButtonActiveCharacterBackup.setName("jButtonActiveCharacterBackup"); // NOI18N
        jButtonActiveCharacterBackup.setPreferredSize(new java.awt.Dimension(29, 29));
        jButtonActiveCharacterBackup.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                actionPerformedButtonActiveBackupPlayer(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 10, 5);
        jPanelActive.add(jButtonActiveCharacterBackup, gridBagConstraints);

        jButtonActiveCharacterDelete.setIcon(resourceMap.getIcon("jButtonActiveCharacterDelete.icon")); // NOI18N
        jButtonActiveCharacterDelete.setText(bundle.getString("jButtonActiveCharacterDelete.text")); // NOI18N
        jButtonActiveCharacterDelete.setToolTipText(bundle.getString("jButtonActiveCharacterDelete.toolTipText")); // NOI18N
        jButtonActiveCharacterDelete.setEnabled(false);
        jButtonActiveCharacterDelete.setMaximumSize(new java.awt.Dimension(29, 29));
        jButtonActiveCharacterDelete.setMinimumSize(new java.awt.Dimension(29, 29));
        jButtonActiveCharacterDelete.setName("jButtonActiveCharacterDelete"); // NOI18N
        jButtonActiveCharacterDelete.setPreferredSize(new java.awt.Dimension(29, 29));
        jButtonActiveCharacterDelete.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                actionPerformedButtonActiveDelete(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 10, 5);
        jPanelActive.add(jButtonActiveCharacterDelete, gridBagConstraints);

        jButtonActiveCharacterClear.setIcon(resourceMap.getIcon("jButtonActiveCharacterClear.icon")); // NOI18N
        jButtonActiveCharacterClear.setText(bundle.getString("jButtonActiveCharacterClear.text")); // NOI18N
        jButtonActiveCharacterClear.setToolTipText(bundle.getString("jButtonActiveCharacterClear.toolTipText")); // NOI18N
        jButtonActiveCharacterClear.setEnabled(false);
        jButtonActiveCharacterClear.setMaximumSize(new java.awt.Dimension(29, 29));
        jButtonActiveCharacterClear.setMinimumSize(new java.awt.Dimension(29, 29));
        jButtonActiveCharacterClear.setName("jButtonActiveCharacterClear"); // NOI18N
        jButtonActiveCharacterClear.setPreferredSize(new java.awt.Dimension(29, 29));
        jButtonActiveCharacterClear.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                actionPerformedButtonActiveClear(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 10, 5);
        jPanelActive.add(jButtonActiveCharacterClear, gridBagConstraints);

        jButtonActiveRescan.setIcon(resourceMap.getIcon("jButtonActiveRescan.icon")); // NOI18N
        jButtonActiveRescan.setText(bundle.getString("jButtonActiveRescan.text")); // NOI18N
        jButtonActiveRescan.setToolTipText(bundle.getString("jButtonActiveRescan.toolTipText")); // NOI18N
        jButtonActiveRescan.setEnabled(false);
        jButtonActiveRescan.setMaximumSize(new java.awt.Dimension(29, 29));
        jButtonActiveRescan.setMinimumSize(new java.awt.Dimension(29, 29));
        jButtonActiveRescan.setName("jButtonActiveRescan"); // NOI18N
        jButtonActiveRescan.setPreferredSize(new java.awt.Dimension(29, 29));
        jButtonActiveRescan.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                actionPerformedButtonActiveRescan(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 7;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 10, 5);
        jPanelActive.add(jButtonActiveRescan, gridBagConstraints);

        jButtonActivePath.setIcon(resourceMap.getIcon("jButtonActivePath.icon")); // NOI18N
        jButtonActivePath.setText(bundle.getString("jButtonActivePath.text")); // NOI18N
        jButtonActivePath.setToolTipText(bundle.getString("jButtonActivePath.toolTipText")); // NOI18N
        jButtonActivePath.setAlignmentX(0.2F);
        jButtonActivePath.setMaximumSize(new java.awt.Dimension(29, 29));
        jButtonActivePath.setMinimumSize(new java.awt.Dimension(29, 29));
        jButtonActivePath.setName("jButtonActivePath"); // NOI18N
        jButtonActivePath.setPreferredSize(new java.awt.Dimension(29, 29));
        jButtonActivePath.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                actionPerformedButtonActiveSelectDirectory(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 8;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 10, 5);
        jPanelActive.add(jButtonActivePath, gridBagConstraints);

        jScrollPaneActiveSaveFiles.setMinimumSize(new java.awt.Dimension(260, 130));
        jScrollPaneActiveSaveFiles.setName("jScrollPaneActiveSaveFiles"); // NOI18N
        jScrollPaneActiveSaveFiles.setPreferredSize(new java.awt.Dimension(260, 130));

        jTableActiveSaveGames.setAutoCreateColumnsFromModel(false);
        jTableActiveSaveGames.setBackground(new java.awt.Color(Integer.valueOf(bundle.getString("jList.background"), 16)));
        jTableActiveSaveGames.setForeground(new java.awt.Color(Integer.valueOf(bundle.getString("jList.foreground"), 16)));
        jTableActiveSaveGames.setEnabled(false);
        jTableActiveSaveGames.setFillsViewportHeight(true);
        jTableActiveSaveGames.setFocusable(false);
        jTableActiveSaveGames.setIntercellSpacing(new java.awt.Dimension(0, 0));
        jTableActiveSaveGames.setName("jTableActiveSaveGames"); // NOI18N
        jTableActiveSaveGames.setRowHeight(25);
        jTableActiveSaveGames.setSelectionBackground(new java.awt.Color(Integer.valueOf(bundle.getString("jList.selectionBackground"), 16)));
        jTableActiveSaveGames.setSelectionForeground(new java.awt.Color(Integer.valueOf(bundle.getString("jList.selectionForeground"), 16)));
        jTableActiveSaveGames.setShowHorizontalLines(false);
        jTableActiveSaveGames.setShowVerticalLines(false);
        jTableActiveSaveGames.setUpdateSelectionOnSort(false);
        jScrollPaneActiveSaveFiles.setViewportView(jTableActiveSaveGames);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 7;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridwidth = 10;
        gridBagConstraints.gridheight = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.ipadx = 400;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(10, 0, 10, 10);
        jPanelActive.add(jScrollPaneActiveSaveFiles, gridBagConstraints);

        jTextFieldActivePath.setBackground(new java.awt.Color(Integer.valueOf(bundle.getString("displayTextfield.backgroundColor"), 16)));
        jTextFieldActivePath.setEditable(false);
        jTextFieldActivePath.setText(bundle.getString("jTextFieldActivePath.text")); // NOI18N
        jTextFieldActivePath.setMaximumSize(new java.awt.Dimension(2147483647, 29));
        jTextFieldActivePath.setMinimumSize(new java.awt.Dimension(150, 29));
        jTextFieldActivePath.setName("jTextFieldActivePath"); // NOI18N
        jTextFieldActivePath.setPreferredSize(new java.awt.Dimension(150, 29));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 9;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.gridwidth = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 10, 5);
        jPanelActive.add(jTextFieldActivePath, gridBagConstraints);

        jButtonActiveSaveGameBackup.setIcon(resourceMap.getIcon("jButtonActiveSaveGameBackup.icon")); // NOI18N
        jButtonActiveSaveGameBackup.setText(bundle.getString("jButtonActiveSaveGameRestore.text")); // NOI18N
        jButtonActiveSaveGameBackup.setToolTipText(bundle.getString("jButtonActiveSaveGameRestore.toolTipText")); // NOI18N
        jButtonActiveSaveGameBackup.setEnabled(false);
        jButtonActiveSaveGameBackup.setMaximumSize(new java.awt.Dimension(29, 29));
        jButtonActiveSaveGameBackup.setMinimumSize(new java.awt.Dimension(29, 29));
        jButtonActiveSaveGameBackup.setName("jButtonActiveSaveGameBackup"); // NOI18N
        jButtonActiveSaveGameBackup.setPreferredSize(new java.awt.Dimension(29, 29));
        jButtonActiveSaveGameBackup.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                actionPerformedButtonActiveSaveGameBackup(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 15;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 10, 5);
        jPanelActive.add(jButtonActiveSaveGameBackup, gridBagConstraints);

        jButtonActiveSaveGameDelete.setIcon(resourceMap.getIcon("jButtonActiveSaveGameDelete.icon")); // NOI18N
        jButtonActiveSaveGameDelete.setText(bundle.getString("jButtonActiveSaveGameDelete.text")); // NOI18N
        jButtonActiveSaveGameDelete.setToolTipText(bundle.getString("jButtonActiveSaveGameDelete.toolTipText")); // NOI18N
        jButtonActiveSaveGameDelete.setEnabled(false);
        jButtonActiveSaveGameDelete.setMaximumSize(new java.awt.Dimension(29, 29));
        jButtonActiveSaveGameDelete.setMinimumSize(new java.awt.Dimension(29, 29));
        jButtonActiveSaveGameDelete.setName("jButtonActiveSaveGameDelete"); // NOI18N
        jButtonActiveSaveGameDelete.setPreferredSize(new java.awt.Dimension(29, 29));
        jButtonActiveSaveGameDelete.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                actionPerformedButtonActiveSaveGameDelete(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 14;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 10, 5);
        jPanelActive.add(jButtonActiveSaveGameDelete, gridBagConstraints);

        jPanelActiveCharacterData.setBorder(javax.swing.BorderFactory.createEtchedBorder());
        jPanelActiveCharacterData.setAlignmentX(0.3F);
        jPanelActiveCharacterData.setMaximumSize(new java.awt.Dimension(520, 32767));
        jPanelActiveCharacterData.setName("jPanelActiveCharacterData"); // NOI18N
        jPanelActiveCharacterData.setPreferredSize(new java.awt.Dimension(349, 263));

        javax.swing.GroupLayout jPanelActiveCharacterDataLayout = new javax.swing.GroupLayout(jPanelActiveCharacterData);
        jPanelActiveCharacterData.setLayout(jPanelActiveCharacterDataLayout);
        jPanelActiveCharacterDataLayout.setHorizontalGroup(
            jPanelActiveCharacterDataLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGap(0, 410, Short.MAX_VALUE)
        );
        jPanelActiveCharacterDataLayout.setVerticalGroup(
            jPanelActiveCharacterDataLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGap(0, 296, Short.MAX_VALUE)
        );

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 17;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridwidth = 5;
        gridBagConstraints.gridheight = 4;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.ipadx = 410;
        gridBagConstraints.ipady = 200;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weighty = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(10, 0, 10, 10);
        jPanelActive.add(jPanelActiveCharacterData, gridBagConstraints);

        jButtonActiveClipboard.setIcon(resourceMap.getIcon("jButtonActiveClipboard.icon")); // NOI18N
        jButtonActiveClipboard.setText(bundle.getString("jButtonActiveClipboard.text")); // NOI18N
        jButtonActiveClipboard.setToolTipText(bundle.getString("jButtonActiveClipboard.toolTipText")); // NOI18N
        jButtonActiveClipboard.setEnabled(false);
        jButtonActiveClipboard.setMaximumSize(new java.awt.Dimension(29, 29));
        jButtonActiveClipboard.setMinimumSize(new java.awt.Dimension(29, 29));
        jButtonActiveClipboard.setName("jButtonActiveClipboard"); // NOI18N
        jButtonActiveClipboard.setPreferredSize(new java.awt.Dimension(29, 29));
        jButtonActiveClipboard.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                actionPerformedButtonActiveClipboard(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 16;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 10, 10);
        jPanelActive.add(jButtonActiveClipboard, gridBagConstraints);

        jTextFieldAmountActive.setBackground(new java.awt.Color(Integer.valueOf(bundle.getString("displayTextfield.backgroundColor"), 16)));
        jTextFieldAmountActive.setEditable(false);
        jTextFieldAmountActive.setText(bundle.getString("jTextFieldAmountActive.text")); // NOI18N
        jTextFieldAmountActive.setMaximumSize(new java.awt.Dimension(60, 29));
        jTextFieldAmountActive.setMinimumSize(new java.awt.Dimension(60, 29));
        jTextFieldAmountActive.setName("jTextFieldAmountActive"); // NOI18N
        jTextFieldAmountActive.setPreferredSize(new java.awt.Dimension(60, 29));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 12;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 10, 5);
        jPanelActive.add(jTextFieldAmountActive, gridBagConstraints);

        jButtonActiveCharacterLaunch.setIcon(resourceMap.getIcon("jButtonActiveCharacterLaunch.icon")); // NOI18N
        jButtonActiveCharacterLaunch.setText(bundle.getString("jButtonActiveCharacterLaunch.text")); // NOI18N
        jButtonActiveCharacterLaunch.setToolTipText(bundle.getString("jButtonActiveCharacterLaunch.toolTipText")); // NOI18N
        jButtonActiveCharacterLaunch.setEnabled(false);
        jButtonActiveCharacterLaunch.setMaximumSize(new java.awt.Dimension(29, 29));
        jButtonActiveCharacterLaunch.setMinimumSize(new java.awt.Dimension(29, 29));
        jButtonActiveCharacterLaunch.setName("jButtonActiveCharacterLaunch"); // NOI18N
        jButtonActiveCharacterLaunch.setPreferredSize(new java.awt.Dimension(29, 29));
        jButtonActiveCharacterLaunch.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                actionPerformedButtonActiveLaunchPlayer(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 10, 10, 5);
        jPanelActive.add(jButtonActiveCharacterLaunch, gridBagConstraints);

        jTextFieldRaceActivePlayer.setBackground(new java.awt.Color(Integer.valueOf(bundle.getString("displayTextfield.backgroundColor"), 16)));
        jTextFieldRaceActivePlayer.setEditable(false);
        jTextFieldRaceActivePlayer.setText(bundle.getString("jTextFieldRaceActivePlayer.text")); // NOI18N
        jTextFieldRaceActivePlayer.setMaximumSize(new java.awt.Dimension(2147483647, 29));
        jTextFieldRaceActivePlayer.setMinimumSize(new java.awt.Dimension(100, 29));
        jTextFieldRaceActivePlayer.setName("jTextFieldRaceActivePlayer"); // NOI18N
        jTextFieldRaceActivePlayer.setPreferredSize(new java.awt.Dimension(100, 29));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 4;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 10, 10);
        jPanelActive.add(jTextFieldRaceActivePlayer, gridBagConstraints);

        jButtonActiveLaunchSavegame.setIcon(resourceMap.getIcon("jButtonActiveLaunchSavegame.icon")); // NOI18N
        jButtonActiveLaunchSavegame.setText(bundle.getString("jButtonActiveLaunchSavegame.text")); // NOI18N
        jButtonActiveLaunchSavegame.setToolTipText(bundle.getString("jButtonActiveLaunchSavegame.toolTipText")); // NOI18N
        jButtonActiveLaunchSavegame.setEnabled(false);
        jButtonActiveLaunchSavegame.setMaximumSize(new java.awt.Dimension(29, 29));
        jButtonActiveLaunchSavegame.setMinimumSize(new java.awt.Dimension(29, 29));
        jButtonActiveLaunchSavegame.setName("jButtonActiveLaunchSavegame"); // NOI18N
        jButtonActiveLaunchSavegame.setPreferredSize(new java.awt.Dimension(29, 29));
        jButtonActiveLaunchSavegame.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                actionPerformedButtonLaunchActiveSavegame(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 13;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 10, 5);
        jPanelActive.add(jButtonActiveLaunchSavegame, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 0.5;
        gridBagConstraints.insets = new java.awt.Insets(18, 10, 0, 10);
        getContentPane().add(jPanelActive, gridBagConstraints);

        pack();
    }// </editor-fold>//GEN-END:initComponents

    private void actionPerformedButtonProcessExit(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_actionPerformedButtonProcessExit
      if (null != m_tGuiConnector) m_tGuiConnector.handleButtonExit();
    }//GEN-LAST:event_actionPerformedButtonProcessExit

    private void actionPerformedButtonProcessCancel(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_actionPerformedButtonProcessCancel
      if (null != m_tGuiConnector) m_tGuiConnector.handleButtonProcessCancel();
    }//GEN-LAST:event_actionPerformedButtonProcessCancel

    private void actionPerformedButtonBackupClipboard(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_actionPerformedButtonBackupClipboard
      if (null != m_tGuiConnector) {m_tGuiConnector.handleButtonBackupClipboard();}   
    }//GEN-LAST:event_actionPerformedButtonBackupClipboard

    private void actionPerformedButtonBackupSaveGameDelete(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_actionPerformedButtonBackupSaveGameDelete
      if (null != m_tGuiConnector) {m_tGuiConnector.handleButtonBackupDeleteSaveGame();} 
	}//GEN-LAST:event_actionPerformedButtonBackupSaveGameDelete

    private void actionPerformedButtonBackupSaveGameRestore(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_actionPerformedButtonBackupSaveGameRestore
      if (null != m_tGuiConnector) {m_tGuiConnector.handleButtonBackupSaveGameRestore();}
	}//GEN-LAST:event_actionPerformedButtonBackupSaveGameRestore

    private void actionPerformedButtonActiveClipboard(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_actionPerformedButtonActiveClipboard
      if (null != m_tGuiConnector) {m_tGuiConnector.handleButtonActiveClipboard();}
    }//GEN-LAST:event_actionPerformedButtonActiveClipboard

  private void actionPerformedButtonActiveSelectDirectory(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_actionPerformedButtonActiveSelectDirectory
      if (null != m_tGuiConnector) {m_tGuiConnector.handleButtonSelectActivePath();} 
  }//GEN-LAST:event_actionPerformedButtonActiveSelectDirectory

  private void actionPerformedButtonBackupSelectDirectory(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_actionPerformedButtonBackupSelectDirectory
      if (null != m_tGuiConnector) {m_tGuiConnector.handleButtonSelectBackupPath();} 
  }//GEN-LAST:event_actionPerformedButtonBackupSelectDirectory

  private void actionPerformedButtonActiveRescan(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_actionPerformedButtonActiveRescan
      if (null != m_tGuiConnector) {m_tGuiConnector.handleButtonSelectActiveRescan(null, true);} 
  }//GEN-LAST:event_actionPerformedButtonActiveRescan

  private void actionPerformedButtonBackupRescan(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_actionPerformedButtonBackupRescan
      if (null != m_tGuiConnector) {m_tGuiConnector.handleButtonSelectBackupRescan();} 
  }//GEN-LAST:event_actionPerformedButtonBackupRescan

  private void actionPerformedButtonActiveClear(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_actionPerformedButtonActiveClear
      if (null != m_tGuiConnector) {m_tGuiConnector.handleButtonActiveClearPlayers();} 
  }//GEN-LAST:event_actionPerformedButtonActiveClear

  private void actionPerformedButtonBackupClear(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_actionPerformedButtonBackupClear
      if (null != m_tGuiConnector) {m_tGuiConnector.handleButtonBackupClearPlayers();} 
  }//GEN-LAST:event_actionPerformedButtonBackupClear

  private void actionPerformedButtonActiveDelete(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_actionPerformedButtonActiveDelete
      if (null != m_tGuiConnector) {m_tGuiConnector.handleButtonActiveDeletePlayer();} 
  }//GEN-LAST:event_actionPerformedButtonActiveDelete

  private void actionPerformedButtonBackupDelete(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_actionPerformedButtonBackupDelete
      if (null != m_tGuiConnector) {m_tGuiConnector.handleButtonBackupDeletePlayer();} 
  }//GEN-LAST:event_actionPerformedButtonBackupDelete

  private void actionPerformedButtonActiveBackupPlayer(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_actionPerformedButtonActiveBackupPlayer
      if (null != m_tGuiConnector) {m_tGuiConnector.handleButtonActiveBackupPlayer();} 
  }//GEN-LAST:event_actionPerformedButtonActiveBackupPlayer

  private void actionPerformedButtonBackupPlayerRestore(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_actionPerformedButtonBackupPlayerRestore
      if (null != m_tGuiConnector) {m_tGuiConnector.handleButtonBackupPlayerRestore();} 
  }//GEN-LAST:event_actionPerformedButtonBackupPlayerRestore

  private void actionPerformedButtonActiveSaveGameBackup(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_actionPerformedButtonActiveSaveGameBackup
      if (null != m_tGuiConnector) {m_tGuiConnector.handleButtonActiveBackupSaveGame();} 
  }//GEN-LAST:event_actionPerformedButtonActiveSaveGameBackup

  private void actionPerformedButtonActiveSaveGameDelete(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_actionPerformedButtonActiveSaveGameDelete
      if (null != m_tGuiConnector) {m_tGuiConnector.handleButtonActiveDeleteSaveGame();} 
  }//GEN-LAST:event_actionPerformedButtonActiveSaveGameDelete

  private void actionPerformedButtonAbout(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_actionPerformedButtonAbout
      if (null != m_tGuiConnector) {m_tGuiConnector.handleButtonAbout();}
  }//GEN-LAST:event_actionPerformedButtonAbout

  private void actionPerformedButtonActiveLaunchPlayer(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_actionPerformedButtonActiveLaunchPlayer
    if (null != m_tGuiConnector) {m_tGuiConnector.handleButtonActiveLaunchPlayer();}
  }//GEN-LAST:event_actionPerformedButtonActiveLaunchPlayer

  private void actionPerformedButtonBackupLaunchPlayer(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_actionPerformedButtonBackupLaunchPlayer
      if (null != m_tGuiConnector) {m_tGuiConnector.handleButtonBackupLaunchPlayer();}
  }//GEN-LAST:event_actionPerformedButtonBackupLaunchPlayer

  private void actionPerformedButtonChooseBinary(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_actionPerformedButtonChooseBinary
      if (null != m_tGuiConnector) {m_tGuiConnector.handleButtonChooseBinary();}
  }//GEN-LAST:event_actionPerformedButtonChooseBinary

  private void actionPerformedButtonRestoreRegistry(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_actionPerformedButtonRestoreRegistry
      if (null != m_tGuiConnector) {m_tGuiConnector.handleButtonRestoreLaunchPath();}
  }//GEN-LAST:event_actionPerformedButtonRestoreRegistry

  private void actionPerformedButtonLaunchSolo(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_actionPerformedButtonLaunchSolo
      if (null != m_tGuiConnector) {m_tGuiConnector.handleButtonLaunchSolo();}
  }//GEN-LAST:event_actionPerformedButtonLaunchSolo

  private void actionPerformedComboBoxImageMode(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_actionPerformedComboBoxImageMode
    setImageModeDisplay(0 == getImageMode());
    if (null != m_tGuiConnector) {m_tGuiConnector.handleComboBoxImageMode();}
  }//GEN-LAST:event_actionPerformedComboBoxImageMode

  private void actionPerformedButtonLaunchActiveSavegame(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_actionPerformedButtonLaunchActiveSavegame
      if (null != m_tGuiConnector) {m_tGuiConnector.handleButtonActiveLaunchSaveGame();}
  }//GEN-LAST:event_actionPerformedButtonLaunchActiveSavegame

  private void actionPerformedButtonLaunchBackupSavegame(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_actionPerformedButtonLaunchBackupSavegame
      if (null != m_tGuiConnector) {m_tGuiConnector.handleButtonBackupLaunchSaveGame();}
  }//GEN-LAST:event_actionPerformedButtonLaunchBackupSavegame

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.Box.Filler filler21;
    private javax.swing.Box.Filler filler31;
    private javax.swing.JButton jButtonAbout;
    private javax.swing.JButton jButtonActiveCharacterBackup;
    private javax.swing.JButton jButtonActiveCharacterClear;
    private javax.swing.JButton jButtonActiveCharacterDelete;
    private javax.swing.JButton jButtonActiveCharacterLaunch;
    private javax.swing.JButton jButtonActiveClipboard;
    private javax.swing.JButton jButtonActiveLaunchSavegame;
    private javax.swing.JButton jButtonActivePath;
    private javax.swing.JButton jButtonActiveRescan;
    private javax.swing.JButton jButtonActiveSaveGameBackup;
    private javax.swing.JButton jButtonActiveSaveGameDelete;
    private javax.swing.JButton jButtonBackupCharacterClear;
    private javax.swing.JButton jButtonBackupCharacterDelete;
    private javax.swing.JButton jButtonBackupCharacterLaunch;
    private javax.swing.JButton jButtonBackupCharacterRestore;
    private javax.swing.JButton jButtonBackupClipboard;
    private javax.swing.JButton jButtonBackupLaunchSavegame;
    private javax.swing.JButton jButtonBackupPath;
    private javax.swing.JButton jButtonBackupRescan;
    private javax.swing.JButton jButtonBackupSaveGameDelete;
    private javax.swing.JButton jButtonBackupSaveGameRestore;
    private javax.swing.JButton jButtonChooseBinary;
    private javax.swing.JButton jButtonLaunchSolo;
    private javax.swing.JButton jButtonProgressCancel;
    private javax.swing.JButton jButtonProgressExit;
    private javax.swing.JButton jButtonRestoreRegistryPath;
    private javax.swing.JComboBox jComboBoxAutoSaves;
    private javax.swing.JComboBox jComboBoxConfirmation;
    private javax.swing.JComboBox jComboBoxImageMode;
    private javax.swing.JComboBox jComboBoxInitMode;
    private javax.swing.JComboBox jComboBoxMinimize;
    private javax.swing.JComboBox jComboBoxMode;
    private javax.swing.JComboBox jComboBoxOverwrite;
    private javax.swing.JComboBox jComboBoxQuickSaves;
    private javax.swing.JLabel jLabelAutoSaves;
    private javax.swing.JLabel jLabelBinaryPath;
    private javax.swing.JLabel jLabelConfirm;
    private javax.swing.JLabel jLabelImageMode;
    private javax.swing.JLabel jLabelInitMode;
    private javax.swing.JLabel jLabelLaunchOptions;
    private javax.swing.JLabel jLabelLaunchTarget;
    private javax.swing.JLabel jLabelMode;
    private javax.swing.JLabel jLabelOnSkyrim;
    private javax.swing.JLabel jLabelOverwrite;
    private javax.swing.JLabel jLabelPrefix;
    private javax.swing.JLabel jLabelProgressAction;
    private javax.swing.JLabel jLabelProgressCurrent;
    private javax.swing.JLabel jLabelProgressTotal;
    private javax.swing.JLabel jLabelQuickSave;
    private javax.swing.JList jListActiveCharacters;
    private javax.swing.JList jListBackupCharacters;
    private javax.swing.JPanel jPanelActive;
    private javax.swing.JPanel jPanelActiveCharacterData;
    private javax.swing.JPanel jPanelBackup;
    private javax.swing.JPanel jPanelBackupCharacterData;
    private javax.swing.JPanel jPanelControl;
    private javax.swing.JPanel jPanelProgress;
    private javax.swing.JProgressBar jProgressBarCurrent;
    private javax.swing.JProgressBar jProgressBarTotal;
    private javax.swing.JScrollPane jScrollPaneActiveCharacters;
    private javax.swing.JScrollPane jScrollPaneActiveSaveFiles;
    private javax.swing.JScrollPane jScrollPaneBackupCharacters;
    private javax.swing.JScrollPane jScrollPaneBackupSaveFiles;
    private javax.swing.JTable jTableActiveSaveGames;
    private javax.swing.JTable jTableBackupSaveGames;
    private javax.swing.JTextField jTextFieldActivePath;
    private javax.swing.JTextField jTextFieldAmountActive;
    private javax.swing.JTextField jTextFieldAmountBackup;
    private javax.swing.JTextField jTextFieldBackupPath;
    private javax.swing.JTextField jTextFieldBinary;
    private javax.swing.JTextField jTextFieldLaunchOptions;
    private javax.swing.JTextField jTextFieldPrefix;
    private javax.swing.JTextField jTextFieldProgressAction;
    private javax.swing.JTextField jTextFieldRaceActivePlayer;
    private javax.swing.JTextField jTextFieldRaceBackupPlayer;
    // End of variables declaration//GEN-END:variables



 /**
  * MouseListener.mouseClicked
  * 
  * All we're interested in here is a mouse click in a table's column header This indicates a sort,
  * which might have moved the current selection out of the visible area. So we need to restore
  * an eventual previous selection and scroll there.
  * 
  * @param pa_tEvent    mouse event
  */
 public void mouseClicked(java.awt.event.MouseEvent pa_tEvent)
 {
  int lc_iIndex = -1;
  int lc_iFirstSelection = -1;
  
  if (true == m_bLocked) return;
  
  if (null != pa_tEvent)
  {
   if (pa_tEvent.getSource() == jTableActiveSaveGames.getTableHeader())
   {
    if (0 <= (lc_iIndex = jTableActiveSaveGames.convertColumnIndexToModel(jTableActiveSaveGames.columnAtPoint(pa_tEvent.getPoint())))) 
    {
     if (lc_iIndex < SkyrimCharacterHelperConstants.SKH_TABLE_COLUMNS)
     {
      if (-1 != (lc_iFirstSelection = jTableActiveSaveGames.getSelectedRow()))
      {
       if (null != m_tStoredActiveSaveGameSelection)
       {
        //
        // reselect previously stored savegames and scroll to first visible entry
        //
        restoreSaveGameSelection(true, false);
       }
      }
     }
    }
   }  
   else if (pa_tEvent.getSource() == jTableBackupSaveGames.getTableHeader())
   {
    if (0 <= (lc_iIndex = jTableBackupSaveGames.convertColumnIndexToModel(jTableBackupSaveGames.columnAtPoint(pa_tEvent.getPoint())))) 
    {
     if (lc_iIndex < SkyrimCharacterHelperConstants.SKH_TABLE_COLUMNS)
     {
      if (-1 != (lc_iFirstSelection = jTableBackupSaveGames.getSelectedRow()))
      {
       if (null != m_tStoredBackupSaveGameSelection)
       {
        //
        // reselect previously stored savegames and scroll to first visible entry
        //
        restoreSaveGameSelection(false, false);
       }
      }
     }
    }
   }
  }
 }
 
 

 /**
  * MouseListener.mouseEntered
     * 
  * @param pa_tEvent    mouse event
  */ 
 public void mouseEntered(java.awt.event.MouseEvent pa_tEvent) {}
 
 
 
 /**
  * MouseListener.mouseExited
     * 
  * @param pa_tEvent    mouse event
  */
 public void mouseExited(java.awt.event.MouseEvent pa_tEvent) {}

 
 
 /**
  * MouseListener.mousePressed. We catch this event from table headers (!) to store the currently active savegame selection
  * for later restoring. This event indicates that a sort process is about to begin, so before that sort, we retrieve the 
  * list of selections and store it. After the sort is finished, we restore this selection.
  * 
  * @param pa_tEvent    mouse event
  */
 public void mousePressed(java.awt.event.MouseEvent pa_tEvent)
 {
  int lc_iIndex          = -1;
  int lc_iFirstSelection = -1;
  
  if (true == m_bLocked) return;

  if (null != pa_tEvent)
  {
   if (pa_tEvent.getSource() == jTableActiveSaveGames.getTableHeader())
   {
    if (0 <= (lc_iIndex = jTableActiveSaveGames.convertColumnIndexToModel(jTableActiveSaveGames.columnAtPoint(pa_tEvent.getPoint())))) 
    {
     if (lc_iIndex < SkyrimCharacterHelperConstants.SKH_TABLE_COLUMNS)
     {
      if (-1 != (lc_iFirstSelection = jTableActiveSaveGames.getSelectedRow()))
      {
       storeSaveGameSelection(true);
      }
     }
    }
   }  
   else if (pa_tEvent.getSource() == jTableBackupSaveGames.getTableHeader())
   {
    if (0 <= (lc_iIndex = jTableBackupSaveGames.convertColumnIndexToModel(jTableBackupSaveGames.columnAtPoint(pa_tEvent.getPoint())))) 
    {
     if (lc_iIndex < SkyrimCharacterHelperConstants.SKH_TABLE_COLUMNS)
     {
      if (-1 != (lc_iFirstSelection = jTableBackupSaveGames.getSelectedRow()))
      {
       storeSaveGameSelection(false);
      }
     }
    }
   }
  }
 }

 
 
 /**
  * MouseListener.mouseReleased. Handles list selections and table selections done by the user via mouse.
  * 
  * @param pa_tEvent    mouse event
  */
 public void mouseReleased(java.awt.event.MouseEvent pa_tEvent) 
 { 
  int lc_iIndex  = -1;
  int lc_iColumn = -1;
  int lc_iRow    = -1;
  
  if (true == m_bLocked) return;
  
  if (null != pa_tEvent && null != m_tGuiConnector )
  {
   //
   // no need for table header events here..
   //
   if (pa_tEvent.getSource() == jTableActiveSaveGames.getTableHeader()) return;
   if (pa_tEvent.getSource() == jTableBackupSaveGames.getTableHeader()) return;

   //
   // active characters list
   //
   if (pa_tEvent.getSource() == jListActiveCharacters)
   {
    if (-1 != (lc_iIndex = jListActiveCharacters.locationToIndex(pa_tEvent.getPoint())))
    {
     if (lc_iIndex < jListActiveCharacters.getModel().getSize())
     {
      m_tGuiConnector.handleListSelectionActivePlayers();  
     }
    }
   }
    
   //
   // backup characters list
   //
   if (pa_tEvent.getSource() == jListBackupCharacters) 
   {
    if (-1 != (lc_iIndex = jListBackupCharacters.locationToIndex(pa_tEvent.getPoint())))
    {
     if (lc_iIndex < jListBackupCharacters.getModel().getSize())
     {
      m_tGuiConnector.handleListSelectionBackupPlayers();  
     }
    }
   }
    
   //
   // active savegames list. ensures that clicks out of bounds do not cause problems.
   //
   if (pa_tEvent.getSource() == jTableActiveSaveGames && null != pa_tEvent.getPoint())
   {
    lc_iRow    = jTableActiveSaveGames.rowAtPoint   (pa_tEvent.getPoint());
    lc_iColumn = jTableActiveSaveGames.columnAtPoint(pa_tEvent.getPoint());

    if ((0 <= lc_iRow && lc_iRow < jTableActiveSaveGames.getModel().getRowCount()) && (0 <= lc_iColumn && lc_iColumn < jTableActiveSaveGames.getModel().getColumnCount()))
    {
     if (-1 != (lc_iIndex = jTableActiveSaveGames.convertRowIndexToModel(lc_iRow))) // conversion point -> view index -> model index
     {
      m_tGuiConnector.handleListSelectionActiveSaveGames();
     }
    }
   }
   
   //
   // backup savegames list. ensures that clicks out of bounds do not cause problems.
   //
   if (pa_tEvent.getSource() == jTableBackupSaveGames && null != pa_tEvent.getPoint())
   {
    lc_iRow    = jTableBackupSaveGames.rowAtPoint   (pa_tEvent.getPoint());
    lc_iColumn = jTableBackupSaveGames.columnAtPoint(pa_tEvent.getPoint());

    if ((0 <= lc_iRow && lc_iRow < jTableBackupSaveGames.getModel().getRowCount()) && (0 <= lc_iColumn && lc_iColumn < jTableBackupSaveGames.getModel().getColumnCount()))
    {
     if (-1 != (lc_iIndex = jTableBackupSaveGames.convertRowIndexToModel(lc_iRow)))  // conversion point -> view index -> model index
     {
      m_tGuiConnector.handleListSelectionBackupSaveGames();
     }
    }
   }
  }
 }

 
 
 /**
  * Checks for windows os
  * 
  * @return   true, if we're runni on windows, false otherwise
  */
 private boolean checkForWindows()
 {
  String lc_sOs = System.getProperty(SkyrimCharacterHelperConstants.SKH_OS_NAME); 

  if (null != lc_sOs) 
  {
   return lc_sOs.toUpperCase().startsWith(SkyrimCharacterHelperConstants.SKH_OS_WINDOWS);
  }
  return false;
 }
  

 
 /**
  * Selects the given player and displays its savegame list
  * 
  * @param pa_tPlayer 
  */
 public final void selectActivePlayer(SkyrimCharacterHelperPlayer pa_tPlayer)
 {
  SkyrimCharacterHelperPlayer     lc_tPlayer     = null; 
  SkyrimCharacterHelperListModel  lc_tModel      = (SkyrimCharacterHelperListModel) jListActiveCharacters.getModel();
  SkyrimCharacterHelperSaveGame[] lc_tSaveGames  = null;
  Object[]                        lc_tEntries    = null;
  int                             lc_iPos        = 0   ;
  
  if (null != pa_tPlayer && null != lc_tModel)
  {
   if (null != (lc_tEntries = lc_tModel.elements()))
   {
    //
    // process every list entry
    //
    for (Object lc_tObject : lc_tEntries) 
    {
     if (lc_tObject instanceof SkyrimCharacterHelperPlayer) 
     {
      if (null != (lc_tPlayer = (SkyrimCharacterHelperPlayer) lc_tObject))
      {
       if (false == lc_tPlayer.equals(pa_tPlayer))
       {
        lc_iPos++; 
       }
       else break;
      }
     }
    }
    
    if (-1 != lc_iPos)
    {
     jListActiveCharacters.getSelectionModel().clearSelection();
     jListActiveCharacters.setSelectedIndex(lc_iPos);
     
     //
     // init savegame list
     //
     if (null != (lc_tSaveGames = pa_tPlayer.getSaveGames())) 
     {
      initSaveGameList(lc_tSaveGames, true, false);
     }
    }
   }
  }
 }

 
 /**
  * Removes array keybindings from lists
  * 
  * @param pa_tList   list
  */
 private void removeArrowKeysFromList(javax.swing.JList pa_tList)
 {
  javax.swing.KeyStroke lc_tKeyRemove1 = javax.swing.KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_UP  , 0);
  javax.swing.KeyStroke lc_tKeyRemove2 = javax.swing.KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_DOWN, 0);
  javax.swing.InputMap  lc_tInputMap   = (null != pa_tList ? pa_tList.getInputMap() : null);
  
  if (null != lc_tInputMap)
  {
   if (null != lc_tKeyRemove1 && null != lc_tKeyRemove2)
   {
    lc_tInputMap.put(lc_tKeyRemove1, "none");
    lc_tInputMap.put(lc_tKeyRemove2, "none");
   }
  }
 }
 


  /**
  * Removes array keybindings from tables
  * 
  * @param pa_tTable   table
  */
 private void removeArrowKeysFromTable(javax.swing.JTable pa_tTable)
 {
  javax.swing.KeyStroke lc_tKeyRemove1 = javax.swing.KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_UP  , 0);
  javax.swing.KeyStroke lc_tKeyRemove2 = javax.swing.KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_DOWN, 0);
  javax.swing.InputMap  lc_tInputMap   = (null != pa_tTable ? pa_tTable.getInputMap(javax.swing.JTable.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT) : null);
  
  if (null != lc_tInputMap)
  {
   if (null != lc_tKeyRemove1 && null != lc_tKeyRemove2)
   {
    lc_tInputMap.put(lc_tKeyRemove1, "none");
    lc_tInputMap.put(lc_tKeyRemove2, "none");
   }
  }
 }
 
 
 
/**
 * check for one activ player
 * 
  * @return true, if 1 player is active in the active list
  * 
 */
 public final boolean hasOneActivePlayer()
 {
  return (1 == getListSize(jListActiveCharacters));
 }
 
 
/**
 * check for one activ player
 * 
  * @return true, if 1 player is active in the active list
  * 
 */
 public final boolean hasOneActiveSaveGame()
 {
  return (1 == getTableSize(jTableActiveSaveGames));
 }
 
 
 
/**
 * disables tableRow-Sorter for initializing the table with a whole set of entries
 * 
 */
 public final java.util.List disableTableRowSorter(javax.swing.JTable pa_tTable)
 {
  javax.swing.table.TableRowSorter lc_tRowSorter = (null != pa_tTable ? (javax.swing.table.TableRowSorter) pa_tTable.getRowSorter() : null);
  java.util.List                   lc_tSortKeys = null;
 
  if (lc_tRowSorter != null)
  {
   lc_tSortKeys = lc_tRowSorter.getSortKeys();
   lc_tRowSorter.setSortsOnUpdates(false);
  }
  return lc_tSortKeys;
 }
    
    
 
/**
 * enables tableRow-Sorter after initializing the table with a whole set of entries
 * 
 */
 public final void enableTableRowSorter(javax.swing.JTable pa_tTable, java.util.List pa_tSortKeys)
 {
  if (pa_tSortKeys != null)
  {
   javax.swing.table.TableRowSorter lc_tRowSorter = (null != pa_tTable ? (javax.swing.table.TableRowSorter) pa_tTable.getRowSorter() : null);
   
   if (null != lc_tRowSorter)
   {
    lc_tRowSorter.setSortsOnUpdates(true);
    lc_tRowSorter.setSortKeys(pa_tSortKeys);
   }
  }
 }
     
 
 
 /**
  * Returns the header arrangement 
  * 
  * @param pa_bMode   true = active savegames, false = backup savegames
  * @return header arrangement string
  */
 public final String getHeaderArrangement(boolean pa_bMode)
 {
  javax.swing.JTable lc_tTable = (true == pa_bMode ? jTableActiveSaveGames : jTableBackupSaveGames);
  StringBuffer lc_tStringBuffer = new StringBuffer();
  
  int lc_iI          = 0;
  int lc_iModelIndex = 0;
  
  if (null != lc_tStringBuffer && null != lc_tTable)
  {
   for (lc_iI = 0; lc_iI < 4; lc_iI++) 
   {
    lc_iModelIndex = lc_tTable.convertColumnIndexToModel(lc_iI);
    lc_tStringBuffer.append(Integer.toString(lc_iModelIndex)); // we store model incides to be able to restoring them in this order on next start
   }
  }
  return (null != lc_tStringBuffer ? lc_tStringBuffer.toString() : SkyrimCharacterHelperConstants.SKH_DEFAULT_HEADER_ARRANGEMENT);
 }
 

 
 /**
  * Returns the sort order setting for the current sort column in format
  * AB, where A is the coulmn number and B is the sortmode.
  * 
  * @param pa_bMode   true = active savegames, false = backup savegames
  * @return header arrangement string
  */
 public final String getSortOrder(boolean pa_bMode)
 {
  
  javax.swing.JTable                                      lc_tTable        = (true == pa_bMode ? jTableActiveSaveGames : jTableBackupSaveGames);
  javax.swing.SortOrder                                   lc_tSortOrder    = javax.swing.SortOrder.ASCENDING;
  StringBuffer                                            lc_tStringBuffer = new StringBuffer();
  java.util.List<? extends javax.swing.RowSorter.SortKey> lc_tSortKeys     = null;

  int      lc_iColumn = 0;
  boolean  lc_bOk     = false;
  
  if (null != lc_tTable && null != lc_tStringBuffer)
  {
   if (null != lc_tTable)
   {
    if (null != lc_tTable.getRowSorter())
    {
     if (null != (lc_tSortKeys = lc_tTable.getRowSorter().getSortKeys()))
     { 
      lc_iColumn    = lc_tSortKeys.get(0).getColumn();
      lc_tSortOrder = lc_tSortKeys.get(0).getSortOrder();

      lc_tStringBuffer.append(Integer.toString(lc_iColumn));
      if      (lc_tSortOrder == javax.swing.SortOrder.ASCENDING ) lc_tStringBuffer.append(Integer.toString(SkyrimCharacterHelperConstants.SKH_PROPERTIES_SORTORDER_ASCENDING ));
      else if (lc_tSortOrder == javax.swing.SortOrder.DESCENDING) lc_tStringBuffer.append(Integer.toString(SkyrimCharacterHelperConstants.SKH_PROPERTIES_SORTORDER_DESCENDING));
      else if (lc_tSortOrder == javax.swing.SortOrder.UNSORTED  ) lc_tStringBuffer.append(Integer.toString(SkyrimCharacterHelperConstants.SKH_PROPERTIES_SORTORDER_UNSORTED  ));

      lc_bOk = true;
     }
    }
   }
  }
  return (true == lc_bOk ? lc_tStringBuffer.toString() : SkyrimCharacterHelperConstants.SKH_DEFAULT_SORT_ORDER);
 }
 
 
 
 /**
  * Very primitive, but efficient check of the given header arrangement, in case of error the default order is returned
  * 
  * @param pa_sInput   input string from properties file
  * @return  order
  */
 private String checkHeaderArrangement(String pa_sInput) 
 {
  boolean lc_b0 = false;
  boolean lc_b1 = false;
  boolean lc_b2 = false;
  boolean lc_b3 = false;
  boolean lc_b4 = false;
  
  //
  // check input validity
  //
  if (null != pa_sInput)
  {
   if (4 == pa_sInput.length())
   {
    lc_b0 = (-1 != pa_sInput.indexOf("0"));
    lc_b1 = (-1 != pa_sInput.indexOf("1"));
    lc_b2 = (-1 != pa_sInput.indexOf("2"));
    lc_b3 = (-1 != pa_sInput.indexOf("3"));
   }
  }  
  lc_b4 = (lc_b0 & lc_b1 & lc_b2 & lc_b3);
  
  return (true == lc_b4 ? pa_sInput : SkyrimCharacterHelperConstants.SKH_DEFAULT_HEADER_ARRANGEMENT);
 }
 
 
 
 
 /**
  * Very primitive, but efficient check of the given sort order, in case of error the default order is returned
  * 
  * @param pa_sInput   input string from properties file
  * @return  order
  */
 private String checkSortOrder(String pa_sInput)
 {
  boolean lc_bOk = false;  
  int     lc_i1  = 0;
  int     lc_i2  = 0;
  
  //
  // check input validity
  //
  if (null != pa_sInput)
  {
   if (2 == pa_sInput.length())
   {
    lc_i1 = Character.digit(pa_sInput.charAt(0),10);
    lc_i2 = Character.digit(pa_sInput.charAt(1),10);

    if (0 <= lc_i1 && lc_i1 < SkyrimCharacterHelperConstants.SKH_TABLE_COLUMNS)
    {
     if (lc_i2 == 0 || lc_i2 == 1 || lc_i2 == 2) 
     {
      lc_bOk  = true;     
     }
    }
   }
  }  
  return (true == lc_bOk ? pa_sInput : SkyrimCharacterHelperConstants.SKH_DEFAULT_SORT_ORDER);
 }
 
 
 
 /**
  * Sets the sort order and the active sort column
  * 
  * @param pa_tRowSorter   rowsorter
  * @param pa_sValue       sort order string
  * @param pa_tTable       table
  */
 private int setSortOrder(javax.swing.table.TableRowSorter<javax.swing.table.TableModel> pa_tRowSorter, String pa_sValue, javax.swing.JTable pa_tTable) 
 {        
  java.util.List<javax.swing.RowSorter.SortKey>           lc_tSortKeys    = new java.util.ArrayList<>();
  javax.swing.SortOrder                                   lc_tSortOrder   = javax.swing.SortOrder.ASCENDING;
  
  int lc_iColumn    = 0;
  int lc_iSortOrder = 0;
  
  if (null != pa_sValue && null != pa_tRowSorter)
  {
   if (2 == pa_sValue.length())
   {
    lc_iColumn    = Character.digit(pa_sValue.charAt(0),10);
    lc_iSortOrder = Character.digit(pa_sValue.charAt(1),10);

    switch(lc_iSortOrder)
    {
     case SkyrimCharacterHelperConstants.SKH_PROPERTIES_SORTORDER_ASCENDING : lc_tSortOrder = javax.swing.SortOrder.ASCENDING ; break;
     case SkyrimCharacterHelperConstants.SKH_PROPERTIES_SORTORDER_DESCENDING: lc_tSortOrder = javax.swing.SortOrder.DESCENDING; break;
     case SkyrimCharacterHelperConstants.SKH_PROPERTIES_SORTORDER_UNSORTED  :
     default                                                                : lc_tSortOrder = javax.swing.SortOrder.UNSORTED; 
    }
    
    if (null != lc_tSortKeys)
    {
     lc_tSortKeys.add(new javax.swing.RowSorter.SortKey(lc_iColumn, lc_tSortOrder));
     pa_tRowSorter.setSortKeys(lc_tSortKeys); 
     pa_tRowSorter.toggleSortOrder(lc_iColumn); // first toggle activates       
     pa_tRowSorter.toggleSortOrder(lc_iColumn); // second flips back to what it shoule be according to settings      
    }
   }
  }
  return lc_iColumn;
 }

    
 
 /**
  * Sets the race for the active player
  * 
  * @param pa_sRace   race
  */
 public final void setActivePlayerRace(String pa_sRace) {jTextFieldRaceActivePlayer.setText(pa_sRace); jTextFieldRaceActivePlayer.setCaretPosition(0);}

 

 /**
  * Sets the race for the backup player
  * 
  * @param pa_sRace   race
  */
 public final void setBackupPlayerRace(String pa_sRace) {jTextFieldRaceBackupPlayer.setText(pa_sRace); jTextFieldRaceBackupPlayer.setCaretPosition(0);}

 
 
 /**
  * Restores a savegame selection
  *  
  * @param pa_bMode    true = Skyrim savegames, false = backup savegames
  */
 public final synchronized void restoreSaveGameSelection(boolean pa_bMode, boolean pa_bNotify)
 {
  if (true == pa_bMode) reselectSaveGames(m_tStoredActiveSaveGameSelection, true , pa_bNotify);
  else                  reselectSaveGames(m_tStoredBackupSaveGameSelection, false, pa_bNotify);
 }
 
 
 
 /**
  * Restores a savegame selection
  *  
  * @param pa_bMode    true = Skyrim savegames, false = backup savegames
  */
 public final synchronized void restorePlayerSelection(boolean pa_bMode)
 {
  if (true == pa_bMode) reselectPlayers(m_tStoredActivePlayerSelection, true);
  else                  reselectPlayers(m_tStoredBackupPlayerSelection, false);
 }
 
 
 
 /**
  * Stores a savegame selection
  *  
  * @param pa_bMode    true = Skyrim savegames, false = backup savegames
  */
 public final synchronized void storeSaveGameSelection(boolean pa_bMode)
 {
  if (true == pa_bMode) m_tStoredActiveSaveGameSelection = getSelectedActiveSaveGames();
  else                  m_tStoredBackupSaveGameSelection = getSelectedBackupSaveGames();
   
 }
 
 
 
 /**
  * Stores a player selection
  *  
  * @param pa_bMode    true = Skyrim savegames, false = backup savegames
  */
 public final synchronized void storePlayerSelection(boolean pa_bMode)
 {
  if (true == pa_bMode) m_tStoredActivePlayerSelection = getSelectedActivePlayers();
  else                  m_tStoredBackupPlayerSelection = getSelectedBackupPlayers();
   
 }
 
 
 /**
  * Enables or disables image display
  * 
  * @param pa_bMode   true ) show image area, false = dont
  */
 public final void setImageModeDisplay(boolean pa_bMode)
 {
  jPanelActiveCharacterData.setVisible(pa_bMode);
  jPanelBackupCharacterData.setVisible(pa_bMode); 
  //pack();
 }
 
 
 
 /**
  * Returns the prefix
  * 
  * @return prefix, may be null
  */
 public final String getPrefix() 
 {
  String lc_sText = jTextFieldPrefix.getText();   
  
  return (null != lc_sText ? lc_sText.trim() : null);
 }
 
 
 
 /**
  * Sets the prefix
  * 
  * @param pa_sValue   prefix
  */
 public final void setPrefix(String pa_sValue) {jTextFieldPrefix.setText(pa_sValue); jTextFieldPrefix.setCaretPosition(0);}

 
  
 /**
  * Handles a change-event any text-field. The Ok button gets enabled as soon
  * as text-length is greater than KSA_MIN_LENGTH chars and both texts are equal.
  * 
  * @param pa_tEvent 
  */
 public void changedUpdate(javax.swing.event.DocumentEvent pa_tEvent) 
 {
  updateUIElements();
 }
  
  
 /**
  * Handles a remove-event any text-field. The Ok button gets enabled as soon
  * as text-length is greater than KSA_MIN_LENGTH chars and both texts are equal.
  * 
  * @param pa_tEvent 
  */
 public void removeUpdate(javax.swing.event.DocumentEvent pa_tEvent) 
 {
  updateUIElements();
 }

  
 /**
  * Handles an insert-event any text-field. The Ok button gets enabled as soon
  * as text-length is greater than KSA_MIN_LENGTH chars and both texts are equal.
  * 
  * @param pa_tEvent 
  */
 public void insertUpdate(javax.swing.event.DocumentEvent pa_tEvent) 
 {
  updateUIElements();
 }

  
 /**
  * Sets the On Skyrim label display. Warning: The component must be opaque!!
  * 
  * @param pa_bState 
  */
 public final void setOnSkyrim(boolean pa_bState)
 {
  jLabelOnSkyrim.setEnabled(pa_bState); // toggle icon :)
  jLabelOnSkyrim.setBackground(true == pa_bState ? m_tLabelOnSkyrimActiveBg : m_tLabelOnSkyrimNormalBg);
  jLabelOnSkyrim.setText(true == pa_bState ? m_tMonitoringDisplays[SkyrimCharacterHelperConstants.SKH_MONITORING    ] 
                                           : m_tMonitoringDisplays[SkyrimCharacterHelperConstants.SKH_NOT_MONITORING]);            
 }
 
 
 /**
  * Makes suire that the selected entry is visible
  * 
  * @param pa_bMode   true = active savegames, false = backup savegames
  */
 public final void ensureSaveGameSelectionIsVisible(boolean pa_bMode)
 {
  javax.swing.JTable lc_tTable = (true == pa_bMode ? jTableActiveSaveGames : jTableBackupSaveGames);
  int                lc_iIndex = (null != lc_tTable ? lc_tTable.getSelectedRow() : -1);
  
   //
   // scroll to last visible selection
   //
   if (-1 != lc_iIndex)
   {
    lc_tTable.scrollRectToVisible(new java.awt.Rectangle(lc_tTable.getCellRect(lc_iIndex, 0, true)));
   }   
 }

 
  
 /**
  * Removes selections from savegames tables and void the image
  * 
  * @param pa_bMode   true = active savegames, false = backup savegames
  */
 public final void removeSaveGameSelections(boolean pa_bMode)
 {
  javax.swing.JTable lc_tTable = (true == pa_bMode ? jTableActiveSaveGames : jTableBackupSaveGames);
   
  if (null != lc_tTable)
  {
   if (null != lc_tTable.getSelectionModel()) lc_tTable.getSelectionModel().clearSelection();
  }
  
  if (true == pa_bMode) setActiveSaveGameScreenshot(null);
  else                  setBackupSaveGameScreenshot(null);
 }
 
 
 
 
} // eoc