/**
 * CharMate
 * 
 */
package charmate;

import java.awt.Dimension     ;
import java.awt.Font          ;
import java.awt.font.FontRenderContext;
import java.awt.font.LineMetrics      ;

import java.awt.Color         ;
import java.awt.Graphics      ;
import java.awt.Graphics2D    ;
import java.awt.RenderingHints;
import javax.swing.JPanel     ;


/**
 * Basic paint panel for drawing the code point
 * 
 * @author Ulf Wagemann
 */
public class CharMatePaintPanel extends JPanel
{
 private RenderingHints m_tRenderingHints ;
 private Color          m_tColorBackground;
 private Color          m_tDrawColor      ;
 private JPanel         m_tPanelParent    ;
 private Font           m_tFont           ;
 private int            m_iCodePoint      ;
 private int[]          m_tFontHeights    ;
 
 
 /**
  * Constructor
  * 
  */
 public CharMatePaintPanel(JPanel pa_tParentPanel)  
 {
  super();  

  m_iCodePoint          = -1;
  m_tColorBackground   = getBackground();
  m_tDrawColor         = Color.black    ;
  m_tPanelParent       = pa_tParentPanel;
  m_tRenderingHints    = new RenderingHints(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);   
  m_tFont              = new Font(CharMateConstants.CHM_DEFAULT_FONT, Font.BOLD, 36);
  m_tFontHeights       = new int[] {36,48,54,72,96,128,146,172,196,224};
 }
 
 
 
 /**
  * Sets the codepoint to the given value
  * 
  * @param pa_iCodePoint   code base
  */
 public void setCodePoint(int pa_iCodePoint)
 {
  m_iCodePoint = pa_iCodePoint;
  repaint();
 }
 
 
 
 /**
  * Returns the code point
  * 
  * @return  code point
  */
 public int getCodePoint() {return m_iCodePoint;}
 
 
 
 /**
  * Sets the best font height
  * 
  * @param pa_iMaxHeight   maximum height in px
  */
 private Dimension setBestFontHeight(Graphics2D pa_tGraphics2D, Dimension pa_tMaxSize)
 {
  int lc_iSize = 0;
  int lc_iDx   = 0;
  int lc_iDy   = 0;
  
  Dimension         lc_tTextDimension     = null;
  Font              lc_tFont              = null;
  FontRenderContext lc_tFontRenderContext = null;
  LineMetrics       lc_tLineMetrics       = null;
  String            lc_sFontName          = (null != m_tFont ? m_tFont.getName() : CharMateConstants.CHM_DEFAULT_FONT);
  String            lc_sDisplay           = getDisplayString();
  
  if (null != pa_tMaxSize && null != m_tFontHeights && null != pa_tGraphics2D && null != lc_sDisplay)
  {
   //
   // compute best font height for the display string
   //
   for (int lc_iFontSize : m_tFontHeights)
   {
    if (null != (lc_tFont = new Font(lc_sFontName, Font.BOLD, lc_iFontSize)))
    {
     pa_tGraphics2D.setFont(lc_tFont);
    
     if (null != (lc_tFontRenderContext = pa_tGraphics2D.getFontRenderContext()))
     {
      lc_iDx = (int) lc_tFont.getStringBounds(lc_sDisplay, lc_tFontRenderContext).getWidth(); 
      
      if (null != (lc_tLineMetrics = lc_tFont.getLineMetrics(lc_sDisplay, lc_tFontRenderContext)))
      {
       lc_iDy = (int) lc_tLineMetrics.getHeight(); 
       
       if (lc_iDy < pa_tMaxSize.height && lc_iDx < pa_tMaxSize.width)
       {
        //
        // store font. correct text dimensions by descent and leading, because we need to compute the baseline
        // 
        lc_iSize = lc_iFontSize; 
        lc_iDy  -= (int) lc_tLineMetrics.getAscent();
        lc_iDy  += (int) lc_tLineMetrics.getDescent();
        lc_tTextDimension = new Dimension(lc_iDx, lc_iDy);
       }
       else
         break;
      }
     }
    }    
   }// for

   //
   // set new font and store it
   //
   if (null != (m_tFont = new Font(lc_sFontName, Font.BOLD, lc_iSize)))
   {
    pa_tGraphics2D.setFont(m_tFont); 
   }   
  }
  return lc_tTextDimension;
 }
 
 
 
 /**
  * Sets a new display Font
  * 
  * @param pa_tFont 
  */
 public void setDisplayFont(String pa_sFont)
 {
  m_tFont = new Font(pa_sFont, Font.BOLD, (null != m_tFont ? m_tFont.getSize() : 36));
  repaint();
 }
 
 
 
 /**
  * Returns the display font
  * 
  * @return   font name
  */
 public String getDisplayFont()
 {
  return (null != m_tFont ? m_tFont.getName() : null);  
 }
 
 
 
 /**
  * Encodes the current codepoint to a unicode string
  * @return 
  */
 private String getDisplayString()
 {
  char[] lc_tDisplayChars = null;
  String lc_sDisplay      = null;
   
  if (null != (lc_tDisplayChars = Character.toChars(m_iCodePoint)))
  {
   if (null != (lc_sDisplay = new String(lc_tDisplayChars)))
   {
    return lc_sDisplay;
   }
  }
  return null;
 }
 
 
 
 /**
  * Overriden paintComponent() method which first lets the superclass perform its drawings and then
  * calls the custom drawing method.
  *  06B3
  * @param pa_tGraphics Paint on it!
  */ 
 @Override public void paintComponent(Graphics pa_tGraphics)
 {
  super.paintComponent(pa_tGraphics);  
  
  Graphics2D  lc_tGraphics2D         = null;
  Dimension   lc_tDimensionContainer = null;
  Dimension   lc_tTextDimension      = null;
  
  
  int lc_iX  = 0;
  int lc_iY  = 0;
  int lc_iDx = 0;
  int lc_iDy = 0;
  
  String lc_sDisplay      = null;
  
  if (0 < m_iCodePoint)
  {
   if (null != (lc_sDisplay = getDisplayString()))
   {
    if (null != pa_tGraphics && null != lc_sDisplay)
    {
     if (null != (lc_tGraphics2D = (Graphics2D) pa_tGraphics))
     {
      //
      // set rendering hints and font
      //
      lc_tGraphics2D.setRenderingHints(m_tRenderingHints);
      lc_tGraphics2D.setFont(m_tFont);

      //
      // draw background
      //
      if (null != (lc_tDimensionContainer = getSize())) 
      {
       lc_tGraphics2D.setColor(m_tColorBackground); 
       lc_tGraphics2D.fillRect(0, 0, lc_tDimensionContainer.width, lc_tDimensionContainer.height);
      }

      //
      // obtain fontmetrics object and compute text width
      //
      if (null != (lc_tTextDimension = setBestFontHeight(lc_tGraphics2D, lc_tDimensionContainer)))
      {
       lc_iDx = lc_tTextDimension.width;
       lc_iDy = lc_tTextDimension.height;
       
       lc_iX = (lc_tDimensionContainer.width  - lc_iDx)/2;
       lc_iY = (lc_tDimensionContainer.height + lc_iDy)/2;
        
       lc_tGraphics2D.setColor(m_tDrawColor);
       lc_tGraphics2D.drawString(lc_sDisplay, lc_iX, lc_iY); // convert unicode-compliant!
      }
     }
    }
   }
  }
 }
 
 
 
 
 /**
  * Returns preferred size
  * 
  * @return   preferred size
  */
 @Override
 public Dimension getPreferredSize()
 {
  Dimension lc_tParentSize = (null != m_tPanelParent ? m_tPanelParent.getSize() : null);
  
  if (null != lc_tParentSize)
  {
   return new Dimension(lc_tParentSize.width-10, lc_tParentSize.height-10); 
  }
  return new Dimension(100,100);
 }
} // eoc
