/**
 * FileSafe
 * 
 */
package filesafe;




/**
 * Imports
 * 
 */
import java.io.IOException     ;
import java.io.File            ;
import java.io.FileInputStream ;
import java.io.FileOutputStream;

import java.util.ResourceBundle;

import org.jdesktop.application.Application           ;
import org.jdesktop.application.SingleFrameApplication;

import java.awt.Toolkit                     ;
import java.awt.event.WindowListener        ;
import java.awt.event.WindowEvent           ;
import java.awt.datatransfer.StringSelection;

import javax.swing.JFrame      ;
import javax.swing.JFileChooser;
import javax.swing.JOptionPane ;
import javax.swing.SwingWorker ;

import javax.crypto.spec.PBEKeySpec      ;
import javax.crypto.spec.PBEParameterSpec;
import javax.crypto.SecretKey            ;
import javax.crypto.SecretKeyFactory     ;
import javax.crypto.Cipher               ;
import javax.crypto.CipherOutputStream   ;
import javax.crypto.CipherInputStream    ;
import java.security.SecureRandom        ;



/**
 * The main class of the application.
 * 
 * @author Ulf Wagemann
 */
public class FileSafeApp extends SingleFrameApplication implements FileSafeGuiConnector,
                                                                   WindowListener
{
 /**
  * Member
  * 
  */
 private FileSafeView    m_tView             ;
 private ResourceBundle  m_tLocalizedMessages;
 
 private String          m_sEncryptFilePath       ;
 private String          m_sEncryptDestinationPath;
 private String          m_sDecryptFilePath       ;
 private String          m_sDecryptDestinationPath;
 private String          m_sPassword              ;

 private FileSafeWorker m_tWorkerThread;
 

 /**--------------------------------------------------------------------------------------------------------------------
  * 
  * Internal class needed to update the progress bars properly. Since lengthy stuff needs to be done inside here,
  * this class also contains the crypto functionality.
  * 
  */
 class FileSafeWorker extends SwingWorker
 {
  private Object[] m_tEncryptFiles; 
  private Object[] m_tDecryptFiles; 
  private String   m_sEncryptDir  ;
  private String   m_sDecryptDir  ;
  private int      m_iEncryptedOk ;
  private int      m_iDecryptedOk ;
  private int      m_iEncryptedAmount ;
  private int      m_iDecryptedAmount;
  
  /**
   * Constructor
   * 
   */
  FileSafeWorker()
  {
   super();
   
   m_tEncryptFiles = null;
   m_tDecryptFiles = null;
   m_sEncryptDir   = null;
   m_sDecryptDir   = null;
   
   m_iEncryptedOk = 0;
   m_iDecryptedOk = 0;
   m_iEncryptedAmount = 0;
   m_iDecryptedAmount = 0;

  }
  
  
  
  /**
   * Initialization
   * 
   */
  public void init()
  {
   m_tDecryptFiles = m_tView.getDecryptFiles();
   m_tEncryptFiles = m_tView.getEncryptFiles();
   
   m_sDecryptDir = m_tView.getDecryptDestination();
   m_sEncryptDir = m_tView.getEncryptDestination();
  }
  
  
  
  /**
   * This is where the lengthy stuff happens
   * 
   * @return null
   */
  public Object doInBackground() 
  {
   File lc_tFile   = null;
   int  lc_iCount  = 0   ;
   int  lc_iSize   = 0   ;
   int  lc_iToGo   = 0   ;
   int  lc_iLength = 0   ;
   
   String lc_sDestinationFileName = null;
   String lc_sSeparator           = System.getProperty("file.separator");
   String lc_sFileName            = null;
   String lc_sLogMessage          = null;
   
   m_iEncryptedAmount = (null != m_tEncryptFiles ? m_tEncryptFiles.length : 0);
   m_iDecryptedAmount = (null != m_tDecryptFiles ? m_tDecryptFiles.length : 0);
           
   lc_iSize = m_iDecryptedAmount + m_iEncryptedAmount;
   lc_iToGo = lc_iSize;
   
   if (0 < lc_iSize)
   {
    m_tView.initTotalProgress(lc_iSize);
    m_tView.setProgressTotalString((new Integer(lc_iSize)).toString());
    
    //
    // iterate over encryption files
    //
    if (null != m_tEncryptFiles)
    {
     if (0 < m_tEncryptFiles.length)
     {
      m_tView.setProgressModeString(getLocalizedString(FileSafeConstants.FSA_MODE_ENCRYPTING));
      m_iEncryptedOk = 0;
      
      for (Object lc_tFileName : m_tEncryptFiles)
      {
       //
       // check for interruption
        //
       if (true == isCancelled()) 
       {
        return null;
       } 
       
       m_tView.setProgressCurrentString(Integer.toString(lc_iToGo--));
       
       if (lc_tFileName instanceof String)
       {
        if (null != (lc_sFileName = (String) lc_tFileName))
        {
         lc_sLogMessage = getLocalizedString(FileSafeConstants.FSA_MSG_ENCRYPTING) + " " + lc_sFileName + " > ";
         
         if (null != (lc_tFile = new File((String) lc_tFileName)))
         {
          if (true == lc_tFile.isFile())
          {
           lc_sDestinationFileName  = m_sEncryptDir + lc_sSeparator + lc_tFile.getName();
           lc_iLength               = (int) lc_tFile.length()                           ;
           lc_sLogMessage          += lc_sDestinationFileName + " ... "                 ;
           
           m_tView.setCurrentFile(lc_sFileName);
           m_tView.initCurrentProgress(lc_iLength);
           
           if (true == encrypt(lc_sFileName, lc_iLength, lc_sDestinationFileName, m_sPassword))
           {
            m_iEncryptedOk++;
            lc_sLogMessage += getLocalizedString(FileSafeConstants.FSA_MSG_RESULT_OK);
           }
           else
           {
            if (true == isCancelled())
            {
             lc_sLogMessage += getLocalizedString(FileSafeConstants.FSA_MSG_CANCELLED);                          
            }
            else
            {       
             lc_sLogMessage += getLocalizedString(FileSafeConstants.FSA_MSG_RESULT_ERROR);             
            }
           }
          }
          m_tView.addLogMessage(lc_sLogMessage);
          if (false == isCancelled()) m_tView.setTotalProgress(++lc_iCount);
         }
        }
       } // for
      }
     }
    }
    
    //
    // iterate over decryption files
    //
    if (null != m_tDecryptFiles)
    {
     if (0 < m_tDecryptFiles.length)
     {
      m_tView.setProgressModeString(getLocalizedString(FileSafeConstants.FSA_MODE_DECRYPTING));
      m_iDecryptedOk = 0;
      
      for (Object lc_tFileName : m_tDecryptFiles)
      {
       //
       // check for interruption
        //
       if (true == isCancelled()) 
       {
        return null;
       } 
        
       m_tView.setProgressCurrentString(Integer.toString(lc_iToGo--));
       
       if (lc_tFileName instanceof String)
       {
        lc_sFileName   = (String) lc_tFileName;
        lc_sLogMessage = getLocalizedString(FileSafeConstants.FSA_MSG_DECRYPTING) + " " + lc_sFileName + " > ";

        if (null != (lc_tFile = new File((String) lc_tFileName)))
        {
         if (true == lc_tFile.isFile())
         {
          lc_sDestinationFileName  = m_sDecryptDir + lc_sSeparator + lc_tFile.getName();
          lc_iLength               = (int) lc_tFile.length()                           ;
          lc_sLogMessage          +=  lc_sDestinationFileName + " ... ";

          m_tView.setCurrentFile((String) lc_tFileName);
          m_tView.initCurrentProgress(lc_iLength);
          
          if (true == decrypt(lc_sFileName, lc_iLength, lc_sDestinationFileName, m_sPassword))
          {
           m_iDecryptedOk++;
           lc_sLogMessage += getLocalizedString(FileSafeConstants.FSA_MSG_RESULT_OK);           
          }
          else
          {
           if (true == isCancelled())
           {
            lc_sLogMessage += getLocalizedString(FileSafeConstants.FSA_MSG_CANCELLED);                          
           }
           else
           {      
            lc_sLogMessage += getLocalizedString(FileSafeConstants.FSA_MSG_RESULT_ERROR);             
           }
          }
         }
         m_tView.addLogMessage(lc_sLogMessage);
         if (false == isCancelled()) m_tView.setTotalProgress(++lc_iCount);
        }
       }
      } // for
     }
    }
   }   
   return null;
  }
  
  
  
  /**
   * When we're done, we speak up
   * 
   */
  @Override
  protected void done()
  {
   String lc_sMessage = "";
   
   if (0 < getEncryptedAmount())
   {
    lc_sMessage += getEncryptedOk() + "/" + getEncryptedAmount() + " " + 
                   getLocalizedString(FileSafeConstants.FSA_DIALOG_TEXT_RESULT_ENCRYPTION) + "\n" ; 
   }
   if (0 < getDecryptedAmount())
   {
    lc_sMessage += getDecryptedOk() + "/" + getDecryptedAmount() + " " + 
                   getLocalizedString(FileSafeConstants.FSA_DIALOG_TEXT_RESULT_DECRYPTION) + "\n" ; 
   }
   handleWorkerThreadFinished(lc_sMessage);
  }
  
  
  
  /**
   * Returns the encryption results
   * 
   * @return   encryption results
   */
  public int getEncryptedOk() {return m_iEncryptedOk;}
  

  
  /**
   * Returns the amount of files for encryption
   * 
   * @return   amount 
   */
  public int getEncryptedAmount() {return m_iEncryptedAmount;}
  
  
  
  /**
   * Returns the decryption results
   * 
   * @return   decryption results
   */
  public int getDecryptedOk() {return m_iDecryptedOk;}  


  
  /**
   * Returns the amount of files for decryption
   * 
   * @return   amount 
   */
  public int getDecryptedAmount() {return m_iDecryptedAmount;}
  
  
  
 /**
  * Writes the byte array to stream
  * 
  * @param pa_tStream   output stream
  * @param pa_tBytes    byte array to be written
  */
 private void writeBytes(FileOutputStream pa_tStream, byte[] pa_tBytes)
 {
  try 
  {
   pa_tStream.write(pa_tBytes);
  } 
  catch (java.io.IOException lc_tException) 
  {}
 }
 
 
 
 /**
  * Creates a new empty file
  * 
  * @param pa_sFileName   file name
  */
 private void createNewFile(String pa_sFileName)
 {
  File lc_tFile = new File(pa_sFileName);  
 
  if (null != lc_tFile)
  {
   try 
   {
    lc_tFile.delete(); 
    lc_tFile.createNewFile();
   } 
   catch (java.io.IOException lc_tException) 
   {}
  }
 }
 


 /**
  * Delets the given file
  * 
  * @param pa_sFileName   file name
  */
 private void deleteFile(String pa_sFileName)
 {
  File lc_tFile = new File(pa_sFileName);  
 
  if (null != lc_tFile)
  {
   lc_tFile.delete(); 
  }
 }
 
 
 
 /**
  * Initializes a cipher object 
  * 
  * @param  pa_tSalt        salt to use
  * @param  pa_sPassword    password to use
  * @param  pa_iCipherMode  Cipher.ENCRYPT_MODE  or Cipher.DECRYPT_MODE
  * @return null in case of error, otherwise a configured cipher object
  */
 private Cipher setupCryptoStuff(byte[] pa_tSalt, String pa_sPassword, int pa_iCipherMode)
 {
  PBEKeySpec       lc_tKeySpec     = null;
  PBEParameterSpec lc_tParamSpec   = null;
  SecretKeyFactory lc_tKeyFactory  = null;
  SecretKey        lc_tPasswordKey = null;
  Cipher           lc_tCipher      = null;

  boolean      lc_bCryptoOk       = false;
  
  //
  // prepare crypto stuff
  //
  try 
  { 
   if (null != (lc_tKeySpec = new PBEKeySpec(pa_sPassword.toCharArray())))
   {
    if (null != (lc_tKeyFactory = SecretKeyFactory.getInstance(FileSafeConstants.FSA_ALGORITHM)))
    {
     try
     {
      if (null != (lc_tPasswordKey = lc_tKeyFactory.generateSecret(lc_tKeySpec)))
      {
       if (null != (lc_tParamSpec = new PBEParameterSpec(pa_tSalt, 100)))
       {
        try 
        {
         if (null != (lc_tCipher = Cipher.getInstance(FileSafeConstants.FSA_ALGORITHM)))
         {
          lc_tCipher.init(pa_iCipherMode, lc_tPasswordKey, lc_tParamSpec); 
          lc_bCryptoOk = true;   
         }
        }
        catch (java.security.InvalidKeyException                 lc_tException) {} 
        catch (java.security.InvalidAlgorithmParameterException  lc_tException) {}
        catch (javax.crypto.NoSuchPaddingException               lc_tException) {}
       }
      }
     }
     catch (java.security.spec.InvalidKeySpecException lc_tException) 
     {}
    }
   }
  } 
  catch (java.security.NoSuchAlgorithmException lc_tException) 
  {} 
  return (true == lc_bCryptoOk ? lc_tCipher : null);
 }
 
 
 
 /**
  * Encrypts a file using a password.
  * 
  * @param pa_sSourceFileName       source file name
  * @param pa_iLength               source file size in bytes
  * @param pa_sDestinationFileName  destination file anme
  * @param pa_sPassword             encryption password
  * @return true, if encrypting was successful, false otherwise
  */
 public final boolean encrypt(String pa_sSourceFileName, int pa_iLength, String pa_sDestinationFileName, String pa_sPassword)
 {
  boolean lc_bResult            = false          ;
  byte[]  lc_tSalt              = new byte[8]    ;
  byte[]  lc_tInputBytes        = new byte[131072];
  int     lc_lBytesRead         = 0              ;
  long    lc_lTotalBytesRead    = 0              ;
  
  FileInputStream     lc_tFileInputStream    = null;
  FileOutputStream    lc_tFileOutputStream   = null;
  CipherOutputStream  lc_tCipherOutputStream = null;
  Cipher              lc_tCipher             = null;
  SecureRandom        lc_tRandom             = new SecureRandom();
  
  if (null != lc_tSalt              && 
      null != lc_tRandom            && 
      null != pa_sPassword          && 
      null != pa_sSourceFileName    && 
      null != pa_sDestinationFileName )
  {
   //
   // prepare crypto stuff
   //
   lc_tRandom.nextBytes(lc_tSalt);
   
   if (null != (lc_tCipher = setupCryptoStuff(lc_tSalt, pa_sPassword, Cipher.ENCRYPT_MODE)))
   {
    //
    // create new empty file
    //
    createNewFile(pa_sDestinationFileName);
    
    //
    // open input stream
    //
    try
    {
     if (null != (lc_tFileInputStream = new FileInputStream(pa_sSourceFileName)))
     {
      //
      // open outputstream
      //
      try
      {
       if (null != (lc_tFileOutputStream = new FileOutputStream(pa_sDestinationFileName)))
       {
        //
        // write salt uncrypted
        //
        writeBytes(lc_tFileOutputStream, lc_tSalt);

        if (null != (lc_tCipherOutputStream = new CipherOutputStream(lc_tFileOutputStream, lc_tCipher)))
        {
         //
         // encode file
         //
         while (-1 != lc_lBytesRead && false == isCancelled()) // check for cancellation
         {
          lc_lBytesRead = lc_tFileInputStream.read(lc_tInputBytes); 
          lc_lTotalBytesRead += lc_lBytesRead;
          if (false == isCancelled()) m_tView.setCurrentProgress((int) lc_lTotalBytesRead);
          
          if (0 < lc_lBytesRead)
          {
           lc_tCipherOutputStream.write(lc_tInputBytes, 0, lc_lBytesRead); 
          }
         } // while         
         lc_bResult = true;
        }         
       }
      }
      catch(java.io.IOException lc_tException) {}      
     }
    }
    catch (java.io.IOException lc_tException) {}
   }
  }
  
  //
  // for safety, we close all stuff at the end
  //
  if (null != lc_tCipherOutputStream) {try {lc_tCipherOutputStream.flush(); lc_tCipherOutputStream.close();} catch (java.io.IOException lc_tException) {}} // close CipherOutputStream
  if (null != lc_tFileOutputStream  ) {try {lc_tFileOutputStream.flush()  ; lc_tFileOutputStream.close()  ;} catch (java.io.IOException lc_tException) {}} // close output file
  if (null != lc_tFileInputStream   ) {try {                                lc_tFileInputStream.close()   ;} catch (java.io.IOException lc_tException) {}} // close input stream
 
  //
  // check for cancellation
  //
  if (true == isCancelled())
  {
   deleteFile(pa_sDestinationFileName); 
   lc_bResult = false;
  }
  return lc_bResult;
 }
 

 
 /**
  * Decrypts a file using a password.
  * 
  * @param pa_sSourceFileName       source file name
  * @param pa_iLength               source file size in bytes
  * @param pa_sDestinationFileName  destination file anme
  * @param pa_sPassword             decryption password
  * @return true, if decrypting was successful, false otherwise
  */
 public final boolean decrypt(String pa_sSourceFileName, int pa_iLength, String pa_sDestinationFileName, String pa_sPassword)
 {
  boolean lc_bResult         = false          ;
  byte[]  lc_tSalt           = new byte[8]    ;
  byte[]  lc_tInputBytes     = new byte[131072];
  int     lc_lBytesRead      = 0              ;
  long    lc_lTotalBytesRead = 0              ;
  
  FileInputStream     lc_tFileInputStream    = null;
  FileOutputStream    lc_tFileOutputStream   = null;
  CipherInputStream   lc_tCipherInputStream  = null;
  Cipher              lc_tCipher             = null;
  SecureRandom        lc_tRandom             = new SecureRandom();
  
  if (null != lc_tSalt              && 
      null != lc_tRandom            && 
      null != pa_sPassword          && 
      null != pa_sSourceFileName    && 
      null != pa_sDestinationFileName )
  {
   //
   // open input stream
   //
   try
   {
    if (null != (lc_tFileInputStream = new FileInputStream(pa_sSourceFileName)))
    {
     //
     // get salt
     //
     lc_lBytesRead = lc_tFileInputStream.read(lc_tSalt);  
     
     if (8 == lc_lBytesRead)
     {
      //
      // prepare crypto stuff
      //
      if (null != (lc_tCipher = setupCryptoStuff(lc_tSalt, pa_sPassword, Cipher.DECRYPT_MODE)))
      {
       //
       // create new empty file
       //
       createNewFile(pa_sDestinationFileName);
       
       //
       // open outputstream
       //
       try
       {
        if (null != (lc_tFileOutputStream = new FileOutputStream(pa_sDestinationFileName)))
        {
         if (null != (lc_tCipherInputStream = new CipherInputStream(lc_tFileInputStream, lc_tCipher)))
         {
          //
          // decode file
          //
          lc_lBytesRead = 0;
          
          while (-1 != lc_lBytesRead && false == isCancelled()) // check for cancellation)
          {
           lc_lBytesRead = lc_tCipherInputStream.read(lc_tInputBytes); 
           lc_lTotalBytesRead += lc_lBytesRead;
           if (false == isCancelled()) m_tView.setCurrentProgress((int) lc_lTotalBytesRead);
           
           if (0 < lc_lBytesRead)
           {
            lc_tFileOutputStream.write(lc_tInputBytes, 0, lc_lBytesRead); 
           }
          } // while
          lc_bResult = true;
         }
        }
       }
       catch(java.io.IOException lc_tException) {}
      }
     }
    }
   }
   catch (java.io.IOException lc_tException) {}
  }

  //
  // for safety, we close all stuff at the end
  //
  if (null != lc_tCipherInputStream) {try {                              lc_tCipherInputStream.close();} catch (java.io.IOException lc_tException) {}} // close CipherOutputStream
  if (null != lc_tFileOutputStream ) {try {lc_tFileOutputStream.flush(); lc_tFileOutputStream.close() ;} catch (java.io.IOException lc_tException) {}} // close output file
  if (null != lc_tFileInputStream  ) {try {                              lc_tFileInputStream.close()  ;} catch (java.io.IOException lc_tException) {}} // close input stream

  //
  // check for cancellation
  //
  if (true == isCancelled())
  {
   deleteFile(pa_sDestinationFileName); 
   lc_bResult = false;
  }
  return lc_bResult;
 }
 
 
 } // end of inner class
 //---------------------------------------------------------------------------------------------------------------------
  
 
 
 
 
 /**
  * Constructor
  * 
  */
 public FileSafeApp()
 {
  super();
  
  m_tLocalizedMessages = ResourceBundle.getBundle(FileSafeConstants.FSA_FILE_LOCALIZED_MESSAGES);
  
  m_sEncryptFilePath        = null;
  m_sEncryptDestinationPath = null;
  m_sDecryptFilePath        = null;
  m_sDecryptDestinationPath = null;
  m_tWorkerThread           = null;
 }
 
 
 
 /**
  * At startup create and show the main frame of the application.
  * 
  */
 @Override protected void startup() 
 {
  if (null != (m_tView = new FileSafeView((FileSafeGuiConnector) this, (WindowListener) this)))
  {
   show(m_tView);
  }
 }

  
  
 /**
  * This method is to initialize the specified window by injecting resources.
  * Windows shown in our application come fully initialized from the GUI
  * builder, so this additional configuration is not needed.
  */
 @Override protected void configureWindow(java.awt.Window root) {}

 
 
 /**
  * A convenient static getter for the application instance.
  * @return the instance of FileSafeApp
  */
 public static FileSafeApp getApplication() {return Application.getInstance(FileSafeApp.class);}

 
 
 /**
  * Main method launching the application.
  * 
  */
 public static void main(String[] args) {launch(FileSafeApp.class, args);}



 /**
  * Quits the application
  * 
  */
 private void handleProgramExit()
 {
  m_tView.setVisible(false); m_tView.dispose();
  
  System.exit(1); 
 }

 

 /**
  * Returns the localized string for the given key
  * 
  * @param pa_sKey
  * @return  localized string
  */
 private String getLocalizedString(String pa_sKey) 
 {
  return (null != m_tLocalizedMessages ? m_tLocalizedMessages.getString(pa_sKey) : ""); 
 }
 
 
 
 /**
  * WindowListener.windowClosing()
  * 
  * @param pa_tEvent  Event
  */ 
  public void windowClosing(WindowEvent pa_tEvent) 
  {
   Object lc_tWindowObject = null;
  
   if (null != (lc_tWindowObject = pa_tEvent.getSource()))
   {
    if (lc_tWindowObject instanceof JFrame)
    {
     if ((JFrame) lc_tWindowObject == m_tView)   
     {
      handleProgramExit();
     }
    }
   }
  }

 
 
 /**
  * WindowListener.windowClosed()
  * 
  * @param pa_tEvent  Event
  */ 
  public void windowClosed(WindowEvent pa_tEvent) 
  {}

 
 
 /**
  * WindowListener.windowOpened()
  *
  * As soon as the main window is open, do all the intialization stuff
  *
  * @param pa_tEvent  Event
  */ 
 public void windowOpened(WindowEvent pa_tEvent) 
 {
  Object               lc_tWindowObject  = null ;


  if (null != (lc_tWindowObject = pa_tEvent.getSource()))
  {
   if (lc_tWindowObject instanceof JFrame)
   {
    if ((JFrame) lc_tWindowObject == m_tView)   
    {}
   }
  }
 }


 

 /**
  * WindowListener.windowIconified()
  * 
  * @param pa_tEvent  Event
  */ 
 public void windowIconified(WindowEvent pa_tEvent) 
 {}



 /**
  * WindowListener.windowDeiconified()
  * 
  * @param pa_tEvent  Event
  */ 
 public void windowDeiconified(WindowEvent pa_tEvent) 
 {}



 /**
  * WindowListener.windowActivated()
  * 
  * @param pa_tEvent  Event
  */ 
 public void windowActivated(WindowEvent pa_tEvent) 
 {}



 /**
  * WindowListener.windowDeactivated()
  * 
  * @param pa_tEvent  Event
  */ 
 public void windowDeactivated(WindowEvent pa_tEvent) 
 {}

 
 
 /**
  * GuiConnector.handleButtonEncryptClear
  * 
  * Handles clicking Clear in the encryption tab
  */
 public void handleButtonEncryptClear()
 {
  m_tView.clearEncryptList();
 }
 

 
 /**
  * GuiConnector.handleButtonEncryptRemove
  * 
  * Handles clicking Remove in the encryption tab
  */
 public void handleButtonEncryptRemove()
 {
  m_tView.removeSelectedEncryptListEntry();
 }
 
 
 
 /**
  * GuiConnector.handleButtonEncryptAdd
  * 
  * Handles clicking Add in the encryption tab
  */
 public void handleButtonEncryptAdd()
 {
  JFileChooser     lc_tFileChooser = new JFileChooser();
  File[]           lc_tFiles       = null;
  
  if (null != lc_tFileChooser)
  {
   if (null != m_sEncryptFilePath)
   {
    lc_tFileChooser.setCurrentDirectory(new File(m_sEncryptFilePath));       
   }                 
   lc_tFileChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
   lc_tFileChooser.setAcceptAllFileFilterUsed(true);
   lc_tFileChooser.setMultiSelectionEnabled(true); 
   lc_tFileChooser.setDialogTitle(getLocalizedString(FileSafeConstants.FSA_DIALOG_TITLE_FILEREQUESTER_ENCRYPT));
   lc_tFileChooser.setApproveButtonText(getLocalizedString(FileSafeConstants.FSA_DIALOG_OPTION_OK));
    
   if (JFileChooser.APPROVE_OPTION == lc_tFileChooser.showOpenDialog(m_tView))
   {
    if (null != (lc_tFiles = lc_tFileChooser.getSelectedFiles()))
    {
     try 
     {
      m_sEncryptFilePath = lc_tFileChooser.getCurrentDirectory().getCanonicalPath();
     } 
     catch (IOException lc_tException) 
     {}
     
     for (File lc_tFile : lc_tFiles) 
     {
      if (null != lc_tFile) 
      {
       if (true == lc_tFile.canRead())
       {
        try 
        {
         m_tView.addEncryptListEntry(lc_tFile.getCanonicalPath());
        } 
        catch (IOException lc_tException)
        {}
       }
      }
     }// for
     m_tView.checkButtons();
    }
   }
  }
 }
 
 
 
 
 /**
  * GuiConnector.handleButtonEncryptClear
  * 
  * Handles clicking Select in the encryption tab
  */
 public void handleButtonEncryptSelectDestination()
 {  
  JFileChooser     lc_tFileChooser = new JFileChooser();
  File             lc_tFile        = null;

  if (null != lc_tFileChooser)
  {
   if (null != m_sEncryptDestinationPath)
   {
    lc_tFileChooser.setCurrentDirectory(new File(m_sEncryptDestinationPath));       
   }         
   lc_tFileChooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
   lc_tFileChooser.setAcceptAllFileFilterUsed(false);
   lc_tFileChooser.setMultiSelectionEnabled(false); 
   lc_tFileChooser.setDialogTitle(getLocalizedString(FileSafeConstants.FSA_DIALOG_TITLE_FILEREQUESTER_ENCRYPT_DESTINATION));
   lc_tFileChooser.setApproveButtonText(getLocalizedString(FileSafeConstants.FSA_DIALOG_OPTION_OK));
    
   if (JFileChooser.APPROVE_OPTION == lc_tFileChooser.showOpenDialog(m_tView))
   {
    if (null != (lc_tFile = lc_tFileChooser.getSelectedFile()))
    {
     if (true == lc_tFile.canWrite())
     {
      try 
      {
       m_sEncryptDestinationPath = lc_tFileChooser.getCurrentDirectory().getCanonicalPath();
       m_tView.setEncryptDestination(lc_tFile.getCanonicalPath());
      } 
      catch (IOException ex) 
      {}
     }
     else
     {
      JOptionPane.showMessageDialog(m_tView, 
                                    getLocalizedString(FileSafeConstants.FSA_DIALOG_TEXT_NOT_WRITEABLE_ERROR), 
                                    getLocalizedString(FileSafeConstants.FSA_DIALOG_TITLE_ERROR             ), 
                                    JOptionPane.ERROR_MESSAGE);                 
       
     }
    }
   }
  }
 }

 
 
 /**
  * GuiConnector.handleButtonEncryptClear
  * 
  * Handles clicking Clear in the encryption tab
  */
 public void handleButtonDecryptClear()
 {
  m_tView.clearDecryptList();
 }
 

 
 /**
  * GuiConnector.handleButtonEncryptRemove
  * 
  * Handles clicking Remove in the encryption tab
  */
 public void handleButtonDecryptRemove()
 {
  m_tView.removeSelectedDecryptListEntry();
 }
 
 
 
 /**
  * GuiConnector.handleButtonDecryptAdd
  * 
  * Handles clicking Add in the decryption tab
  */
 public void handleButtonDecryptAdd()
 {
  JFileChooser     lc_tFileChooser = new JFileChooser();
  File[]           lc_tFiles       = null;
  
  if (null != lc_tFileChooser)
  {
   if (null != m_sDecryptFilePath)
   {
    lc_tFileChooser.setCurrentDirectory(new File(m_sDecryptFilePath));       
   }                     
   lc_tFileChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
   lc_tFileChooser.setAcceptAllFileFilterUsed(true);
   lc_tFileChooser.setMultiSelectionEnabled(true); 
   lc_tFileChooser.setDialogTitle(getLocalizedString(FileSafeConstants.FSA_DIALOG_TITLE_FILEREQUESTER_DECRYPT));
   lc_tFileChooser.setApproveButtonText(getLocalizedString(FileSafeConstants.FSA_DIALOG_OPTION_OK));
    
   if (JFileChooser.APPROVE_OPTION == lc_tFileChooser.showOpenDialog(m_tView))
   {
    if (null != (lc_tFiles = lc_tFileChooser.getSelectedFiles()))
    {
     try 
     {
      m_sDecryptFilePath = lc_tFileChooser.getCurrentDirectory().getCanonicalPath();
     } 
     catch (IOException lc_tException) 
     {}
      
     for (File lc_tFile : lc_tFiles) 
     {
      if (null != lc_tFile) 
      {
       if (true == lc_tFile.canRead())
       {
        try 
        {
         m_tView.addDecryptListEntry(lc_tFile.getCanonicalPath());
        } 
        catch (IOException lc_tException)
        {}
       }
      }
     }// for
     m_tView.checkButtons();
    }
   }
  }
 }
 
 
 
 
 /**
  * GuiConnector.handleButtonEncryptClear
  * 
  * Handles clicking Select in the encryption tab
  */
 public void handleButtonDecryptSelectDestination()
 {  
  JFileChooser     lc_tFileChooser = new JFileChooser();
  File             lc_tFile        = null;

  if (null != lc_tFileChooser)
  {
   if (null != m_sDecryptDestinationPath)
   {
    lc_tFileChooser.setCurrentDirectory(new File(m_sDecryptDestinationPath));       
   }             
   lc_tFileChooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
   lc_tFileChooser.setAcceptAllFileFilterUsed(false);
   lc_tFileChooser.setMultiSelectionEnabled(false); 
   lc_tFileChooser.setDialogTitle(getLocalizedString(FileSafeConstants.FSA_DIALOG_TITLE_FILEREQUESTER_DECRYPT_DESTINATION));
   lc_tFileChooser.setApproveButtonText(getLocalizedString(FileSafeConstants.FSA_DIALOG_OPTION_OK));
    
   if (JFileChooser.APPROVE_OPTION == lc_tFileChooser.showOpenDialog(m_tView))
   {
    if (null != (lc_tFile = lc_tFileChooser.getSelectedFile()))
    {
     if (true == lc_tFile.canWrite())
     {
      try 
      {
       m_sDecryptDestinationPath = lc_tFileChooser.getCurrentDirectory().getCanonicalPath();
       m_tView.setDecryptDestination(lc_tFile.getCanonicalPath());
      } 
      catch (IOException ex) 
      {}
     }
     else
     {
      JOptionPane.showMessageDialog(m_tView, 
                                    getLocalizedString(FileSafeConstants.FSA_DIALOG_TEXT_NOT_WRITEABLE_ERROR), 
                                    getLocalizedString(FileSafeConstants.FSA_DIALOG_TITLE_ERROR             ), 
                                    JOptionPane.ERROR_MESSAGE);                 
       
     }
    }
   }
  }
 }
 
 
 
 /**
  * Handles clicking the Cancel button
  * 
  */
 public void handleButtonCancel()
 {
  if (null != m_tWorkerThread)
  {
   m_tWorkerThread.cancel(true);
  }
 }
 
 
 
 /**
  * Handles clicking the start button
  * 
  */
 public void handleButtonStart()
 {
  FileSafePasswordDialog lc_tPasswordDialog = null;
  boolean                lc_bContinue       = true;

  if (null != (lc_tPasswordDialog = new FileSafePasswordDialog(m_tView, true)))
  {
   lc_tPasswordDialog.setVisible(true); 
   if (-1 == lc_tPasswordDialog.getButtonPressed())
   {
    lc_bContinue = false;
   }
   else
   {
    m_sPassword = lc_tPasswordDialog.getPassword();
   }       
   lc_tPasswordDialog.dispose();

   if (true == lc_bContinue)
   {
    if (null != (m_tWorkerThread = new FileSafeWorker()))
    {
     m_tView.lockGui(true);
     m_tView.clearLog();
     
     m_tWorkerThread.init(); 
     m_tWorkerThread.execute();      
    } 
   }
  }
 }

 
 
 /**
  * Handles clicking the Copy to clipboard button and copies
  * the log contents as text to the system clipboard
  * 
  */
 public void handleButtonCopyToClipboard()
 {
  StringSelection lc_tStringSelection = null;

  if (null != (lc_tStringSelection = new StringSelection(m_tView.getLog())))
  {
   Toolkit.getDefaultToolkit().getSystemClipboard().setContents(lc_tStringSelection, null);
  }
 }
 
 
 
 /**
  * Handles termination of worker thread
  * 
  */
 public final void handleWorkerThreadFinished(String pa_sMessage)
 {
  //
  // display results as message
  //
  JOptionPane.showMessageDialog(m_tView    , 
                                pa_sMessage, 
                                getLocalizedString(FileSafeConstants.FSA_DIALOG_TITLE_INFORMATION), 
                                JOptionPane.INFORMATION_MESSAGE);                      
  
  //
  // unlock and update UI
  //
  m_tView.addLogMessage(getLocalizedString(FileSafeConstants.FSA_MSG_FINISHED));
  m_tView.resetProgress();    
  m_tView.lockGui(false);
  m_tView.checkButtons();
 }
 
} // eoc
