/*
 * KeySafe
 * 
 * A simple password safe.
 * 
 */
package keysafe;



/**
 * Imports
 * 
 */
import java.io.IOException     ;
import java.io.File            ;
import java.util.ResourceBundle;

import org.jdesktop.application.Application;
import org.jdesktop.application.SingleFrameApplication;

import javax.swing.JFrame           ; 
import javax.swing.JOptionPane      ;
import java.awt.event.WindowListener;
import java.awt.event.WindowEvent   ;



/**
 * The main class of the application.
 * 
 * @author Ulf Wagemann
 */
public class KeySafeApp extends    SingleFrameApplication 
                        implements KeySafeGuiConnector   ,
                                   WindowListener
{
 /**
  * Class members
  * 
  */
 private KeySafeView          m_tView             ;
 private String               m_sMasterPassword   ; 
 private ResourceBundle       m_tLocalizedMessages;
 
 
 
 
 /**
  * At startup create and show the main frame of the application.
  * 
  */
 @Override protected void startup() 
 {
  m_tLocalizedMessages = ResourceBundle.getBundle(KeySafeConstants.KSA_FILE_LOCALIZED_MESSAGES);
  
  if (null != (m_tView = new KeySafeView((KeySafeGuiConnector) this, (WindowListener) this)))
  {
   show(m_tView);
  }
  m_sMasterPassword = "";
 }

 
 
 /**
  * This method is to initialize the specified window by injecting resources.
  * Windows shown in our application come fully initialized from the GUI
  * builder, so this additional configuration is not needed.
  */
 @Override protected void configureWindow(java.awt.Window root) {}
 
 

 /**
  * A convenient static getter for the application instance.
  * @return the instance of KeySafeApp
  */
 public static KeySafeApp getApplication() {return Application.getInstance(KeySafeApp.class);}
 


 /**
  * Main method launching the application.
  * 
  */
 public static void main(String[] args) {launch(KeySafeApp.class, args);}
 
 

 /**
  * WindowListener.windowClosing()
  * 
  * @param pa_tEvent  Event
  */ 
  public void windowClosing(WindowEvent pa_tEvent) 
  {
   Object lc_tWindowObject = null;
  
   if (null != (lc_tWindowObject = pa_tEvent.getSource()))
   {
    if (lc_tWindowObject instanceof JFrame)
    {
     if ((JFrame) lc_tWindowObject == m_tView)   
     {
      handleProgramExit();
     }
    }
   }
  }

 
 
 /**
  * WindowListener.windowClosed()
  * 
  * @param pa_tEvent  Event
  */ 
  public void windowClosed(WindowEvent pa_tEvent) 
  {}

 
 
 /**
  * WindowListener.windowOpened()
  *
  * As soon as the main window is open, do all the intialization stuff
  *
  * @param pa_tEvent  Event
  */ 
 public void windowOpened(WindowEvent pa_tEvent) 
 {
  boolean              lc_bExit          = false;
  Object               lc_tWindowObject  = null ;
  KeySafeLoginDialog   lc_tLoginDialog   = null ;
  KeySafeSetupDialog   lc_tSetupDialog   = null ;
  KeySafeEncryptedData lc_tEncryptedData = null ;


  if (null != (lc_tWindowObject = pa_tEvent.getSource()))
  {
   if (lc_tWindowObject instanceof JFrame)
   {
    if ((JFrame) lc_tWindowObject == m_tView)   
    {
     //
     // if there is no file yet, we need to setup master password
     //
     if (false == existsFile())
     {
      if (null != (lc_tSetupDialog = new KeySafeSetupDialog(m_tView, true)))
      {
       lc_tSetupDialog.setVisible(true); 
       if (-1 == lc_tSetupDialog.getButtonPressed())
       {
        lc_bExit = true;
       }
       else
       {
        m_sMasterPassword = lc_tSetupDialog.getPassword();
        createNewFile();
       }       
       lc_tSetupDialog.dispose();
      }       
     }
     //
     // if there is a file, we need the master password to be entered
     //
     else
     {
      if (null != (lc_tLoginDialog = new KeySafeLoginDialog(m_tView, true)))
      {
       lc_tLoginDialog.setVisible(true); 
       
       if (-1 == lc_tLoginDialog.getButtonPressed())
       {
        lc_bExit = true;
       }
       else
       {
        if (null != (lc_tEncryptedData = new KeySafeEncryptedData()))
        {
         m_sMasterPassword = lc_tLoginDialog.getPassword();
         if (true == lc_tEncryptedData.load(KeySafeConstants.KSA_FILE, m_sMasterPassword))
         {
          m_tView.initList(lc_tEncryptedData.get()); // update display 
         }
         else
         {
          lc_bExit = true;
         }
        }
       }
       lc_tLoginDialog.dispose();  
      }
     }
    }
   }
  }
  if (true == lc_bExit)
  {
   handleProgramExit();  
  }
 }


 

 /**
  * WindowListener.windowIconified()
  * 
  * @param pa_tEvent  Event
  */ 
 public void windowIconified(WindowEvent pa_tEvent) 
 {}



 /**
  * WindowListener.windowDeiconified()
  * 
  * @param pa_tEvent  Event
  */ 
 public void windowDeiconified(WindowEvent pa_tEvent) 
 {}



 /**
  * WindowListener.windowActivated()
  * 
  * @param pa_tEvent  Event
  */ 
 public void windowActivated(WindowEvent pa_tEvent) 
 {}



 /**
  * WindowListener.windowDeactivated()
  * 
  * @param pa_tEvent  Event
  */ 
 public void windowDeactivated(WindowEvent pa_tEvent) 
 {}


 
 
 /**
  * Checks whether the key file exists
  * 
  * @return   true, if it exists, false otherwise
  */
 private boolean existsFile()
 {
  File lc_tFile = new File(KeySafeConstants.KSA_FILE);  
  
  if (null != lc_tFile)
  {
   if (true == lc_tFile.exists() && true == lc_tFile.isFile() && true == lc_tFile.canRead())
   {
    return true; 
   }
  }
  return false;
 }

 
 
 /**
  * Creates an empty new key file
  * 
  */
 private void createNewFile()
 {
  File lc_tFile = new File(KeySafeConstants.KSA_FILE);  
 
  if (null != lc_tFile)
  {
   try 
   {
    lc_tFile.createNewFile();
   } 
   catch (IOException lc_tException) 
   {}
  }
 }
 
 
 
 /**
  * Quits the application
  * 
  */
 private void handleProgramExit()
 {
  handleButtonSave(false);   
  m_tView.setVisible(false); m_tView.dispose();
  
  System.exit(1); 
 }



 /**
  * Returns the localized string for the given key
  * 
  * @param pa_sKey
  * @return  localized string
  */
 private String getLocalizedString(String pa_sKey) 
 {
  return (null != m_tLocalizedMessages ? m_tLocalizedMessages.getString(pa_sKey) : ""); 
 }

 
 
 /**
  * Adds a new password record to the list
  * 
  */
 public void handleButtonNew()
 {
  KeySafeEntryDialog  lc_tEntryDialog  = null;

  if (null != (lc_tEntryDialog = new KeySafeEntryDialog(m_tView, true)))
  {
   lc_tEntryDialog.setVisible(true); 
   
   if (-1 != lc_tEntryDialog.getButtonPressed())
   {
    m_tView.addListEntry(new KeySafeEntry(lc_tEntryDialog.getLabel(), lc_tEntryDialog.getUser(), lc_tEntryDialog.getPassword()));
    m_tView.checkButtons();
   }       
   lc_tEntryDialog.dispose();
  }
 }        

 
 
 /**
  * Edits the current password record
  * 
  */
 public void handleButtonEdit()
 {
  KeySafeEntryDialog  lc_tEntryDialog  = null;
  KeySafeEntry        lc_tOldEntry     = null;
  KeySafeEntry        lc_tNewEntry     = null;
  
  if (null != (lc_tOldEntry = m_tView.getSelectedListEntry()))
  {
   if (null != (lc_tEntryDialog = new KeySafeEntryDialog(m_tView, true)))
   {
    lc_tEntryDialog.setData(lc_tOldEntry);
    lc_tEntryDialog.setVisible(true); 
   
    if (-1 != lc_tEntryDialog.getButtonPressed())
    {
     if (null != (lc_tNewEntry = new KeySafeEntry(lc_tEntryDialog.getLabel()   , 
                                                  lc_tEntryDialog.getUser()    , 
                                                  lc_tEntryDialog.getPassword())))
     {
      if (false == lc_tOldEntry.isEqual(lc_tNewEntry))
      {
       m_tView.updateListEntry(lc_tOldEntry, lc_tNewEntry); 
       m_tView.checkButtons();
      }
     }    
    }       
    lc_tEntryDialog.dispose();
   }
  }
 }        

 
 
 /**
  * Deletes the current password record
  * 
  */
 public void handleButtonDelete()
 {
  m_tView.deleteSelectedListEntry();
  m_tView.checkButtons();
 }        

 
 
 
 /**
  * Clears all password records
  * 
  */
 public void handleButtonClear()
 {
  m_tView.clearList();
  m_tView.checkButtons();
 }        

 
 
 /**
  * Loads the passwords and decodes them
  * 
  */
 public void handleButtonLoad()
 {
  boolean              lc_bResult        = false;
  KeySafeEncryptedData lc_tEncryptedData = null ;

  if (null != (lc_tEncryptedData = new KeySafeEncryptedData()))
  {
   if (true == (lc_bResult = lc_tEncryptedData.load(KeySafeConstants.KSA_FILE, m_sMasterPassword)))
   {
    m_tView.initList(lc_tEncryptedData.get());
   }
   JOptionPane.showMessageDialog(m_tView, 
                                 getLocalizedString(true == lc_bResult ? KeySafeConstants.KSA_DIALOG_TEXT_LOAD_OK      : KeySafeConstants.KSA_DIALOG_TEXT_LOAD_FAILED), 
                                 getLocalizedString(true == lc_bResult ? KeySafeConstants.KSA_DIALOG_TITLE_INFORMATION : KeySafeConstants.KSA_DIALOG_TITLE_ERROR     ), 
                                 true == lc_bResult ? JOptionPane.INFORMATION_MESSAGE : JOptionPane.ERROR_MESSAGE);
  }
 }        
 
 
 
 /**
  * Saves the passwords encoded
  * 
  */
 public void handleButtonSave(boolean pa_bMessage)
 {
  boolean              lc_bResult        = false;
  KeySafeEntry[]       lc_tListData      = null ;
  KeySafeEncryptedData lc_tEncryptedData = null ;

  if (null != (lc_tListData = m_tView.getListContent()))
  {
   if (null != (lc_tEncryptedData = new KeySafeEncryptedData(lc_tListData)))
   {
    lc_bResult = lc_tEncryptedData.save(KeySafeConstants.KSA_FILE, m_sMasterPassword);

    if (true == pa_bMessage)
    {
     JOptionPane.showMessageDialog(m_tView, 
                                   getLocalizedString(true == lc_bResult ? KeySafeConstants.KSA_DIALOG_TEXT_SAVE_OK      : KeySafeConstants.KSA_DIALOG_TEXT_SAVE_FAILED), 
                                   getLocalizedString(true == lc_bResult ? KeySafeConstants.KSA_DIALOG_TITLE_INFORMATION : KeySafeConstants.KSA_DIALOG_TITLE_ERROR     ), 
                                   true == lc_bResult ? JOptionPane.INFORMATION_MESSAGE : JOptionPane.ERROR_MESSAGE);
    }
   }
  }
 }


 
} // eoc
