/*
 * KeySafe
 * 
 * A simple password safe.
 * 
 */
package keysafe;


/**
 * Imports
 * 
 */ 
import java.util.HashMap        ;
import java.util.Set            ;
import java.util.Arrays         ;
import java.util.Comparator     ;
import java.security.SecureRandom         ;
import java.util.StringTokenizer;

import java.io.File            ;
import java.io.FileInputStream ;
import java.io.FileOutputStream;

import javax.crypto.BadPaddingException;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.spec.PBEKeySpec      ;
import javax.crypto.spec.PBEParameterSpec;
import javax.crypto.SecretKey            ;
import javax.crypto.SecretKeyFactory     ;
import javax.crypto.Cipher               ;



/**
 * Hashmap with encrypted load()/save() functionality
 * 
 * @author Ulf Wagemann
 */
public class KeySafeEncryptedData 
{
 private HashMap<String,KeySafeEntry> m_tPasswordData;
 
 
 /**
  * Constructor
  * 
  */
 public KeySafeEncryptedData()
 {
  m_tPasswordData = new HashMap<String,KeySafeEntry>(); 
 }
 
 
 /**
  * Constructor
  * 
  */
 public KeySafeEncryptedData(KeySafeEntry[] pa_tData)
 {
  m_tPasswordData = new HashMap<String,KeySafeEntry>(); 
  
  if (null != pa_tData)
  {
   for (KeySafeEntry lc_tEntry : pa_tData) 
   {
    addEntry(lc_tEntry.getLabel(), lc_tEntry.getUser(), lc_tEntry.getPassword()); 
   }
  }
 }
 
 
 /**
  * Returns the amount of stored passwords
  * 
  * @return  amount, may be 0
  */
 public final int getSize()
 {
  return (null != m_tPasswordData ? m_tPasswordData.size() : 0);  
 }
 
 
 /**
  * Clears all entries
  * 
  */
 public final void clear()
 {
  m_tPasswordData.clear(); 
 }
 
 
 
 /**
  * Removes the given entry
  * 
  * param pa_sLabel    label
  * @return             true, if the entry could be removed, false otherwise
  */
 public final boolean removeEntry(String pa_sLabel)
 {
  return (null != m_tPasswordData.remove(pa_sLabel)); 
 }
 
 
 
 /**
  * Adds a new entry to the hashmap
  * 
  * @param pa_sKey       entry key
  * @param pa_sPassword  entry password
  */
 public final void addEntry(String pa_sLabel, String pa_sUser, String pa_sPassword)
 {
  if (null != m_tPasswordData) 
  {
   m_tPasswordData.put(pa_sLabel, new KeySafeEntry(pa_sLabel, pa_sUser, pa_sPassword)); 
  }
 }
 
 
  
  /**
  * Returns the password of an entry for its pa_sLabel
  * 
  * @param pa_sLabel   label
  * @return          password, null in case of error
  */
 public final String getEntryUser(String pa_sLabel)
 {
  KeySafeEntry lc_tEntry = null;
  
  if (null != m_tPasswordData) 
  {
   if (null != (lc_tEntry = m_tPasswordData.get(pa_sLabel)))
   {
    return lc_tEntry.getUser(); 
   }
  }
  return null; 
 }
 
 
 
 /**
  * Returns the password of an entry for its pa_sLabel
  * 
  * @param pa_sLabel   label
  * @return          password, null in case of error
  */
 public final String getEntryPassword(String pa_sLabel)
 {
  KeySafeEntry lc_tEntry = null;
  
  if (null != m_tPasswordData) 
  {
   if (null != (lc_tEntry = m_tPasswordData.get(pa_sLabel)))
   {
    return lc_tEntry.getPassword(); 
   }
  }
  return null; 
 }
 
 
 
 /**
  * Returns a sorted array of labels
  * 
  * @return  null in case of error
  */
 public final KeySafeEntry[] get()
 {
  KeySafeEntry[] lc_tEntries = null;
  KeySafeEntry   lc_tEntry   = null;
  Set<String>    lc_tLabels  = null;
  
  int lc_iCount = 0;
  
  if (null != (lc_tLabels = m_tPasswordData.keySet()))
  {
   if (null != (lc_tEntries = new KeySafeEntry[getSize()])) 
   {
    for (String lc_sLabel : lc_tLabels)  
    { 
     if (null != (lc_tEntry = m_tPasswordData.get(lc_sLabel)))
     {
      lc_tEntries[lc_iCount++] = new KeySafeEntry(lc_tEntry); 
     }
    }
    Arrays.sort(lc_tEntries, new Comparator<KeySafeEntry>() 
                             {
                              public int compare(KeySafeEntry o1, KeySafeEntry o2)
                              { 
                               return o1.getLabel().compareTo(o2.getLabel());
                              }
                              });
    return (lc_tEntries.length > 0 ? lc_tEntries : null);
   }
  }
  return null;
 }
 
 

 /**
  * Creates a new empty file
  * 
  * @param pa_sFileName   file name
  */
 private void createNewFile(String pa_sFileName)
 {
  File lc_tFile = new File(KeySafeConstants.KSA_FILE);  
 
  if (null != lc_tFile)
  {
   try 
   {
    lc_tFile.createNewFile();
   } 
   catch (java.io.IOException lc_tException) 
   {}
  }
 }
 
 

 /**
  * Returns the size in bytes of the key file
  * 
  * @return   size in bytes
  */
 private long getFileSize()
 {
  File lc_tFile = new File(KeySafeConstants.KSA_FILE);  
  
  if (null != lc_tFile)
  {
   return lc_tFile.length(); 
  }
  return 0;
 }
 
 
 
 /**
  * Writes the byte array to stream
  * 
  * @param pa_tStream   output stream
  * @param pa_tBytes    byte array to be written
  */
 private void writeBytes(FileOutputStream pa_tStream, byte[] pa_tBytes)
 {
  try 
  {
   pa_tStream.write(pa_tBytes);
  } 
  catch (java.io.IOException lc_tException) 
  {}
 }
 
 
 
 /**
  * Initializes a cipher object 
  * 
  * @param  pa_tSalt        salt to use
  * @param  pa_sPassword    password to use
  * @param  pa_iCipherMode  Cipher.ENCRYPT_MODE  or Cipher.DECRYPT_MODE
  * @return null in case of error, otherwise a configured cipher object
  */
 private Cipher setupCryptoStuff(byte[] pa_tSalt, String pa_sPassword, int pa_iCipherMode)
 {
  PBEKeySpec       lc_tKeySpec     = null;
  PBEParameterSpec lc_tParamSpec   = null;
  SecretKeyFactory lc_tKeyFactory  = null;
  SecretKey        lc_tPasswordKey = null;
  Cipher           lc_tCipher      = null;

  boolean      lc_bCryptoOk       = false;
  
  //
  // prepare crypto stuff
  //
  try 
  { 
   if (null != (lc_tKeySpec = new PBEKeySpec(pa_sPassword.toCharArray())))
   {
    if (null != (lc_tKeyFactory  = SecretKeyFactory.getInstance(KeySafeConstants.KSA_ALGORITHM)))
    {
     try
     {
      if (null != (lc_tPasswordKey = lc_tKeyFactory.generateSecret(lc_tKeySpec)))
      {
       if (null != (lc_tParamSpec = new PBEParameterSpec(pa_tSalt, 100)))
       {
        try 
        {
         if (null != (lc_tCipher = Cipher.getInstance(KeySafeConstants.KSA_ALGORITHM)))
         {
          lc_tCipher.init(pa_iCipherMode, lc_tPasswordKey, lc_tParamSpec); 
          lc_bCryptoOk = true;   
         }
        }
        catch (java.security.InvalidKeyException                 lc_tException) {} 
        catch (java.security.InvalidAlgorithmParameterException  lc_tException) {}
        catch (javax.crypto.NoSuchPaddingException               lc_tException) {}
       }
      }
     }
     catch (java.security.spec.InvalidKeySpecException lc_tException) 
     {}
    }
   }
  } 
  catch (java.security.NoSuchAlgorithmException lc_tException) 
  {} 
  return (true == lc_bCryptoOk ? lc_tCipher : null);
 }
 
 
 
 /**
  * Saves the data to the given file. The given password is used for encryption
  * 
  * @param pa_sFileName   file name
  * @param pa_sPassword   encryption password
  */
 public final boolean save(String pa_sFileName, String pa_sPassword)
 {
  Set<String>  lc_tLabels         = null ;
  StringBuffer lc_tOutLine        = new StringBuffer();
  boolean      lc_bResult         = false;

  byte[] lc_tOutBytes = null       ;
  byte[] lc_tSalt     = new byte[8];
  
  FileOutputStream lc_tFileOutputStream = null;
  SecureRandom     lc_tRandom           = new SecureRandom();
  Cipher           lc_tCipher           = null;
  KeySafeEntry     lc_tEntry            = null;
  
  if (null != m_tPasswordData && 
      null != lc_tSalt        && 
      null != lc_tRandom      && 
      null != pa_sPassword    && 
      null != pa_sFileName    && 
      null != lc_tOutLine      )
  {
   //
   // prepare crypto stuff
   //
   lc_tRandom.nextBytes(lc_tSalt);
   if (null != (lc_tCipher = setupCryptoStuff(lc_tSalt, pa_sPassword, Cipher.ENCRYPT_MODE)))
   {
    //
    // obtain keys
    // 
    if (null != (lc_tLabels = m_tPasswordData.keySet()))
    {
     //
     // create new empty file
     //
     createNewFile(pa_sFileName);
     
     if (0 < lc_tLabels.size())
     {
      //
      // open outputstream
      //
      try
      {
       if (null != (lc_tFileOutputStream = new FileOutputStream(pa_sFileName)))
       {
        //
        // write salt
        //
        writeBytes(lc_tFileOutputStream, lc_tSalt);    

        //
        // create output string
        // 
        for (String lc_sLabel : lc_tLabels)  
        {
         if (null != (lc_tEntry = m_tPasswordData.get(lc_sLabel)))
         {
          lc_tOutLine.append(lc_tEntry.getLabel());
          lc_tOutLine.append(KeySafeConstants.KSA_VALUE_DELIMITER);
          lc_tOutLine.append(lc_tEntry.getUser());
          lc_tOutLine.append(KeySafeConstants.KSA_VALUE_DELIMITER);
          lc_tOutLine.append(lc_tEntry.getPassword());
          lc_tOutLine.append(KeySafeConstants.KSA_ENTRY_DELIMITER);
         }
        }
                   
        //
        // encode an write output string
        //
        if (null != (lc_tOutBytes = lc_tOutLine.toString().getBytes()))
        {
         try 
         {
          byte[] lc_tEncBytes = lc_tCipher.doFinal(lc_tOutBytes);
          
          if (null != lc_tEncBytes)
          {
           writeBytes(lc_tFileOutputStream, lc_tEncBytes);
          }
         }
         catch (javax.crypto.IllegalBlockSizeException lc_tException) {}
         catch (javax.crypto.BadPaddingException       lc_tException) {}          
        }
         
        //
        // close output file
        //
        try
        {
         lc_tFileOutputStream.flush();
         lc_tFileOutputStream.close();
        }
        catch (java.io.IOException lc_tException) {}
        lc_bResult = true;
       }
      }
      catch(java.io.IOException lc_tException) {}
     }
     else
     {
      lc_bResult = true; 
     }
    }
   }
  }
  return lc_bResult;
 }
 
 
 
 /**
  * Loads and decrpyts a file
  * 
  * @param pa_sFileName   file name
  * @param pa_sPassword   password
  * @return               true, if the file could be opened, false otherwise
  */
 
 
 public final boolean load(String pa_sFileName, String pa_sPassword)
 {
  byte[]  lc_tSalt     = new byte[8];
  boolean lc_bResult   = false      ;
  long    lc_lSize     = getFileSize();
  int     lc_iBytes    = 0;
  
  FileInputStream  lc_tFileInputStream = null;
  Cipher           lc_tCipher          = null;
  String           lc_sInString        = null;
  
  byte[] lc_tEncBytes= null;
  
  if (null != lc_tSalt && null != pa_sPassword && null != pa_sFileName && null != m_tPasswordData)
  {
   //
   // open inputstream
   //
   try
   {
    if (null != (lc_tFileInputStream = new FileInputStream(pa_sFileName)))
    {
     if (8 == lc_tFileInputStream.read(lc_tSalt))
     {      
      //
      // prepare crypto stuff
      //
      if (null != (lc_tCipher = setupCryptoStuff(lc_tSalt, pa_sPassword, Cipher.DECRYPT_MODE)))
      {    
       lc_iBytes = ((int) lc_lSize) -8; // subtract salt length
       
       //
       // read file completely
       //
       if (null != (lc_tEncBytes = new byte[lc_iBytes]))
       {
        lc_tFileInputStream.read(lc_tEncBytes, 0, lc_iBytes); 
        
        try 
        {
         byte[] lc_tInBytes = lc_tCipher.doFinal(lc_tEncBytes);
         if (null != lc_tInBytes)
         {
          if (null != (lc_sInString = new String(lc_tInBytes))) 
          {
           fillHashMap(lc_sInString); 
           lc_bResult = true;
          }
         }
        } 
        catch (IllegalBlockSizeException ex) {}
        catch (BadPaddingException       ex) {}
       }
      }
     }

     //
     // close input file
     //
     try
     {
      lc_tFileInputStream.close();
     }
     catch (java.io.IOException lc_tException) {}
    }
   }
   catch(java.io.IOException lc_tException) {}
  }
  return lc_bResult;
 }
 
 
 
 /**
  * Fills the internal map from one huge input string
  * 
  * @param pa_sInputString    input string
  */
 private void fillHashMap(String pa_sInputString)
 {
  StringTokenizer lc_tLineTokenizer  = null;
  StringTokenizer lc_tEntryTokenizer = null;
  
  String lc_sLine     = null;
  String lc_sLabel    = null;
  String lc_sUser     = null;
  String lc_sPassword = null;
  
  if (null != pa_sInputString)
  {
   if (0 < pa_sInputString.length())
   {
    clear();
    
    if (null != (lc_tLineTokenizer = new StringTokenizer(pa_sInputString, KeySafeConstants.KSA_ENTRY_DELIMITER))) 
    {
     //
     // loop over lines
     //
     while (lc_tLineTokenizer.hasMoreTokens())
     {
      if (null != (lc_sLine = lc_tLineTokenizer.nextToken()))
      {
       //
       // for each line, set up an internal tokenizer for extracting values
       //
       if (0 < lc_sLine.length()) 
       {
        if (null != (lc_tEntryTokenizer = new StringTokenizer(lc_sLine, KeySafeConstants.KSA_VALUE_DELIMITER))) 
        {
         lc_sLabel    = lc_tEntryTokenizer.nextToken(); 
         lc_sUser     = lc_tEntryTokenizer.nextToken(); 
         lc_sPassword = lc_tEntryTokenizer.nextToken(); 
         addEntry(lc_sLabel, lc_sUser, lc_sPassword);
        }
       }
      }
     }
    }
   }
  }
 }
 
 
} // eoc
