/*
 * KeySafeApp
 * 
 * A simple password safe.
 * 
 */
package keysafe;



/**
 * Main window
 * 
 * @author Ulf Wagemann
 */
public class KeySafeView extends javax.swing.JFrame 
                         implements javax.swing.event.ListSelectionListener,
                                    java.awt.event.MouseListener 
{
 private KeySafeGuiConnector m_tGuiConnector;

 
 
 /**
  * Constructor
  * 
  * @param pa_tGuiConnector   GUI connector object
  * @param pa_tListener       window listener
  */
  public KeySafeView(KeySafeGuiConnector pa_tGuiConnector, java.awt.event.WindowListener pa_tListener) 
  {
   initComponents();
   
   jListEntries.setModel( new KeySafeListModel());
   jListEntries.setCellRenderer(new KeySafeListCellRenderer());
   jListEntries.addListSelectionListener((javax.swing.event.ListSelectionListener) this);
   jListEntries.addMouseListener((java.awt.event.MouseListener) this);
   
   m_tGuiConnector = pa_tGuiConnector;
   addWindowListener(pa_tListener);
  }


  
 /**
  * Handles a list selection
  * 
  * @param pa_tEvent 
  */
 public void valueChanged(javax.swing.event.ListSelectionEvent pa_tEvent)
 {
  if (null != pa_tEvent && false == pa_tEvent.getValueIsAdjusting())
  {
   setSelectedPassword();
   checkButtons();
  }
 }
 
  
  /**
   * Populates the list according to the given data
   * 
   * @param pa_tEntries    arraylist of records with data
   * 
   */
  public void initList(KeySafeEntry[] pa_tEntries)
  {
   KeySafeListModel lc_tModel  = (KeySafeListModel) jListEntries.getModel();

   if (null != pa_tEntries)
   {
    if (null != lc_tModel)
    {
     clearList(); 
     lc_tModel.addAll(pa_tEntries); 
    }
    setDisplayData("", "");
    checkButtons();
   }
  }
  
  
  
  /**
   * Adds a new record to the list
   * 
   * @param pa_tRecord   record to add
   */
  public void addListEntry(KeySafeEntry pa_tRecord)
  {
   KeySafeListModel lc_tModel = (KeySafeListModel) jListEntries.getModel();

   if (null != lc_tModel)
   {
    //
    // since only the label is considered by the comparable interface of KeySafeEntry,
    // we can safely delete any eventually existing object with the same label
    //
    lc_tModel.removeElement(pa_tRecord);
    lc_tModel.add(pa_tRecord);
    jListEntries.setSelectedValue(pa_tRecord, true);
    setSelectedPassword();
   }
  }
  
  
  
  /**
   * Updates the old entry with the new one
   * 
   * @param pa_tOldRecord  record to update
   * @param pa_tNewRecord  record to be updated with
   */
  public void updateListEntry(KeySafeEntry pa_tOldRecord, KeySafeEntry pa_tNewRecord)
  {
   KeySafeListModel lc_tModel = (KeySafeListModel) jListEntries.getModel();

   if (null != lc_tModel)
   {
    lc_tModel.removeElement(pa_tOldRecord);
    lc_tModel.add(pa_tNewRecord);
    jListEntries.setSelectedValue(pa_tNewRecord, true);
    setSelectedPassword();
   }    
  }
  
  
  
  /**
   * Returns the selected list entry
   * 
   * @return   selected entry, may be null
   */
  public KeySafeEntry getSelectedListEntry()
  {
   int lc_iPos = jListEntries.getSelectedIndex();   
   return (-1 != lc_iPos ? (KeySafeEntry) jListEntries.getSelectedValue() : null);
  }
  
  
  
  /**
   * Deletes the selected list entry
   * 
   */
  public void deleteSelectedListEntry()
  {
   int lc_iPos     = jListEntries.getSelectedIndex(); 
   int lc_iNewSize = 0;
   
   KeySafeListModel lc_tModel = (KeySafeListModel) jListEntries.getModel();

   if (-1 != lc_iPos)
   {
    if (null != lc_tModel)
    {
     lc_tModel.removeElement(getSelectedListEntry());
     jListEntries.clearSelection();
     
     if (0 < (lc_iNewSize = lc_tModel.getSize()))
     {
      if (lc_iPos >= lc_iNewSize)
      {
       lc_iPos = lc_iNewSize-1;
      }
      jListEntries.setSelectedIndex(lc_iPos);
     }
     setSelectedPassword();
    }
   }
  }
  
  
  
  /**
   * Clears the list
   * 
   */
  public void clearList()
  {
   KeySafeListModel lc_tModel = (KeySafeListModel) jListEntries.getModel();

   if (null != lc_tModel)
   {
    lc_tModel.clear();
    jListEntries.clearSelection();
    setSelectedPassword();
   }
  }
  
  
  
  /**
   * Returns the name of the selected palette
   * 
   * @return   Palette name
   */
  public void setSelectedPassword()
  {
   KeySafeEntry lc_tRecord = (KeySafeEntry) jListEntries.getSelectedValue();
   setDisplayData(null != lc_tRecord ? lc_tRecord.getUser()     : "", 
                  null != lc_tRecord ? lc_tRecord.getPassword() : "");
  }

  
  
  /**
   * Returns the list content as an array of KeySafeEntry
   * 
   * @return   array, may be null
   */
  public KeySafeEntry[] getListContent()
  {
   KeySafeEntry[] lc_tResult  = null;  
   KeySafeEntry   lc_tEntry   = null;
   Object[]       lc_tObjects = null;
   int            lc_iSize    = 0   ;
   int            lc_iCount   = 0   ;
   
   KeySafeListModel lc_tModel = (KeySafeListModel) jListEntries.getModel();

   if (null != lc_tModel)
   {
    if (0 < (lc_iSize = lc_tModel.getSize()))
    {
     if (null != (lc_tObjects = lc_tModel.elements()))
     {
      if (null != (lc_tResult = new KeySafeEntry[lc_iSize]))
      {
       for (Object lc_tObject : lc_tObjects) 
       {
        if (null != lc_tObject) 
        {
         if (lc_tObject instanceof KeySafeEntry) 
         {
          if (null != (lc_tEntry = (KeySafeEntry) lc_tObject)) 
          {
           lc_tResult[lc_iCount++] = lc_tEntry;
          }
         }
        }
       } // for
       
      }
     }
    }
   }   
   return lc_tResult;
  }
  
  
  /**
   * Sets the password display to the given value
   * 
   * @param pa_sString    password
   */
  public void setDisplayData(String pa_sUser, String pa_sPassword) 
  {
   jTextFieldUser.setText(pa_sUser);
   jTextFieldPassword.setText(pa_sPassword);
  }
  
  
  
  /** This method is called from within the constructor to
   * initialize the form.
   * WARNING: Do NOT modify this code. The content of this method is
   * always regenerated by the Form Editor.
   */
  @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        jScrollPaneEntries = new javax.swing.JScrollPane();
        jListEntries = new javax.swing.JList();
        jPanelDataDisplay = new javax.swing.JPanel();
        jLabelUser = new javax.swing.JLabel();
        jLabelPassword = new javax.swing.JLabel();
        jTextFieldUser = new javax.swing.JTextField();
        jTextFieldPassword = new javax.swing.JTextField();
        jButtonLoad = new javax.swing.JButton();
        jButtonSave = new javax.swing.JButton();
        jButtonNew = new javax.swing.JButton();
        jButtonDelete = new javax.swing.JButton();
        jButtonClear = new javax.swing.JButton();
        jButtonEdit = new javax.swing.JButton();

        setDefaultCloseOperation(javax.swing.WindowConstants.DO_NOTHING_ON_CLOSE);
        java.util.ResourceBundle bundle = java.util.ResourceBundle.getBundle("keysafe/resources/KeySafeView"); // NOI18N
        setTitle(bundle.getString("Form.title")); // NOI18N
        setName("Form"); // NOI18N

        jScrollPaneEntries.setAlignmentX(1.0F);
        jScrollPaneEntries.setAlignmentY(1.0F);
        jScrollPaneEntries.setName("jScrollPaneEntries"); // NOI18N

        jListEntries.setBackground(new java.awt.Color(Integer.valueOf(bundle.getString("jList.background"), 16)));
        jListEntries.setForeground(new java.awt.Color(Integer.valueOf(bundle.getString("jList.foreground"), 16)));
        jListEntries.setSelectionMode(javax.swing.ListSelectionModel.SINGLE_SELECTION);
        jListEntries.setName("jListEntries"); // NOI18N
        jListEntries.setSelectionBackground(new java.awt.Color(Integer.valueOf(bundle.getString("jList.selectionBackground"), 16)));
        jListEntries.setSelectionForeground(new java.awt.Color(Integer.valueOf(bundle.getString("jList.selectionForeground"), 16)));
        jScrollPaneEntries.setViewportView(jListEntries);

        jPanelDataDisplay.setBorder(javax.swing.BorderFactory.createEtchedBorder());
        jPanelDataDisplay.setName("jPanelDataDisplay"); // NOI18N

        jLabelUser.setLabelFor(jTextFieldUser);
        jLabelUser.setText(bundle.getString("jLabelUser.text")); // NOI18N
        jLabelUser.setName("jLabelUser"); // NOI18N

        jLabelPassword.setLabelFor(jTextFieldPassword);
        jLabelPassword.setText(bundle.getString("jLabelPassword.text")); // NOI18N
        jLabelPassword.setName("jLabelPassword"); // NOI18N

        jTextFieldUser.setEditable(false);
        jTextFieldUser.setText(bundle.getString("jTextFieldUser.text")); // NOI18N
        jTextFieldUser.setAlignmentY(0.0F);
        jTextFieldUser.setName("jTextFieldUser"); // NOI18N

        jTextFieldPassword.setEditable(false);
        jTextFieldPassword.setText(bundle.getString("jTextFieldPassword.text")); // NOI18N
        jTextFieldPassword.setAlignmentY(0.0F);
        jTextFieldPassword.setName("jTextFieldPassword"); // NOI18N

        javax.swing.GroupLayout jPanelDataDisplayLayout = new javax.swing.GroupLayout(jPanelDataDisplay);
        jPanelDataDisplay.setLayout(jPanelDataDisplayLayout);
        jPanelDataDisplayLayout.setHorizontalGroup(
            jPanelDataDisplayLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(jPanelDataDisplayLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(jPanelDataDisplayLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(jPanelDataDisplayLayout.createSequentialGroup()
                        .addComponent(jLabelUser)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(jTextFieldUser, javax.swing.GroupLayout.DEFAULT_SIZE, 129, Short.MAX_VALUE))
                    .addGroup(jPanelDataDisplayLayout.createSequentialGroup()
                        .addComponent(jLabelPassword)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(jTextFieldPassword, javax.swing.GroupLayout.DEFAULT_SIZE, 129, Short.MAX_VALUE)))
                .addContainerGap())
        );

        jPanelDataDisplayLayout.linkSize(javax.swing.SwingConstants.HORIZONTAL, new java.awt.Component[] {jLabelPassword, jLabelUser});

        jPanelDataDisplayLayout.setVerticalGroup(
            jPanelDataDisplayLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(jPanelDataDisplayLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(jPanelDataDisplayLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(jLabelUser)
                    .addComponent(jTextFieldUser, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(jPanelDataDisplayLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(jLabelPassword)
                    .addComponent(jTextFieldPassword, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );

        jPanelDataDisplayLayout.linkSize(javax.swing.SwingConstants.VERTICAL, new java.awt.Component[] {jLabelPassword, jLabelUser, jTextFieldPassword, jTextFieldUser});

        jButtonLoad.setText(bundle.getString("jButtonLoad.text")); // NOI18N
        jButtonLoad.setAlignmentY(0.0F);
        jButtonLoad.setName("jButtonLoad"); // NOI18N
        jButtonLoad.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                actionPerformedButtonLoad(evt);
            }
        });

        jButtonSave.setText(bundle.getString("jButtonSave.text")); // NOI18N
        jButtonSave.setAlignmentY(0.0F);
        jButtonSave.setName("jButtonSave"); // NOI18N
        jButtonSave.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                actionPerformedButtonSave(evt);
            }
        });

        jButtonNew.setText(bundle.getString("jButtonNew.text")); // NOI18N
        jButtonNew.setAlignmentY(0.0F);
        jButtonNew.setName("jButtonNew"); // NOI18N
        jButtonNew.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                actionPerformedButtonNew(evt);
            }
        });

        jButtonDelete.setText(bundle.getString("jButtonDelete.text")); // NOI18N
        jButtonDelete.setAlignmentY(0.0F);
        jButtonDelete.setEnabled(false);
        jButtonDelete.setName("jButtonDelete"); // NOI18N
        jButtonDelete.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                actionPerformedButtonDelete(evt);
            }
        });

        jButtonClear.setText(bundle.getString("jButtonClear.text")); // NOI18N
        jButtonClear.setAlignmentY(0.0F);
        jButtonClear.setEnabled(false);
        jButtonClear.setName("jButtonClear"); // NOI18N
        jButtonClear.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                actionPerformedButtonClear(evt);
            }
        });

        jButtonEdit.setText(bundle.getString("jButtonEdit.text")); // NOI18N
        jButtonEdit.setAlignmentY(0.0F);
        jButtonEdit.setEnabled(false);
        jButtonEdit.setName("jButtonEdit"); // NOI18N
        jButtonEdit.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                actionPerformedButtonEdit(evt);
            }
        });

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(getContentPane());
        getContentPane().setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addComponent(jScrollPaneEntries, javax.swing.GroupLayout.DEFAULT_SIZE, 203, Short.MAX_VALUE)
                    .addComponent(jPanelDataDisplay, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addComponent(jButtonNew, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .addComponent(jButtonEdit, javax.swing.GroupLayout.DEFAULT_SIZE, 65, Short.MAX_VALUE)
                    .addComponent(jButtonSave, javax.swing.GroupLayout.DEFAULT_SIZE, 65, Short.MAX_VALUE)
                    .addComponent(jButtonLoad, javax.swing.GroupLayout.DEFAULT_SIZE, 65, Short.MAX_VALUE)
                    .addComponent(jButtonDelete, javax.swing.GroupLayout.DEFAULT_SIZE, 65, Short.MAX_VALUE)
                    .addComponent(jButtonClear, javax.swing.GroupLayout.DEFAULT_SIZE, 65, Short.MAX_VALUE))
                .addContainerGap())
        );

        layout.linkSize(javax.swing.SwingConstants.HORIZONTAL, new java.awt.Component[] {jButtonClear, jButtonDelete, jButtonEdit, jButtonLoad, jButtonNew, jButtonSave});

        layout.setVerticalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(layout.createSequentialGroup()
                        .addComponent(jButtonLoad)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(jButtonSave, javax.swing.GroupLayout.PREFERRED_SIZE, 20, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addGap(18, 18, 18)
                        .addComponent(jButtonEdit)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(jButtonNew)
                        .addGap(18, 18, 18)
                        .addComponent(jButtonDelete)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(jButtonClear))
                    .addComponent(jScrollPaneEntries, javax.swing.GroupLayout.DEFAULT_SIZE, 258, Short.MAX_VALUE))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(jPanelDataDisplay, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addContainerGap())
        );

        layout.linkSize(javax.swing.SwingConstants.VERTICAL, new java.awt.Component[] {jButtonClear, jButtonDelete, jButtonEdit, jButtonLoad, jButtonNew, jButtonSave});

        pack();
    }// </editor-fold>//GEN-END:initComponents

  private void actionPerformedButtonSave(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_actionPerformedButtonSave
  if (null != m_tGuiConnector) {m_tGuiConnector.handleButtonSave(true);}    
  }//GEN-LAST:event_actionPerformedButtonSave

  private void actionPerformedButtonLoad(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_actionPerformedButtonLoad
  if (null != m_tGuiConnector) {m_tGuiConnector.handleButtonLoad();}    
  }//GEN-LAST:event_actionPerformedButtonLoad

  private void actionPerformedButtonEdit(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_actionPerformedButtonEdit
  if (null != m_tGuiConnector) {m_tGuiConnector.handleButtonEdit();}    
  }//GEN-LAST:event_actionPerformedButtonEdit

  private void actionPerformedButtonNew(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_actionPerformedButtonNew
  if (null != m_tGuiConnector) {m_tGuiConnector.handleButtonNew();}    
  }//GEN-LAST:event_actionPerformedButtonNew

  private void actionPerformedButtonDelete(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_actionPerformedButtonDelete
  if (null != m_tGuiConnector) {m_tGuiConnector.handleButtonDelete();}    
  }//GEN-LAST:event_actionPerformedButtonDelete

  private void actionPerformedButtonClear(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_actionPerformedButtonClear
  if (null != m_tGuiConnector) {m_tGuiConnector.handleButtonClear();}    
  }//GEN-LAST:event_actionPerformedButtonClear

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton jButtonClear;
    private javax.swing.JButton jButtonDelete;
    private javax.swing.JButton jButtonEdit;
    private javax.swing.JButton jButtonLoad;
    private javax.swing.JButton jButtonNew;
    private javax.swing.JButton jButtonSave;
    private javax.swing.JLabel jLabelPassword;
    private javax.swing.JLabel jLabelUser;
    private javax.swing.JList jListEntries;
    private javax.swing.JPanel jPanelDataDisplay;
    private javax.swing.JScrollPane jScrollPaneEntries;
    private javax.swing.JTextField jTextFieldPassword;
    private javax.swing.JTextField jTextFieldUser;
    // End of variables declaration//GEN-END:variables


 
 /**
  * Checks button states
  *
  */
 public void checkButtons()
 {
  int lc_iPos    = jListEntries.getSelectedIndex();   
  int lc_iAmount = jListEntries.getModel().getSize();
  
  jButtonClear.setEnabled (lc_iAmount > 0);
  jButtonDelete.setEnabled(lc_iAmount > 0 && -1 != lc_iPos );
  jButtonEdit.setEnabled  (lc_iAmount > 0 && -1 != lc_iPos );
 }

 
 
 /**
  * MouseListener.mouseClicked
  * 
  * All we're interested in here is a mouse click in a table's column header This indicates a sort,
  * which might have moved the current selection out of the visible area. So we need to restore
  * an eventual previous selection and scroll there.
  * 
  * @param pa_tEvent    mouse event
  */
 public void mouseClicked(java.awt.event.MouseEvent pa_tEvent)
 {
  int lc_iIndex = -1;
  
  if (null != pa_tEvent)
  {
   if (pa_tEvent.getSource() == jListEntries)
   {
    if (-1 != (lc_iIndex = jListEntries.locationToIndex(pa_tEvent.getPoint())))
    {
     if (lc_iIndex < jListEntries.getModel().getSize())
     {
      if (2 == pa_tEvent.getClickCount())
      {
       if (null != m_tGuiConnector) {m_tGuiConnector.handleButtonEdit();}    
      }
     }
    }
   }
  }
 }
 
 

 /**
  * MouseListener.mouseEntered
     * 
  * @param pa_tEvent    mouse event
  */ 
 public void mouseEntered(java.awt.event.MouseEvent pa_tEvent) {}
 
 
 
 /**
  * MouseListener.mouseExited
     * 
  * @param pa_tEvent    mouse event
  */
 public void mouseExited(java.awt.event.MouseEvent pa_tEvent) {}

 
 
 /**
  * MouseListener.mousePressed. We catch this event from table headers (!) to store the currently active savegame selection
  * for later restoring. This event indicates that a sort process is about to begin, so before that sort, we retrieve the 
  * list of selections and store it. After the sort is finished, we restore this selection.
  * 
  * @param pa_tEvent    mouse event
  */
 public void mousePressed(java.awt.event.MouseEvent pa_tEvent) {}

 
 
 /**
  * MouseListener.mouseReleased. Handles list selections and table selections done by the user via mouse.
  * 
  * @param pa_tEvent    mouse event
  */
 public void mouseReleased(java.awt.event.MouseEvent pa_tEvent) {}
 
}
